# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for Hyper-V vm"""

import os
import re
import time
from AutomationUtils import machine
from AutomationUtils import logger
from VirtualServer.VSAUtils.VMHelper import HypervisorVM


class HyperVVM(HypervisorVM):
    """
    This is the main file for all  Hyper-V VM operations
    """

    def __init__(self, hv_obj, vm_name):
        """
        Initialization of hyper-v vm properties

        _get_vm_host()            - get the host of the VM among the servers list

        _get_vm_info()            - get the particular  information of VM

        _get_disk_list()        - gets the disk list opff the VM

        _merge_vhd()            - Merge the VHD with its snapshots

        mount_vhd()                - Mount the Vhd/VHDX and return the drive letter

        un_mount_vhd()            - Unmount the VHD mounted provided the path

        get_disk_in_controller()- get the disk in controller attached

        get_disk_path_from_pattern() - get the list of disk from pattern

        power_off()            - power off the VM

        power_on()            -power on the VM

        delete_vm()            - delete the VM

        update_vm_info()    - updates the VM info


        get_vhdx()         - get type all virtual hard disk




        """

        super(HyperVVM, self).__init__(hv_obj, vm_name)
        self.server_client_name = hv_obj.server_list
        self.serverlist = hv_obj.server_list
        self.vmserver_host_name = self.server_name

        self.vm_props_file = "GetHypervProps.ps1"
        self.vm_operation_file = "HyperVOperation.ps1"

        self.prop_dict = {
            "server_name": self.vmserver_host_name,
            "extra_args": "$null",
            "vm_name": self.vm_name
        }
        self.vmserver_host_name = self.vm_host

        self.host_machine = machine.Machine(
            self.server_client_name, self.commcell)

        self.operation_dict = {
            "server_name": self.vmserver_host_name,
            "extra_args": "$null",
            "vm_name": self.vm_name,
            "vhd_name": "$null"
        }
        self.guid = None
        self.ip = None
        self.guest_os = None
        self.host_name = None
        self._disk_list = None
        self.disk_dict = None
        self.disk_path = None
        self._basic_props_initialized = False
        if self.vm_exist:
            self.update_vm_info()

    class VmConversionValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):
            return other.NicName == self.vm_restore_options.network

    class LiveSyncVmValidation(object):
        def __init__(self, vmobj, schedule, replicationjob=None):
            self.vm = vmobj
            self.schedule = schedule
            self.replicationjob = replicationjob
            self.log = logger.get_log()

        def __eq__(self, other):
            """ validates vm replicated through livesync """
            try:
                if '__GX_BACKUP__' not in other.vm.vm.VMSnapshot:
                    self.log.info('snapshot validation failed')
                    return False
                self.log.info('snapshot validation successful')

                config_val = (int(self.vm.vm.no_of_cpu) == int(other.vm.vm.no_of_cpu) and
                              int(self.vm.vm.disk_count) == int(other.vm.vm.disk_count) and
                              int(self.vm.vm.memory) == int(other.vm.vm.memory))
                if not config_val:
                    return False

                # disk type validation
                sourcedisks = self.vm.vm.get_vhdx()
                destdisks = other.vm.vm.get_vhdx()
                for cont in sourcedisks:
                    if cont in destdisks:
                        if sourcedisks[cont][1] != destdisks[cont][1]:
                            self.log.info('Disk type validation failed')
                            return False
                    else:
                        self.log.info("Disk validation failed")
                        return False
                self.log.info('Disk validation successful')

                # network validation
                scheduleprops = self.schedule.virtualServerRstOptions
                schdetails = scheduleprops['diskLevelVMRestoreOption']['advancedRestoreOptions']
                for vmdetails in schdetails:
                    if vmdetails['name'] == self.vm.vm.vm_name:
                        if 'nics' in vmdetails:
                            if vmdetails['nics'][0]['networkName'] != other.vm.vm.NicName:
                                return False

                return True

            except Exception as err:
                self.log.exception(
                    "Exception at Validating  {0}".format(err))
                raise err

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.restore_job = self.vm_restore_options.restore_job
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the source vm and restored vm"""
            sourcedisks = self.vm.vm.get_vhdx()
            destdisks = other.vm.vm.get_vhdx()
            for cont in sourcedisks:
                if cont in destdisks:
                    if sourcedisks[cont][1] != destdisks[cont][1]:
                        self.log.info('Disk type validation failed')
                        return False
            self.log.info('Disk type validation passed')
            return True

    class BackupValidation(object):
        def __init__(self, vm_obj, backup_option):
            self.vm = vm_obj
            self.backup_option = backup_option
            self.backup_job = self.backup_option.backup_job
            self.log = logger.get_log()

        def validate(self):
            """perform postbackup validation"""

            pass

    @property
    def disk_list(self):
        """to fetch the disk in the VM
        Return:
            disk_list   (list)- list of disk in VM
            e.g:[ide0-0-test1.vhdx]
        """
        self.disk_dict = self._get_disk_list
        if self.disk_dict:
            self._disk_list = self.disk_dict.keys()

        else:
            self._disk_list = []

        return self._disk_list

    @property
    def vm_host(self):
        """
        get the Host of the VM

        Return:
            vm_host     (str)- VM Host of the VM
        """
        if not isinstance(self.serverlist, list):
            server_list = [self.serverlist]
        else:
            server_list = self.serverlist

        _vm_host = self._get_vm_host(server_list)
        self.server_client_name = _vm_host
        client = self.commcell.clients.get(_vm_host)
        self.prop_dict["server_name"] = client.client_hostname
        return client.client_hostname

    @property
    def nic_count(self):
        """
        Returns: Nic count of the VM
        """
        return self.nic

    def recheck_vm_host(self):
        self.vmserver_host_name = self.vm_host
        self.host_machine = machine.Machine(
            self.server_client_name, self.commcell)

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
        fetches all the properties of the VM

        Args:
                should have code for two possibilties

                Basic - Basic properties of VM like HostName,
                            especially the properties with which VM can be added as dynamic content

                All   - All the possible properties of the VM

                os_info - Set the property VMGuestOS for creating OS Object

                force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

        exception:
                if failed to get all the properties of the VM
        """

        try:
            self.power_on()
            attempt = 1
            if not self._basic_props_initialized or force_update:
                self._get_vm_info(prop)
            if os_info or prop == 'All':
                while attempt < 5:
                    if self.ip is None or self.ip == "":
                        self._get_vm_info(prop="ip")
                        attempt += 1
                    else:
                        break
                self.vm_guest_os = self.guest_os
                self.get_drive_list()

        except Exception as err:
            self.log.exception("Failed to Get  the VM Properties of the VM")
            raise Exception(err)

    def _get_vm_host(self, server_list):
        """
        from the list of
        """
        try:
            server_name = None
            for _each_server in server_list:
                client = self.commcell.clients.get(_each_server)
                _ps_path = os.path.join(self.utils_path, self.vm_props_file)
                self.prop_dict["server_name"] = client.client_hostname
                self.prop_dict["property"] = "GetAllVM"
                attempt = 0
                while attempt < 5:
                    attempt += 1
                    try:
                        self.log.info("Getting Server Machine object  for %s  attempt: %s", client.client_hostname,
                                      attempt)
                        server_machine = machine.Machine(
                            client, self.commcell)
                        output = server_machine._execute_script(_ps_path, self.prop_dict)
                        _psoutput = output.output
                        _stdout = _psoutput.rsplit("=", 1)[1]
                        _stdout = _stdout.strip()
                        _temp_vm_list = _stdout.split(",")
                    except Exception as err:
                        if attempt < 5:
                            self.log.error("Retrying as following error occurred : %s", str(err))

                        else:
                            self.log.error(err)
                            raise Exception(err)
                    else:
                        break
                for each_vm in _temp_vm_list:
                    if each_vm != "":
                        each_vm = each_vm.strip()
                        if re.match("^[A-Za-z0-9_-]*$", each_vm):
                            if each_vm == self.vm_name:
                                server_name = _each_server
                                break
                            else:
                                continue
                        else:
                            self.log.info(
                                "Unicode VM are not supported for now")

            if not server_name:
                server_name = server_list[0]
                self.log.info("server cannot be identified , it can be a coversion case ")
                self.vm_exist = False

            self.log.info("Server where VM is located is {0}".format(server_name))

            return server_name

        except Exception as err:
            self.log.exception(
                "An exception occurred while getting all Vms from Hypervisor")
            raise Exception(err)

    def _get_vm_info(self, prop, extra_args="$null"):
        """
        get the basic or all or specific properties of VM

        Args:
                prop         -    basic, All or specific property like Memory

                ExtraArgs    - Extra arguments needed for property listed by ","

        exception:
                if failed to get all the properties of the VM

        """
        try:

            self.log.info(
                "Collecting all the VM properties for VM %s" % self.vm_name)
            _ps_path = os.path.join(self.utils_path, self.vm_props_file)
            self.prop_dict["property"] = prop
            self.prop_dict["extra_args"] = extra_args
            output = self.host_machine._execute_script(_ps_path, self.prop_dict)
            _stdout = output.output
            self.log.info("output of all vm prop is {0}".format(_stdout))
            if _stdout != "":
                if ";" in _stdout:
                    stdlines = _stdout.split(';')
                else:
                    stdlines = [_stdout.strip()]
                for _each_prop in stdlines:
                    key = _each_prop.split("=")[0]
                    val = _each_prop.split("=")[1]
                    val = val.strip()
                    if val == "":
                        val = None

                    if key == "ip":
                        setattr(self, key, val.split(" ")[0])
                    elif key == "memory":
                        setattr(self, key, str(int(int(val)/1024)))
                    else:
                        setattr(self, key, val)
            self._basic_props_initialized = True

        except Exception as err:
            self.log.exception("Failed to Get all the VM Properties of the VM")
            raise Exception(err)

    def get_vhdx(self):
        """
        Gets all controller ,associated hard disk and type of hard disk

        returns :
            Dictionary with key as controller and value is with path of associated disk and type of disk
            ex : {'IDE00' : ['C:\virtualdisk.vhd' , 'Dynamic']}
         exception:
                if failed to get property
        """

        try:
            _ps_path = os.path.join(self.utils_path, self.vm_props_file)
            self.prop_dict["property"] = 'Vhdx'

            output = self.host_machine._execute_script(_ps_path, self.prop_dict)
            _stdout = output.output
            if _stdout != "":
                disk_info = _stdout[0:-1]
                return eval(disk_info)
            else:
                raise Exception("Error while getting disk type details")
        except Exception as err:
            self.log.exception("Failed to Get VDHX popertires")
            raise Exception(err)

    @property
    def _get_disk_list(self):
        """
        get the list of disk in the VM

        Returns:
                _disk_dict : with keys as disk name and value as snapshot associated with diskname

                _diks_dict = {
                        "test1.vhdx":["test1_184BDFE9-1DF5-4097-8BC3-06128C581C42.avhdx",
                        "test1_184BDEF9-1DF5-4097-8BC3-06128C581c82.avhdx"]
                }

        """
        try:
            _disk_dict = {}
            if self.disk_path is None:
                self._get_vm_info('disk_path')

            if self.disk_path is not None:
                if "," in self.disk_path:
                    temp_list = self.disk_path.split(",")

                else:
                    temp_list = [self.disk_path]

                for each_disk_list in temp_list:
                    final_disk = (each_disk_list.split("::")[0]).strip()
                    _temp_disk_list = (each_disk_list.split("::")[1]).strip()
                    if " " in _temp_disk_list:
                        _disk_list = _temp_disk_list.split(" ")
                    else:
                        _disk_list = [_temp_disk_list]

                    _disk_dict[final_disk] = _disk_list

            else:
                self.log.info("Cannot collect Disk  Path information")

            return _disk_dict
        except Exception as err:
            self.log.exception("Failed to Get all the VM Properties of the VM")
            raise Exception(err)

    @property
    def set_no_of_cpu(self):
        """Return the mber cpu set in vm"""
        return self._set_no_of_cpu

    @set_no_of_cpu.setter
    def set_no_of_cpu(self, count):
        """
        set the number of processor for  the VM.

         Args:
                count            (int)    - Number of CPU to be set in VM


        Exception:
                When setting cpu count fails

        """

        try:

            vm_name = self.vm_name

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "SetVmProcessor"
            self.operation_dict["vm_name"] = vm_name
            self.operation_dict["extra_args"] = count
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if "Success" in _stdout:
                self._set_no_of_cpu = count
            else:
                self.log.error("The error occurred %s" % _stdout)
                raise Exception("Exception in setting number of cpu")

        except Exception as err:
            self.log.exception("Exception in setting the processor count of VM {0}".format(err))
            raise err

    def _merge_vhd(self, vhd_name):
        """
        merge all the snapshot disk with base disk

        Args:
                vhd_name (str)    - name of the VHD which all snapshots has to be merged

        Return:
                disk_merge     (bool)    - true when merge is success
                                                          False when Merge fails

                base_vhd_name    (str)- base Vhd Name after merging snapshots

        """
        try:
            disk_merge = True
            do_merge = False
            _base_vhd_name = None

            for each_key in self.disk_dict.keys():
                if (os.path.basename(vhd_name)) == (os.path.basename(each_key)):
                    _base_src_vhd_name = (self.disk_dict[each_key])[-1]
                    _base_vhd_name = os.path.join(os.path.dirname(
                        vhd_name), os.path.basename(_base_src_vhd_name))
                    do_merge = True
                    break

            if do_merge:
                _ps_path = os.path.join(
                    self.utils_path, self.vm_operation_file)
                self.operation_dict["operation"] = "Merge"
                self.operation_dict["extra_args"] = vhd_name, _base_vhd_name
                output = self.host_machine._execute_script(_ps_path, self.operation_dict)
                _stdout = output.output
                if _stdout != 0:
                    self.log.info(
                        "Failed to marge disk but still will try to mount it")
                    disk_merge = False

            else:
                self.log.info(
                    "Cannot find the disk at all please check the browse")
                return False, None

            return disk_merge, _base_vhd_name

        except Exception as err:
            self.log.exception("Failed to Get all the VM Properties of the VM")
            raise Exception(err)

    def mount_vhd(self, vhd_name, destination_client=None):
        """
        Mount the VHD provided

        Args:
                vhd_name            (str)    - vhd name that has to be mounted


                destination_client  (obj)   - client where the disk to be mounted are located

        returns:
                _drive_letter_list    (list)    - List of drive letters that is retuned after mount
                                                        [A:,D:]

                                                        Fasle    - if failed to mount
        """
        try:
            _drive_letter_list = []

            if not destination_client:
                destination_client = self.host_machine

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "MountVHD"
            self.operation_dict["vhd_name"] = vhd_name.strip()
            output = destination_client._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output

            if "Success" in _stdout:
                _stdout = _stdout.split("\n")
                for line in _stdout:
                    if "DriveLetter" in line:
                        _drive_letter = line.split("=")[1]
                        _drive_letter = _drive_letter.strip()
                        if "," in _drive_letter:
                            _temp_drive_letter_list = _drive_letter.split(",")
                        else:
                            _temp_drive_letter_list = [_drive_letter]

                for each_drive in _temp_drive_letter_list:
                    if each_drive:
                        each_drive = each_drive + ":"
                        _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
                        self.operation_dict["operation"] = "DriveSize"
                        self.operation_dict["extra_args"] = each_drive
                        output = destination_client._execute_script(_ps_path, self.operation_dict)
                        _stdout = output.output
                        _stdout = _stdout.split("\n")
                        for line in _stdout:
                            if "DriveSize" in line:
                                size = float(line.split("=")[1])
                                if size > 600:
                                    _drive_letter_list.append(each_drive)

                for each_drive in _drive_letter_list:
                    self.log.info("drive letter %s" % each_drive)

                return _drive_letter_list
            else:
                self.log.error("The error occurred %s" % _stdout)
                return False

        except Exception as err:
            self.log.exception("Exception in MountVM")
            raise Exception("Exception in MountVM:{0}".format(err))

    def un_mount_vhd(self, vhd_name):
        """
        Un-mount the vhd name provided

        args:
                vhd_name : vhd needs to be unmounted

        return:
                True    (bool)         - if vhd is unmounted

        Exception:
                if fails to unmount

        """

        try:
            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "UnMountVHD"
            self.operation_dict["vhd_name"] = vhd_name.strip()
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if "Success" in _stdout:
                return True
            else:
                self.log.error("The error occurred %s" % _stdout)
                raise Exception("Exception in UnMountVM")

        except Exception as err:
            self.log.exception("Exception in UnMountVM")
            raise Exception("Exception in UnMountVM:{0}".format(err))

    def power_on(self):
        """
        power on the VM.

        return:
                True - when power on is successful

        Exception:
                When power on failed

        """

        try:

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "PowerOn"
            attempt = 0
            while attempt < 5:
                attempt = attempt + 1
                output = self.host_machine._execute_script(_ps_path, self.operation_dict)
                _stdout = output.output
                if "Success" in _stdout:
                    return True
                else:
                    time.sleep(60)
                    self.log.error(" Error occurred : %s" % _stdout)
            if "Success" in _stdout:
                return True
            else:
                self.log.error("The error occurred %s" % _stdout)
                raise Exception("Exception in PowerOn")

        except Exception as err:
            self.log.exception("Exception in PowerOn")
            raise Exception("Exception in PowerOn:{0}".format(err))

    def power_off(self):
        """
        power off the VM.

        return:
                True - when power off is successful

        Exception:
                When power off failed

        """

        try:

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "PowerOff"
            attempt = 0
            while attempt < 5:
                attempt = attempt + 1
                output = self.host_machine._execute_script(_ps_path, self.operation_dict)
                _stdout = output.output
                if "Success" in _stdout:
                    return True
                else:
                    time.sleep(60)
                    self.log.error(" Error occurred : %s" % _stdout)

            if "Success" in _stdout:
                return True
            else:
                self.log.error("The error occurred %s" % _stdout)
                raise Exception("Exception in PowerOff")

        except Exception as err:
            self.log.exception("Exception in PowerOff")
            raise Exception("Exception in PowerOff:" + str(err))

    def delete_vm(self, vm_name=None):
        """
        Delete the VM.

        return:
                True - when Delete  is successful
                False -  when delete is failed

        """

        try:

            if vm_name is None:
                vm_name = self.vm_name

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "Delete"
            self.operation_dict["vm_name"] = vm_name
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if "Success" in _stdout:
                return True
            else:
                self.log.error("The error occurred %s" % _stdout)
                return False

        except Exception as err:
            self.log.exception("Exception in DeleteVM {0}".format(err))
            return False

    @property
    def set_memory(self):
        """ return the memory which has set in the vm """
        return self._set_memory

    @set_memory.setter
    def set_memory(self, memory):
        """
        Set Startup Memory of  the VM.

                memory            (int)    -  memory in MB to be set in VM


        Execption:
                when setting memory failed

        """

        try:

            vm_name = self.vm_name

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "SetVmMemory"
            self.operation_dict["vm_name"] = vm_name
            self.operation_dict["extra_args"] = memory
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if "Success" in _stdout:
                self._set_memory = memory
            else:
                self.log.error("The error occurred %s" % _stdout)
                raise Exception("Exception in setting memory")

        except Exception as err:
            self.log.exception("Exception in setting Memory of VM {0}".format(err))
            raise err

    def get_disk_in_controller(self, controller_type, number, location):
        """
        get the disk assocaited with controller

        Args:
                controller_type (str)    - IDE/SCSI

                number            (int)    - IDE(1:0) 1 is the disk number

                location        (int)    - IDE(1:0) 0 is the location in disk number 1

        Return:
                DiskType    (str)    - diks in location of args(eg: disk in IDE(1:0))

        """
        try:
            _extr_args = "%s,%s,%s" % (controller_type, number, location)
            self._get_vm_info("DiskType", _extr_args)
            return self.DiskType

        except Exception as err:
            self.log.exception("Exception in GetDiskInController")
            raise err

    def get_disk_path_from_pattern(self, disk_pattern):
        """
        find the disk that matches the disk apttern form disk list

        Args:
                disk_pattern    (str)    - pattern which needs to be matched

        Return:
                eachdisk    (str)        - the disk that matches the pattern
        """
        try:
            _disk_name = os.path.basename(disk_pattern)
            for each_disk in self.DiskList:
                _vm_disk_name = os.path.basename(each_disk)
                if _vm_disk_name == _disk_name:
                    self.log.info("Found the Disk to be filtered in the VM")
                    return each_disk

        except Exception as err:
            self.log.exception("Exception in GetDiskInController")
            raise err

    def migrate_vm(self, vm_name=None):
        """
        Migrate VM to best possible node

        Return:
                NewHost (String) - Hostname the VM migrated to

        Exception:
                An error occurred while checking/creating the registry
        """
        try:
            if vm_name is None:
                vm_name = self.vm_name

            _ps_path = os.path.join(self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "MigrateVM"
            self.operation_dict["vm_name"] = vm_name
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if "failed" in _stdout:
                self.log.error("The error occurred %s" % _stdout)

        except Exception as err:
            self.log.exception(
                "An error occurred migrating the VM")
            raise err

    def revert_snap(self, snap_name="Fresh"):
        """
        Revert snap of the machine specified
        :param snap_name: name of the snap to be reverted
        :return:
            true - if revert snap succeds
            False - on Failure
        """
        try:
            _ps_path = os.path.join(
                self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "RevertSnap"
            self.operation_dict["extra_args"] = snap_name
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if '0' in _stdout:
                self.log.info("Snapshot revert was successfull")
                return True
            else:
                return False

        except Exception as err:
            self.log.exception("Exception in revert_snap")
            raise err

    def create_snap(self, snap_name="Fresh"):
        """
        Create snap of the machine specified
        :param snap_name: name of the snap to be created
        :return:
            true - if create snap succeds
            False - on Failure
        """
        try:
            _ps_path = os.path.join(
                self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "CreateSnap"
            self.operation_dict["extra_args"] = snap_name
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if '0' in _stdout:
                self.log.info("Snapshot creation was successfull")
                return True
            else:
                return False

        except Exception as err:
            self.log.exception("Exception in Create_snap")
            raise err

    def delete_snap(self, snap_name="Fresh"):
        """
        delete snap of the machine specified
        :param snap_name: name of the snap to be delete
        :return:
            true - if delete snap succeds
            False - on Failure
        """
        try:
            _ps_path = os.path.join(
                self.utils_path, self.vm_operation_file)
            self.operation_dict["operation"] = "DeleteSnap"
            self.operation_dict["extra_args"] = snap_name
            output = self.host_machine._execute_script(_ps_path, self.operation_dict)
            _stdout = output.output
            if '0' in _stdout:
                self.log.info("Snapshot deletion  was successfull")
                return True
            else:
                return False

        except Exception as err:
            self.log.exception("Exception in delete_snap")
            raise err

    def get_vm_generation(self):
        """Gets the generation of VM
           Retruns   (str):  generation of vm

        """
        return self.Generation

    def clean_up(self):
        """
        Does the cleanup after the testcase.

        Raises:
            Exception:
                When cleanup failed or unexpected error code is returned

        """

        try:
            self.log.info("Powering off VMs after restore")
            self.power_off()
        except Exception as exp:
            raise Exception("Exception in Cleanup: {0}".format(exp))
