# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for Google Cloud vm"""

import socket
import requests
from AutomationUtils import machine
from AutomationUtils import logger
from VirtualServer.VSAUtils.VMHelper import HypervisorVM


class GoogleCloudVM(HypervisorVM):
    """
    This is the main file for all Google Cloud VM operations

    Methods:

        _get_vm_info()            -  get the information about the particular VM

        power_off()               -  power off the VM

        power_on()                -  power on the VM

        restart_vm()              -  restarts the VM

        delete_vm()               -  delete the VM

        get_vm_guid()             -  gets the GUID of the VM

        get_nic_info()            -  get all networks attached to that VM

        get_disk_list()           -  gets all the disks attached to the VM

        get_disk_info()           -  get all the disk info of VM

        get_disk_size()           -  gets the total disk size of all disks in VM

        get_status_of_vm()        - get the status of VM like started, stopped

        get_OS_type()             - update the OS type of the VM

        get_subnet_ID()           - Update the subnet_ID for VM

        get_IP_address()          - gets the internal and external IP addresses of the VM

        update_vm_info()          - updates the VM info

    """

    def __init__(self, hvobj, vm_name):
        """
         Initialization of Google Cloud vm properties

         Args:
            vm_name (basestring)    --  name of the VM for which properties can be fetched

            hvobj   (object)        --  Hypervisor class object for Google Cloud

        """

        super(GoogleCloudVM, self).__init__(hvobj, vm_name)
        self.Hvobj = hvobj
        self.vm_name = vm_name
        if self.Hvobj.project:
            self.project_name = self.Hvobj.project
        else:
            self.project_name = self.Hvobj.restore_project
        if 'del' in vm_name:
            self.zone_name = self.Hvobj.get_vm_zone(vm_name, self.project_name)
        else:
            self.zone_name = self.Hvobj.get_vm_zone(vm_name)
        self.access_token = self.Hvobj.access_token
        self._vm_info = {}
        self.nic = []
        self.disk_list = []
        self.ip = None
        self.disk_dict = {}
        self.internal_IPs = []
        self.external_IPs = []
        self._external_ip_enabled_dict = {}
        self.subnet_IDs = []
        self.log = logger.get_log()
        self.google_session = requests.Session()
        self._basic_props_initialized = False
        self._get_vm_info()
        self.update_vm_info()

    class BackupValidation(object):
        def __init__(self, vm_obj, backup_option):
            self.vm = vm_obj
            self.backup_option = backup_option
            self.backup_job = self.backup_option.backup_job
            self.log = logger.get_log()

        def validate(self):
            """Validates the post backup validation"""
            # Snapshot pruning validation
            self.log.info("Post backup validations for Google Cloud")
            if self.vm.Hvobj.check_snapshot_pruning(f'gx-backup-{self.backup_option.backup_job.job_id}'):
                self.log.info(
                    "Google Cloud snapshot pruning validation successful")
            else:
                raise Exception("Google Cloud snapshot pruning validation failed")

            # Disk pruning validation
            if self.vm.Hvobj.check_disk_pruning(f'gx-backup-{self.backup_option.backup_job.job_id}'):
                self.log.info(
                    "Google Cloud disk pruning validation successful")
            else:
                raise Exception("Google Cloud disk pruning validation failed")

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.restore_job = self.vm_restore_options.restore_job
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the source vm and restored vm"""
            try:
                # Disk type validation, pd-standard, pd-ssd, local-ssd etc
                if len(self.vm.vm.disk_dict) != len(other.vm.vm.disk_dict):
                    self.log.info("Disk validation failed")
                    return False

                for disk_name in other.vm.vm.disk_dict:
                    source_disk_name = "-".join(disk_name.split("-")[:-2])
                    disk_type_source = self.vm.vm.disk_dict[source_disk_name]['type'].split("/")[-1]
                    disk_type_dest = other.vm.vm.disk_dict[disk_name]['type'].split("/")[-1]
                    if disk_type_source != disk_type_dest:
                        self.log.info(f"Disk type validation failed for source disk: {source_disk_name}")
                        return False
                self.log.info("Disk type validation successful")

                # Public IP validation
                if self.vm.vm.public_ip_enabled != other.vm.vm.public_ip_enabled:
                    self.log.info("Public IP validation failed")
                    return False
                self.log.info("Public IP validation successful")

                # Google cloud multi project restore snapshot pruning validation
                proxy_machine = machine.Machine(machine_name=self.vm_restore_options.proxy_client,
                                                commcell_object=self.vm.vm.Hvobj.commcell)
                proxy_machine_name = socket.getfqdn(proxy_machine.ip_address).split(".")[0]
                proxy_project = self.vm.vm.Hvobj.get_project_by_instance_name(proxy_machine_name)
                if proxy_project != other.vm.vm.project_name:
                    if self.vm.vm.Hvobj.check_snapshot_pruning(
                            f'gx-restore-{self.vm_restore_options.restore_job.job_id}'):
                        self.log.info("Google Cloud restore snapshot pruning successful")
                    else:
                        raise Exception("Google Cloud restore snapshot pruning failed")

                    if self.vm.vm.Hvobj.check_disk_pruning_by_description(proxy_project,
                                                                       self.vm_restore_options.restore_job.job_id):
                        self.log.info("Disk pruning validation successful")
                    else:
                        raise Exception("Disk pruning validation failed")

                return True

            except Exception as exp:
                self.log.exception("Exception in Vm Validation")
                raise Exception("Exception in Vm Validation:" + str(exp))

    @property
    def nic_count(self):
        """
                To fetch the nic count of the VM
                Returns:
                    nic_count         (Integer): Count of nic
                """
        return len(self.nic)

    @property
    def google_vmurl(self):
        google_vmurl = \
            "https://www.googleapis.com/compute/v1/projects" \
            "/{}/zones/{}/instances/{}/".format(self.project_name, self.zone_name, self.vm_name)

        return google_vmurl

    @property
    def default_headers(self):
        self.Hvobj._headers = {'Authorization': 'Bearer %s' % self.access_token}
        return self.Hvobj._headers

    @property
    def public_ip_enabled(self):
        if self._external_ip_enabled_dict == {}:
            for network_interface in self._vm_info['networkInterfaces']:
                if "accessConfigs" in network_interface and "natIP" in network_interface["accessConfigs"]:
                    if network_interface["accessConfigs"]["natIP"] != "":
                        self._external_ip_enabled_dict[network_interface['name']] = True
                self._external_ip_enabled_dict[network_interface['name']] = False
        return self._external_ip_enabled_dict

    def clean_up(self):
        """
        Clean up the VM and its reources.

        Raises:
            Exception:
                When cleanup failed or unexpected error code is returned

        """
        try:
            # if 'del' in self.vm_name:
            #     self.log.info("Deleting Restore VM : {0}".format(self.vm_name))
            #     self.delete_vm()
            # else:
            self.log.info("Powering off VMs after restore{0}".format(self.vm_name))
            self.power_off()

        except Exception as exp:
            self.log.exception("Exception in Cleanup : {0}".format(str(exp)))
            raise Exception("Exception in Cleanup:" + str(exp))

    def _get_vm_info(self):
        """
        Gets all VM info related to the given VM

        Raises:
            Exception:
                If the vm data cannot be collected

        """
        try:
            self.log.info("Getting all information of VM [%s]" % (self.vm_name))
            self._vm_info = False
            vm_infourl = self.google_vmurl
            self.log.info("Google APi url is : '{0}'".format(vm_infourl))
            data = self.Hvobj._execute_google_api(vm_infourl)
            self._vm_info = data

        except Exception as err:
            self.log.exception("Exception in _get_vm_info : {0}".format(str(err)))
            raise Exception(err)

    def power_on(self):
        """
        Powers on the VM

        Returns:
            ret_code    (bool)      -   True if the VM is powered on

        Raises:
            Exception:
                If the VM cannot be powered on

        """
        try:
            self.log.info("powering on vm [%s]" % (self.vm_name))
            vmurl = self.google_vmurl + "start"
            _ = self.Hvobj._execute_google_api(vmurl, 'post')
            ret_code = True
            return ret_code

        except Exception as err:
            self.log.exception("Exception in power on: {0}".format(str(err)))
            raise Exception(err)

    def power_off(self):
        """
        Powers off the VM

        Returns:
            ret_code    (bool)      -   True if the VM is powered off

        Raises:
            Exception:
                If the VM cannot be powered off

        """
        try:
            self.log.info("powering off vm [%s]" % (self.vm_name))
            vmurl = self.google_vmurl + "stop"
            _ = self.Hvobj._execute_google_api(vmurl, 'post')
            ret_code = True
            return ret_code
        except Exception as err:
            self.log.exception("Exception in power off: {0}".format(str(err)))
            raise Exception(err)

    def restart_vm(self):
        """
        Resets the VM

        Returns:
            ret_code    (bool)      -   True if the VM is reset

        Raises:
            Exception:
                If the VM cannot be reset

        """
        try:
            self.log.info("powering on vm [%s]" % (self.vm_name))
            vmurl = self.google_vmurl + "reset"
            _ = self.Hvobj._execute_google_api(vmurl, 'post')
            ret_code = True
            return ret_code

        except Exception as err:
            self.log.exception("Exception in Restart VM: {0}".format(str(err)))
            raise Exception(err)

    def delete_vm(self):
        """
        Deletes the VM

        Returns:
            ret_code    (bool)      -   True if the VM is deleted

        Raises:
            Exception:
                If the VM cannot be deleted

        """
        try:
            self.log.info("Deleting VM [%s]" % (self.vm_name))
            vmurl = self.google_vmurl
            _ = self.Hvobj._execute_google_api(vmurl, 'delete')
            ret_code = True
            return ret_code

        except Exception as err:
            self.log.exception("Exception in Delete VM: {0}".format(str(err)))
            raise Exception(err)

    def _get_status_of_vm(self):
        """
        gets status of VM, e.g. running, stopped, etc

        Raises:
            Exception:
                if status of vm cannot be found
        """
        try:
            self.log.info("Getting status of VM [%s]" % (self.vm_name))
            vm_infourl = self.google_vmurl
            data = self.Hvobj._execute_google_api(vm_infourl)
            self.vm_state = data['status']

        except Exception as err:
            self.log.exception("Exception in _get_vm_info: {0}".format(str(err)))
            raise Exception(err)

    def _get_nic_info(self):
        """
        Gets all networks attached to VM

        Raises:
            Exception:
                If the nic_info cannot be found
        """
        try:
            self.log.info("Getting the network cards info for VM %s" % self.vm_name)
            data = self._vm_info
            all_nic_info = data["networkInterfaces"]
            if self.nic != []:
                self.nic = []
            for nic in all_nic_info:
                nic_name = nic['network'].rpartition('/')[2]
                self.nic.append(nic_name)

        except Exception as err:
            self.log.exception("Exception in get_nic_info: {0}".format(str(err)))
            raise Exception(err)

    def _get_disk_list(self):
        """
        Gets all the disks attached to VM
        """
        data = self._vm_info
        if self.disk_list != []:
            self.disk_list = []
        for disk in data['disks']:
            disk_name = disk['source'].rpartition('/')[2]
            self.disk_list.append(disk_name)
        self.disk_count = len(self.disk_list)
        return self.disk_list

    def _get_disk_info(self):
        """
        Get all info about disks attached to VM

        Raises:
            Exception:
                If the disk info cannot be found
        """
        try:
            data = self._vm_info
            for disk in data['disks']:
                disk_info_url = disk['source']
                disk_name = disk['source'].rpartition('/')[2]
                data = self.Hvobj._execute_google_api(disk_info_url)
                self.disk_dict[disk_name] = data

        except Exception as err:
            self.log.exception("Exception in get_disk_info: {0}".format(str(err)))
            raise Exception(err)

    def _get_disk_size(self):
        """
        gets the total used space of the VM

        Raises:
            Exception:
                If the disk size cannot be determined
        """
        try:

            total_disk_size = 0

            if self.disk_list == []:
                self.get_disk_list()
                self.get_disk_info()

            for disk in self.disk_list:
                disk_size = int(self.disk_dict[disk]['sizeGb'])
                total_disk_size += disk_size

            self.vm_size = total_disk_size

        except Exception as err:
            self.log.exception("Exception in get_disk_size: {0}".format(str(err)))
            raise Exception(err)

    def _get_vm_guid(self):
        """
        gets the GUID of VM

        Raises:
            Exception:
                If the guid cannot be retrieved
        """
        try:
            self.log.info("Getting the size information for VM %s" % self.vm_name)
            data = self._vm_info
            self.guid = data['id']

        except Exception as err:
            self.log.exception("Exception in get_vm_guid: {0}".format(str(err)))
            raise Exception(err)

    def _get_IP_address(self):
        """
        gets the internal and external  (if it has one) IPs of the VM

        Raises:
            Exception:
                if the ip information cannot be retrieved
        """

        try:
            data = self._vm_info
            for internal in data['networkInterfaces']:
                self.ip = internal['networkIP']
                if 'accessConfigs' in internal:
                    for external in internal['accessConfigs']:
                        if 'natIP' in external:
                            self.ip = external['natIP']
                        else:
                            self.external_IPs = None
                else:
                    self.external_IPs = None

        except Exception as err:
            self.log.exception("Exception in get_vm_ips: {0}".format(str(err)))
            raise Exception(err)

    def _get_OS_type(self):
        """
        Updates the OS type

        Raises:
            Exception:
                if the os type cannot be retrieved
        """
        try:
            self.log.info("Getting the os disk info for VM %s" % self.vm_name)
            data = self._vm_info['disks'][0]['licenses'][0]
            disk_os = data.rpartition('/')[2]
            if "windows" in disk_os:
                self.guest_os = "windows"
            # include for unix VMs also
            self.log.info("OS type is : %s" % self.guest_os)

        except Exception as err:
            self.log.exception("Exception in GetOSType: {0}".format(str(err)))
            raise Exception(err)

    def _get_subnet_ID(self):
        """
        Update the subnet ID for VM

        Raises:
            Exception:
                if the subent id cannot be colelcted
        """
        try:
            data = self._vm_info
            if self.subnet_IDs != []:
                self.subnet_IDs = []
            for network in data['networkInterfaces']:
                subnet_url = network['subnetwork']
                subnet_data = self.Hvobj._execute_google_api(subnet_url)
                self.subnet_IDs.append(subnet_data['ipCidrRange'])

        except Exception as err:
            self.log.exception("Exception in GetSubnetID: {0}".format(str(err)))
            raise Exception(err)

    def _get_cpu_memory(self):
        data = self.Hvobj._execute_google_api(self.google_vmurl)
        machine_type = data['machineType'].rpartition('/')[2]
        if 'custom' in machine_type:
            self.no_of_cpu = machine_type.split('-')[1]
            self.memory = int(machine_type.split('-')[2]) / 1024
        else:
            self.no_of_cpu = self.memory = machine_type

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
        fetches properties of the VM
        Args:
            prop                (str):  Basic - Basic properties of VM like HostName,
                                                especially the properties with which
                                                VM can be added as dynamic content

                                        All   - All the possible properties of the VM

            os_info             (bool): To fetch os info or not

            force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

        Raises:
            Exception:
                if failed to update all the properties of the VM
        """
        try:
            self.power_on()
            if self._vm_info:
                if not self._basic_props_initialized or force_update:
                    self._get_vm_guid()
                    self._get_disk_list()
                    self._get_disk_info()
                    self._get_disk_size()
                    self._get_status_of_vm()
                    self._get_cpu_memory()
                if prop == 'All':
                    self._get_IP_address()
                    self._get_nic_info()
                    self._get_subnet_ID()
                    self._get_OS_type()
                    self.vm_guest_os = self.guest_os
                    self.get_drive_list()
                    self._get_disk_list()
                    self._get_disk_info()
            else:
                self.log.info("VM info was not collected for this VM")
        except Exception as err:
            self.log.exception("Failed to update the VM Properties: {0}".format(str(err)))
            raise Exception(err)
