# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for Fusion Compute vm"""

from VirtualServer.VSAUtils.VMHelper import HypervisorVM
from VirtualServer.VSAUtils import FusionComputeServices


class FusionComputeVM(HypervisorVM):
    """
    Class for Fusion Compute  VMs
    """

    def __init__(self, hvobj, vm_name):
        """
         Initialization of vmware vm properties
        :param hvobj:       (obj) Hypervisor class object for Fusion Compute
        :param vm_name:     (str) name of the VM for which properties can be fetched
        """

        super(FusionComputeVM, self).__init__(hvobj, vm_name)
        self.server_name = hvobj.server_host_name
        self.hvobj = hvobj
        self.vm_url = 'http://{0}:7070{1}'.format(self.server_name, self.hvobj.vm_dict[vm_name])
        self._vm_operation_services_dict = FusionComputeServices.get_vm_operation_services(self.server_name,
                                                                                           self.hvobj.vm_dict[vm_name])
        self.guid = None
        self.ip = None
        self.guest_os = None
        self.host_name = None
        self._disk_list = None
        self.disk_path = None
        self._basic_props_initialized = False
        self.update_vm_info()

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
        fetches all the properties of the VM

        Args:
                should have code for two possibilties

                Basic - Basic properties of VM like HostName,
                            especially the properties with which VM can be added as dynamic content

                All   - All the possible properties of the VM

                Set the property VMGuestOS for creating OS Object

                force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

        exception:
                if failed to get all the properties of the VM
        """

        try:
            if not self._basic_props_initialized or force_update:
                self._get_vm_info(prop)

            if prop == 'All':
                self.vm_guest_os = self.guest_os
                self.get_drive_list()

            elif hasattr(self, prop):
                return getattr(self, prop, None)

        except Exception as err:
            self.log.exception("Failed to Get  the VM Properties of the VM with the exception {0}".format(err))
            raise Exception(err)

    def _get_vm_info(self, prop, extra_args=None):
        """
        get the basic or all or specific properties of VM

        Args:
                prop         -    basic, All or specific property like Memory

                ExtraArgs    - Extra arguments needed for property listed by ","

        exception:
                if failed to get all the properties of the VM

        """
        try:
            self.log.info(
                "Collecting all the VM properties for VM %s" % self.vm_name)

            flag, response = self.hvobj._make_request('GET', self.vm_url)
            if flag:
                response_json = response.json()
                self.guid = response_json['uuid']
                self.ip = response_json['vmConfig']['nics'][0]['ip']
                self.guest_os = response_json['osOptions']['osType']
                self.host_name = response_json['osOptions']['hostname']
                self.nic_count = len(response_json['vmConfig']['nics'])
                self.disks = response_json['vmConfig']['disks']
                self.disk_list = [disk['datastoreUrn'] for disk in self.disks]
                self.disk_count = len(self.disks)
                self.no_of_cpu = response_json['vmConfig']['cpu']['quantity']
                self.memory = (response_json['vmConfig']['memory']['quantityMB']) / 1024
                self.vm_space = self._get_disk_size(response_json)
                self._basic_props_initialized = True
                return

            raise Exception("failed with error {0}".format(response.json()))

        except Exception as err:
            self.log.exception(
                "Failed to Get  the VM Properties of the VM with the exception {0}".format(err)
            )
            raise Exception(err)

    def _get_disk_size(self, response):
        """
        get teh total used space of the VM
        :param
            response: response object of VMs API
        :return:
        disk_space  (int)   : total space occupied by VM in GB
        """

        try:
            _disk_list = response['vmConfig']['disks']
            disk_space = 0
            for disk in _disk_list:
                disk_space += disk['quantityGB']

            return disk_space

        except Exception as err:
            self.log.exception(
                "Failed to Get  the VM disk space of the VM with the exception {0}".format(err))
            raise Exception(err)

    def power_on(self):
        """
        power on the VM.

        return:
                True - when power on is successful

        Exception:
                When power on failed

        """

        try:

            flag, response = self.hvobj._make_request('GET',
                                                      self._vm_operation_services_dict['START_VM'])
            if flag:
                return True

            self.log.error("Error occurred while powering on VM {0}".format(response.json()))

        except Exception as exp:
            self.log.exception("Exception in PowerOn{0}".format(exp))
            return False

    def power_off(self):
        """
        power off the VM.

        return:
                True - when power off is successful

        Exception:
                When power off failed

        """

        try:

            flag, response = self.hvobj._make_request('GET',
                                                      self._vm_operation_services_dict['STOP_VM'])
            if flag:
                return True

            self.log.error("Error occurred while powering on VM {0}".format(response.json()))

        except Exception as exp:
            self.log.exception("Exception in PowerOff{0}".format(exp))
            return False

    def delete_vm(self):
        """
        Delete the VM.

        return:
                True - when Delete  is successful
                False -  when delete is failed

        """

        try:

            flag, response = self.hvobj._make_request('GET', self.vm_url)
            if flag:
                return True

            self.log.error("Error occurred while powering on VM {0}".format(response.json()))

        except Exception as exp:
            self.log.exception("Exception in Deleting the VM {0}".format(exp))
            return False

    def restart_vm(self):
        """
        restart the VM.

        return:
                True - when restart  is successful
                False -  when restart is failed

        """

        try:

            flag, response = self.hvobj._make_request('GET',
                                                      self._vm_operation_services_dict['RESTART_VM'])
            if flag:
                return True

            self.log.error("Error occurred while restarting on VM {0}".format(response.json()))

        except Exception as exp:
            self.log.exception("Exception in restarting the VM {0}".format(exp))
            return False
