# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Does all the Operation for Azure vm

    AzureVM:

            get_drive_list()                -  get the drive list for the VM

            _get_vm_info()                  -  get the information about the particular VM

            get_vm_guid()                   -  gets the GUID of Azure VM

            get_nic_info()                  -  get all network attached to that VM

            get_VM_size()                   -  gets the size of the Azure VM

            get_cores()                     -  get the cores of the VM

            get_Disk_info()                 -  get all the disk info of VM

            get_status_of_vm()              - get the status of VM like started.stopped

            get_OS_type()                   - update the OS type of the VM

            get_subnet_ID()                 - Update the subnet_ID for VM

            get_IP_address()                - gets the IP address of the VM

            update_vm_info()                - updates the VM info

            get_snapshotsonblobs()          - gets all snapshots associate with blobs of VM

            get_disk_snapshots()            - gets all snapshots associated with disks of VM

            get_snapshots()                 - gets all snapshots on blob and disk of VM

            get_bloburi()                   - gets uri of blob associated with VM

            get_publicipid()                - gets id pub;ic id associated with VM

            check_vmsizeof_destination_vm() - Checks destination  vmsize as mentioned in schedule

            check_storageacc_destination_vm() - Checks destination storage account as mentioned in schedule

            check_regionof_destinationvm()    -  Checks destination vm region is same as mentioned in schedule

            check_publicipof_destinationvm()  - Checks if destination vm has public ip if mentioned in schedule

            get_snapshot_jobid()              - Gets the job id of job that created the dik snapshot

            get_nic_details()                 - Gets details of network interface attacked to vm

            get_vm_generation()               - Gets VM Generation

            check_gen_of_vm()                 - Checks if restored and source VM have same generation


"""

import time
import json
import os
import re
import datetime
import xmltodict
from AutomationUtils import machine
from AutomationUtils import logger
from VirtualServer.VSAUtils.VMHelper import HypervisorVM
from VirtualServer.VSAUtils import VirtualServerUtils


class AzureVM(HypervisorVM):
    """
    This is the main file for all AzureRM VM operations
    """

    def __init__(self, hvobj, vm_name):
        """
        Initialization of AzureRM VM properties

        Args:

            hvobj           (obj):  Hypervisor Object

            vm_name         (str):  Name of the VM

        """
        import requests
        super(AzureVM, self).__init__(hvobj, vm_name)
        self.azure_baseURL = 'https://management.azure.com'
        self.api_version = "?api-version=2015-06-15"
        self.subscription_id = self.hvobj.subscription_id
        self.app_id = self.hvobj.app_id
        self.xmsdate = None
        self.storage_token = None
        self.tenant_id = self.hvobj.tenant_id
        self.app_password = self.hvobj.app_password
        self.azure_session = requests.Session()
        self.vm_name = vm_name
        self.access_token = self.hvobj.access_token
        self.storage_access_token = None
        self._basic_props_initialized = False
        self.storageaccount_type = None
        self.vm_operation_file = "AzureOperation.ps1"
        self.resource_group_name = self.hvobj.get_resourcegroup_name(self.vm_name)
        self.vm_files_path = self.resource_group_name
        self.network_name = None
        self.subnet_id = None
        self._vm_info = {}
        self.disk_info = {}
        self.disk_dict = {}
        self.nic_count = 0
        self.nic = []
        self.nic_id = []
        self.managed_disk = True
        self._disk_list = None
        self.disk_sku_dict = {}
        self.restore_storage_acc = ''
        self.restore_resource_grp = ''
        self.filtered_disks = []
        self.operation_dict = {
            "subscription_id": self.subscription_id,
            "tenant_id": self.tenant_id,
            "client_id": self.app_id,
            "cred_password": self.app_password,
            "vm_name": self.vm_name,
            "property": "$null",
            "extra_args": "$null"
        }
        self._get_vm_info()
        self.update_vm_info()

    class BackupValidation(object):
        def __init__(self, vm_obj, backup_option):
            self.vm = vm_obj
            self.backup_option = backup_option
            self.backup_job = self.backup_option.backup_job

            self.log = logger.get_log()

        def validate(self):
            self.log.info("Performing Post Backup Snapshot Validation on : {0}".format(self.vm.vm_name))
            if not self.backup_job.details['jobDetail']['clientStatusInfo']['vmStatus'][0]['CBTStatus']:
                snap_exists = self.vm.check_disk_snapshots_by_jobid(self.backup_job)[0]
                if snap_exists:
                    self.log.info("Post backup snapshot validation Failed")
                    raise Exception("Post backup snapshot validation Failed")
                self.log.info("snapshot validation successful")

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.restore_job = self.vm_restore_options.restore_job
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the source vm and restored vm"""
            try:
                if not self.vm.vm.validate_sku(self.vm.vm, other.vm.vm):
                    self.log.info("Disk SKU Validation failed")
                    return False
                if not self.vm.vm.check_vmsizeof_destination_vm(self.vm.vm, other.vm.vm, self.vm_restore_options, vm=True):
                    self.log.info("vmSize validation of vm failed")
                    return False
                self.log.info("vmSize validation of vm passed")
                if not self.vm.vm.check_regionof_destinationvm(self.vm.vm, other.vm.vm, self.vm_restore_options, vm=True):
                    self.log.info("region of vm validation of vm failed")
                    return False
                self.log.info("region of vm validation of vm passed")
                if not self.vm.vm.check_gen_of_vm(other.vm.vm):
                    self.log.info("VM generation  validation of vm failed")
                    return False
                if not self.vm.vm.check_storageacc_destination_vm(self.vm.vm, other.vm.vm, self.vm_restore_options, vm=True):
                    self.log.info("Failure in Validation :Storage Account")
                    return False
                self.log.info("Validation :Storage Account passed")
                if not self.vm.vm.check_publicipof_destinationvm(self.vm.vm, other.vm.vm, self.vm_restore_options, vm=True):
                    self.log.info("Public IP validation of vm failed")
                    return False
                self.log.info("Public IP validation of vm passed")
                if not self.vm.vm.check_disk_type(self.vm.vm, other.vm.vm, self.vm_restore_options):
                    self.log.info("Disk Type validation of vm failed")
                    return False
                self.log.info("Disk Type validation of vm passed")

                if not self.vm.vm.nic_count == other.vm.vm.nic_count:
                    self.log.info("Nic Count validation failed")
                    return False
                self.log.info("Nic Count validation passed")
                return True

            except Exception as exp:
                self.log.exception("Exception in Vm Validation")
                raise Exception("Exception in Vm Validation:" + str(exp))

    class LiveSyncVmValidation(object):
        def __init__(self, vmobj, schedule, replicationjob):
            self.vm = vmobj
            self.schedule = schedule
            self.replicationjob = replicationjob
            self.log = logger.get_log()

        def __eq__(self, other):
            """ performs hypervisor specific LiveSync validation """
            try:
                # Blob snap shot validation
                destsnaps = other.vm.vm.get_snapshotsonblobs()
                startime = self.replicationjob.start_time
                endtime = self.replicationjob.end_time
                for disk in destsnaps:
                    snap_exists = False
                    if len(destsnaps[disk]) <= 1:
                        snap_exists = other.vm.vm.check_snap_exist_intimeinterval(destsnaps[disk],
                                                                                  startime, endtime)
                    else:
                        self.log.error("More than snapshot exist , please check the snapshot tree for this VM ")
                        raise Exception("More than one snapshot exist")

                    if not snap_exists:
                        self.log.info("snapshot validation Failed")
                        return False

                self.log.info("snapshot validation successful")

                vmsize_check = self.vm.vm.check_vmsizeof_destination_vm(self.schedule, other.vm.vm)
                if not vmsize_check:
                    self.log.info("vmSize validation of vm failed")
                    return False
                region_check = self.vm.vm.check_regionof_destinationvm(self.schedule, other.vm.vm)
                if not region_check:
                    self.log.info("region of vm validation of vm failed")
                    return False
                check_vmgen = self.vm.vm.check_gen_of_vm(other.vm.vm)
                if not check_vmgen:
                    self.log.info("VM generation  validation of vm failed")
                    return False
                storageacc_check = self.vm.vm.check_storageacc_destination_vm(self.schedule, other.vm.vm)
                if not storageacc_check:
                    self.log.info("storage account  validation of vm failed")
                    return False
                publicip_check = self.vm.vm.check_publicipof_destinationvm(self.schedule, other.vm.vm)
                if not publicip_check:
                    self.log.info("publicip  validation of vm failed")
                    return False

                return True

            except Exception as exp:
                self.log.exception("Exception in LiveSyncValidation")
                raise Exception("Exception in LiveSyncValidation:" + str(exp))

    class VmConversionValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the restored vm with user inputs"""
            return other.vm_restore_options.datacenter == self.vm_restore_options.datacenter and \
                   other.vm_restore_options.Storage_account == self.vm_restore_options.Storage_account
    @property
    def azure_vmurl(self):
        """
            The azure URL for the VM, for making API calls
        """

        if self.resource_group_name is None:
            self.hvobj.get_resourcegroup_name(self.vm_name)

        azure_vmurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                      "/%s/providers/Microsoft.Compute/virtualMachines" \
                      "/%s" % (self.subscription_id, self.resource_group_name, self.vm_name)
        return azure_vmurl

    @property
    def azure_diskurl(self):
        """
        The azure disk URL for making API calls
        """
        if self.resource_group_name is None:
            self.hvobj.get_resourcegroup_name(self.vm_name)

        azure_diskurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                        "/%s/providers/Microsoft.Compute/disks/" % (self.subscription_id, self.resource_group_name)
        return azure_diskurl

    @property
    def vm_info(self):
        """
            It is used to fetch VM info. This is read only property
        """
        if self._vm_info[self.vm_name] == {}:
            self._get_vm_info()
        return self._vm_info[self.vm_name]

    @vm_info.setter
    def vm_info(self, value):
        """
             This is to set vmname for VM info
        """
        self._vm_info[self.vm_name] = value

    @property
    def vm_hostname(self):
        """gets the vm hostname as IP (if available or vm name). It is a read only attribute"""

        if self.vm_name[0:3] == 'del':
            return self.vm_name[3:]
        else:
            return self.vm_name

    @property
    def access_token(self):
        """
            Returns access token for session. This is read only property
        """
        self.hvobj.check_for_access_token_expiration()
        return self.hvobj.access_token

    @access_token.setter
    def access_token(self, token):
        """
            This is to set Access token for current session
        """
        self.hvobj._access_token = token

    @property
    def default_headers(self):
        """
            Returns the default header for making API calls. This is read only property
        """

        self.hvobj._default_headers = {"Content-Type": "application/json",
                                       "Authorization": "Bearer %s" % self.access_token}
        return self.hvobj._default_headers

    @property
    def storage_data(self):
        """
            Returns the default data parameter for making API calls for storage accounts. This is read only property
        """

        self.hvobj._storage_data = {"grant_type": "client_credentials",
                                    "client_id": self.app_id,
                                    "client_secret": self.app_password,
                                    "resource": "https://storage.azure.com"}
        return self.hvobj._storage_data

    @property
    def storage_headers(self):
        """
            Returns the default header for making API calls for storage account. This is read only property
        """

        self.hvobj._storage_headers = {"Authorization": "Bearer %s" % self.storage_token,
                                       "x-ms-date": "%s" % self.xmsdate,
                                       "x-ms-version": "2018-03-28",
                                       "x-ms-delete-snapshots": "include"}
        return self.hvobj._storage_headers

    @property
    def disk_list(self):
        """to fetch the disk in the VM
        Return:
            disk_list   (list)- list of disk in VM

        """
        if self.disk_dict:
            self._disk_list = self.disk_dict.keys()

        else:
            self._disk_list = []

        return self._disk_list

    def _set_credentials(self, os_name):
        """
        set the credentials for VM by reading the config INI file
        """
        machine_obj = machine.Machine(commcell_object=self.commcell)
        try:
            machine_obj.remove_host_file_entry(self.vm_hostname)
        except:
            self.log.error("Soft Error : Error in removing host file entry")
            pass
        machine_obj.add_host_file_entry(self.vm_hostname, self.ip)
        retry = 0
        while retry < 6:
            try:
                self.log.info("Pinging the vm and getting os info Attempt : {0}".format(retry))
                os_name = self.get_os_name(self.vm_hostname)
                break
            except Exception as err:
                self.log.info("OS Info wasn't updated. Trying again")
                time.sleep(80)
                retry = retry + 1

        if self.user_name and self.password:
            try:
                vm_machine = machine.Machine(self.vm_hostname,
                                             username=self.user_name,
                                             password=self.password)
                if vm_machine:
                    self.machine = vm_machine
                    return
            except:
                raise Exception("Could not create Machine object! The existing username and "
                                "password are incorrect")

        self.guest_os = os_name
        sections = VirtualServerUtils.get_details_from_config_file(os_name.lower())
        user_list = sections.split(",")
        incorrect_usernames = []
        for each_user in user_list:
            user_name = each_user.split(":")[0]
            password = VirtualServerUtils.decode_password(each_user.split(":")[1])
            try:
                vm_machine = machine.Machine(self.vm_hostname,
                                             username=user_name,
                                             password=password)
                if vm_machine:
                    self.machine = vm_machine
                    self.user_name = user_name
                    self.password = password
                    return
            except:
                incorrect_usernames.append(each_user.split(":")[0])

        self.log.exception("Could not create Machine object! The following user names are "
                           "incorrect: {0}".format(incorrect_usernames))

    def get_drive_list(self, drives=None):
        """
        Returns the drive list for the VM
        """
        try:
            super(AzureVM, self).get_drive_list()
            if self.guest_os == "Windows":
                if 'D' in self._drives:
                    del self._drives['D']
            if self.guest_os == "Linux":
                if "MountDir-3" in self._drives:
                    del self._drives["MountDir-3"]
                if '/mnt' in self._drives:
                    del self._drives['/mnt']

        except Exception as err:
            self.log.exception(
                "An Exception Occurred in Getting the Volume Info for the VM {0}".format(err))
            return False

    def power_on(self):
        """
        Power on the VM.

        Raises:
            Exception:
                When power on fails or unexpected error code is returned

        """
        try:

            self.api_version = "/start?api-version=2018-10-01"

            vm_poweronurl = self.azure_vmurl + self.api_version
            response = self.azure_session.post(vm_poweronurl, headers=self.default_headers, verify=False)
            # data = response.json()

            if response.status_code == 202:
                self.log.info('VMs found and powering on')

            elif response.status_code == 404:
                self.log.info('No VMs found')

            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("VM cannot be started")

        except Exception as exp:
            self.log.exception("Exception in PowerOn")
            raise Exception("Exception in PowerOn:" + str(exp))

    def power_off(self):
        """
        Power off the VM.

        Raises:
            Exception:
                When power off fails or unexpected error code is returned

        """

        try:

            self.api_version = "/powerOff?api-version=2018-10-01"

            vm_poweroffurl = self.azure_vmurl + self.api_version
            response = self.azure_session.post(vm_poweroffurl, headers=self.default_headers, verify=False)

            if response.status_code == 202:
                self.log.info('VMs found and turning off')

            elif response.status_code == 404:
                self.log.info('No VMs found')

            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("VM cannot be turned off")

        except Exception as exp:
            self.log.exception("Exception in PowerOff")
            raise Exception("Exception in Poweroff:" + str(exp))

    def clean_up_network(self):
        """
                 Clean up NIC

                 Raises:
                     Exception:
                         When cleanup failed or unexpected error code is returned

                 """

        for eachname in self.nic:
            azure_networkurl = "https://management.azure.com/subscriptions/%s/resourceGroups/%s/providers/Microsoft.Network/networkInterfaces/%s?api-version=2018-07-01" % (
                self.subscription_id, self.resource_group_name, eachname)
            response = self.azure_session.delete(azure_networkurl, headers=self.default_headers,
                                                 verify=False)

            if response.status_code == 202:
                self.log.info('Network interface %s found and deleting' % eachname)
                time.sleep(120)
                response = self.azure_session.delete(azure_networkurl, headers=self.default_headers,
                                                     verify=False)
                if response.status_code == 204:
                    self.log.info('Network interface deleted')
            elif response.status_code == 204:
                self.log.info('Network interface %s not found' % eachname)

            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("Network interface %s cannot be deleted" % eachname)

    def clean_up_disk(self, os_disk_details):
        """
                 Clean up managed/unmanaged disk

                 Raises:
                     Exception:
                         When cleanup failed or unexpected error code is returned

                 """

        if 'managedDisk' in os_disk_details:
            azure_osdiskurl = "https://management.azure.com%s?api-version=2017-03-30" % self.disk_info["OsDisk"]
            response = self.azure_session.delete(azure_osdiskurl, headers=self.default_headers, verify=False)

            if response.status_code == 202:
                self.log.info('OS disk found and deleting')

            elif response.status_code == 404:
                self.log.info('OS disk not found')

            elif response.status_code == 204:
                self.log.info('OS disk not found')

            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("OS disk cannot be deleted")

            for each in self.disk_info:
                azure_datadiskurl = "https://management.azure.com%s?api-version=2017-03-30" % self.disk_info[
                    each]
                response = self.azure_session.delete(azure_datadiskurl, headers=self.default_headers,
                                                     verify=False)

                if response.status_code == 202:
                    self.log.info('Data disk %s found and deleting' % each)

                elif response.status_code == 404:
                    self.log.info('Data disk %s not found' % each)

                elif response.status_code == 204:
                    self.log.info('Data disk %s not found' % each)

                else:
                    self.log.error('Azure response [{0}] Status code [{1}]'.format(
                        response.text, response.status_code))
                    raise Exception("Data disk %s cannot be deleted" % each)

        else:
            # deleting unmanaged disks or blobs
            storage_request = self.azure_session.post(
                "https://login.microsoftonline.com/%s/oauth2/token" % self.tenant_id,
                headers={"Content-Type": "application/x-www-form-urlencoded"},
                data=self.storage_data)
            if storage_request.status_code == 200:
                self.xmsdate = storage_request.headers["Date"]
                self.storage_token = storage_request.json()['access_token']

                response = self.azure_session.delete(self.disk_info["OsDisk"],
                                                     headers=self.storage_headers)

                if response.status_code == 202:
                    self.log.info('OS disk found and deleting')

                elif response.status_code == 404:
                    self.log.info('OS disk not found')

                elif response.status_code == 204:
                    self.log.info('OS disk not found')
                else:
                    self.log.error('Azure response [{0}] Status code [{1}]'.format(
                        response.text, response.status_code))
                    raise Exception("OS disk cannot be deleted")

                for each in self.disk_info:
                    response = self.azure_session.delete(self.disk_info[each],
                                                         headers=self.storage_headers)
                    if response.status_code == 202:
                        self.log.info('Data disk %s found and deleting' % each)
                    elif response.status_code == 404:
                        self.log.info('Data disk %s not found' % each)
                    elif response.status_code == 204:
                        self.log.info('Data disk %s not found' % each)
                    else:
                        self.log.error('Azure response [{0}] Status code [{1}]'.format(
                            response.text, response.status_code))
                        raise Exception("Data disk %s cannot be deleted" % each)

            else:
                self.log.info('Error in getting authorization token: %s' % json.loads(storage_request.text)[
                    "error_description"])

    def get_vm_location(self, resource_group, restored_vm_name):
        """
                Get location of out of place restored VM

                Raises:
                    Exception:
                        When getting response failed or unexpected error code is returned

                Returns:

                    vm location: string

                """
        try:

            self.api_version = "?api-version=2018-10-01"
            azure_restored_vmurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                                   "/%s/providers/Microsoft.Compute/virtualMachines" \
                                   "/%s" % (self.subscription_id, resource_group, restored_vm_name)
            vm_infourl = azure_restored_vmurl + self.api_version
            response = self.azure_session.get(vm_infourl, headers=self.default_headers, verify=False)
            data = response.json()
            return data['location']

        except Exception as exp:
            self.log.exception("Exception in getting restored VM location")
            raise Exception("Exception in getting restored VM location:" + str(exp))

    def clean_up(self):
        """
        Clean up the VM and ts reources.

        Raises:
            Exception:
                When cleanup failed or unexpected error code is returned
        """
        try:
            data = self.vm_info
            self.api_version = "?api-version=2018-10-01"
            vm_deleteurl = self.azure_vmurl + self.api_version
            response = self.azure_session.delete(vm_deleteurl, headers=self.default_headers, verify=False)
            if response.status_code == 202:
                self.log.info('VM found and deleting')
                time.sleep(360)
                response = self.azure_session.delete(vm_deleteurl, headers=self.default_headers, verify=False)
                if response.status_code == 204:
                    self.log.info("VM deleted")
            elif response.status_code == 204:
                self.log.info('VM not found')
            elif response.status_code == 404:
                self.log.info('Resource group not found')
            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("VM cannot be deleted")
            self.clean_up_network()
            os_disk_details = data["properties"]["storageProfile"]["osDisk"]
            self.clean_up_disk(os_disk_details)

        except Exception as exp:
            self.log.exception("Exception in Clenaup")
            raise Exception("Exception in Cleanup:" + str(exp))

    def _get_vm_info(self):
        """
        Get all VM information

        Raises:
            Exception:
                if failed to get information about VM
        """
        try:
            self.log.info("Fetching VM info")
            if self.instance_type == "azure resource manager":
                self.api_version = "?api-version=2018-10-01"

            vm_infourl = self.azure_vmurl + self.api_version
            response = self.azure_session.get(vm_infourl, headers=self.default_headers, verify=False)
            data = response.json()

            if response.status_code == 200:
                self.vm_info = data

            elif response.status_code == 404:
                self.vm_info = False
                self.log.info("No VMs found")
                # No VMs found

            else:
                self.log.error('Azure response [{0}] Status code [{1}]'.format(
                    response.text, response.status_code))
                raise Exception("VM  data cannot be collected")

        except Exception as err:
            self.log.exception("Exception in get_vm_info")
            raise Exception(err)

    def get_disks_by_repository(self, storage_acc):
        """
        Get Disks by Storage Account [for unmanaged fetching the storage account from blob uri]

        Args:
            storage_acc (basestring) : Account for which disks are to fetched

        Returns:
            A list of disks in the specified storage account

        Raises:
            Exception
                When issues while getting disks in the given storage account
        """
        try:
            self.log.info("Filtering disks by Storage Account")
            _disk_in_sa = []
            if not self.managed_disk:
                blob_details = self.get_bloburi()
                for disk, blob in blob_details.items():
                    if blob.partition('//')[2].split('.')[0] == storage_acc:
                        if disk in self.disk_dict.get('OsDisk'):
                            self.log.info("Disk Filtered is : {0}".format(disk))
                            _disk_in_sa.append('OsDisk')
                        else:
                            disk_name = [disk for disk, blob_uri in self.disk_dict.items() if blob_uri == blob]
                            self.log.info("Disk Filtered is : {0}".format(disk_name))
                            _disk_in_sa.append(disk_name[0])
            return _disk_in_sa
        except Exception as err:
            self.log.exception("Exception in get_disks_by_repository : {0}".format(err))
            raise Exception(err)

    def get_disk_path_from_pattern(self, disk_pattern):
        """
        Find the disk that matches the disk pattern form disk list

        Args:
                disk_pattern			(str):  pattern which needs to be matched

        Returns:
                matched_disks			   (str):  the disk that matches the pattern

        Raises:
            Exception:
                When issues while getting disk path from the pattern passed
        """
        try:
            self.log.info("Filtering disks by Pattern")
            _disk_name = os.path.basename(disk_pattern)
            matched_disks = []

            for each_disk in self.disk_dict:
                _vm_disk_name = os.path.basename(self.disk_dict[each_disk])
                if re.match(_disk_name, _vm_disk_name):
                    self.log.info("Disk Filtered is : {0}".format(_disk_name))
                    matched_disks.append(each_disk)
            return matched_disks

        except Exception as err:
            self.log.exception("Exception in get_disk_path_from_pattern : {0}".format(err))
            raise Exception(err)

    def get_datastore_uri_by_pattern(self, blob_uri_pattern):
        """
        Find the disk that matches the blob uri pattern form disk list

        Args:
                blob_uri_pattern			(str):  pattern which needs to be matched

        Returns:
                matched_disks			   (str):  the disk that matches the pattern

        Raises:
            Exception:
                When issues while getting disk path from the pattern passed
        """
        try:
            self.log.info("Filtering disks by Blob URI")
            matched_disks = []
            if not self.managed_disk:
                blob_details = self.get_bloburi()
                for disk, blob in blob_details.items():
                    if re.match(blob_uri_pattern, blob):
                        disk_name = [disk for disk, blob_uri in self.disk_dict.items() if blob_uri == blob]
                        self.log.info("Disk Filtered is : {0}".format(disk_name))
                        matched_disks.append(disk_name[0])
            return matched_disks

        except Exception as err:
            self.log.exception("Exception in get_datastore_uri_by_pattern : {0}".format(err))
            raise Exception(err)

    def get_disks_by_tag(self, tag_name, tag_value):
        """
        Filter disk by tag name and value

        Args:
            tag_name (basestring) : tag name to be searched
            tag_value (basestring) : tag value for tag name

        Returns:
                matched_disks (str):  the disk that matches the tag name and value

        Raises:
            Exception:
                When issues while getting disk from the tag and value passed
        """
        try:
            self.log.info("Filtering disks by Tag")
            matched_disks = []
            tag_dict = {}
            for disk_name, disk_val in self.disk_dict.items():
                if self.managed_disk:
                    tag_dict = self.get_tag_for_disk(disk_name)
                else:
                    tag_dict = self.get_metadata_for_disk(disk_val)
                if tag_dict:
                    for tag, value in tag_dict.items():
                        if tag == tag_name and value == tag_value:
                            self.log.info("Disk Filtered is : {0}".format(disk_name))
                            matched_disks.append(disk_name)
            return matched_disks
        except Exception as err:
            self.log.exception("Exception in get_disks_by_tag : {0}".format(err))
            raise Exception(err)

    def _get_disks_by_os_type(self):
        """
        Find the disk that matches the disk type

        Returns:
                matched_disks       (str):  the disk that matches the disk_type

        Raises:
            Exception:
                When issues while getting disk path from the disk type passed
        """
        try:
            self.log.info("Filtering disks by Os Type")
            matched_disks = []
            for disk in self.disk_dict.keys():
                if disk == 'OsDisk':
                    matched_disks.append(disk)
            return matched_disks

        except Exception as err:
            self.log.exception("Exception in get_disks_by_os_type : {0}".format(err))
            raise Exception(err)

    def get_metadata_for_disk(self, disk_blob):
        """
        Returns the metadata dictionary for the disk
        Args:
            disk_blob (basestring) : disk name for which metadat are to be fetched

        Returns:
             metadata dictionary with metadata for the disk

        Raises
            Exception:
                    When error in getting metadat for disk
        """
        try:
            header = self.get_storage_header('2019-07-07')
            uri = ""
            metadata_dict = {}
            uri = uri + disk_blob + '?comp=metadata'
            response = self.azure_session.get(uri, headers=header, verify=False)
            if response.status_code == 200:
                headers = response.headers
                for key, val in headers.items():
                    if 'x-ms-meta' in key:
                        metadata_dict[key.split('-')[-1]] = val
            return metadata_dict
        except Exception as err:
            self.log.exception("Exception in get metadata method")
            raise Exception(err)

    def get_tag_for_disk(self, disk_name):
        """
        Returns the tag dictionary for the disk

        Args:
            disk_name (basestring) : disk name for which tags are to be fetched

        Returns:
             tag dictionary with tags for the disk

        Raises
            Exception:
                    When error in getting tags for disk
        """
        try:
            tag_dict = {}
            if disk_name != 'OsDisk':
                if self.instance_type == "azure resource manager":
                    self.api_version = "?api-version=2019-07-01"
                disk_url = self.azure_diskurl + disk_name + self.api_version
                response = self.azure_session.get(disk_url, headers=self.default_headers,
                                                  verify=False)
                data = response.json()
                tag_dict = {}

                if response.status_code == 200:
                    tag_dict = data.get('tags')

                elif response.status_code == 404:
                    self.log.info("There was No VM in Name %s , please check the VM name")

            return tag_dict

        except Exception as err:
            self.log.exception("Exception in get tag details")
            raise Exception(err)

    def get_storage_header(self, xmsversion):
        if not self.storage_access_token:
            self.storage_access_token = self.hvobj.get_storage_access_token()
        else:
            self.storage_access_token = self.hvobj.check_for_storage_access_token_expiration()

        curr_time = time.strftime("%a, %d %b %Y %H:%M:%S ", time.gmtime())
        curr_time = curr_time + 'GMT'
        header = {
            'Authorization': 'Bearer ' + self.storage_access_token,
            'x-ms-version': xmsversion,
            'Date': curr_time}
        return header

    def check_snapshot_bymetadta_forjob(self, metadata):
        """ Checks if the snapshot exist on vm for the Job
        args:
           metadata (list) : lsit of metadata from db query

        return:
            (bool) : true if snapshot exist else false
        """

        self.log.info("checking snapshots from db")
        self.get_vm_guid()
        job_metadata_forvm = None
        if metadata[0] != '':
            for entry in metadata:
                if self.guid == entry[0]:
                    job_metadata_forvm = entry[1]
        if not job_metadata_forvm:
            return False
        snapshots = eval(job_metadata_forvm.split('|')[8])
        if self.managed_disk:
            for snapshot in snapshots['listManagedSnapshotIDs']:
                if not self.check_disksnapshot_byid(snapshot):
                    return False
        else:
            for snapshot in snapshots['listBlobSnapshotURIs']:
                if not self.check_blobsnapshot_byurl(snapshot):
                    return False
        return True

    def validate_sku(self, vm, dest_vm):
        """Checks if the sku type of the destination

           Args:

                vm  (object) : vm object for which sku vm has to be validated
                dest_vm     (object) :destination VM object
                restore_options  (object) : object of vm retore options

            Returns:
                If destination VM'S sku is not as expected
        """
        if vm.managed_disk:
            for disk, details in dest_vm.disk_sku_dict.items():
                if not details['storageAccountType']  == vm.disk_sku_dict[disk]['storageAccountType']:
                    self.log.info("Storage Account of restored  disk not as expected")
                    return False
        return True

    def check_disksnapshot_byid(self, snapshot_id):
        """ checks if the snapshot on disk exist
        args:
            snapshot_id  (str): id to for snapshot to be checked
        Return :
                (bool) : True if snapshot exist else false
        """
        try:
            azure_url = "https://management.azure.com"
            snapshot_url = azure_url + snapshot_id + "?api-version=2019-07-01"
            response = self.azure_session.get(snapshot_url, headers=self.default_headers, verify=False)
            if response.status_code == 200:
                return True
            if response.status_code in [204, 404]:
                return False
            else:
                raise Exception("Exception in checking snapshot existence response was not success")
        except Exception as err:
            raise Exception("Exception in checking snapshot existence " + str(err))

    def check_blobsnapshot_byurl(self, snapshot_url):
        """ checks if the snapshot on disk exist
                args:
                    snapshot_id  (str): id to for snapshot to be checked
                Return :
                        (bool) : True if snapshot exist else false
                """
        try:
            header = self.get_storage_header("2017-11-09")
            snapshot_url = snapshot_url.split("?")[0] + "?comp=metadata&" + snapshot_url.split("?")[1]
            response = self.azure_session.get(snapshot_url, headers=header, verify=False)
            if response.status_code == 200:
                return True
            if response.status_code in [204, 404]:
                return False
            else:
                raise Exception("Exception in checking snapshot existence response was not success")
        except Exception as err:
            raise Exception("Exception in checking snapshot existence " + str(err))

    def check_vmsizeof_destination_vm(self, vm_schedule, dest_vm, restore_options=None, vm=False):
        """Checks if the destination vm in has the the same vmsize as mentioned in schedule

               Args:
                vm_schedule  (object) :schedule or vm object for which destination vm has to be validated
                dest_vm     (object) :destination VM object
                restore_options  (object) : object of vm retore options
                vm          (bool) : whether we are checking for vm restore validation or live sync validation

               Returns:
                    If destination VM has vmsize as in schedule else false

               """
        if not vm:
            vmdeatils = vm_schedule.virtualServerRstOptions['diskLevelVMRestoreOption']['advancedRestoreOptions']
            for vm in vmdeatils:
                if vm['name'] == self.vm_name:
                    if 'vmSize' in vm:
                        if vm['vmSize'] and dest_vm.vm_size != vm['vmSize']:
                            self.log.info('vmSize validation failed')
                            return False
        elif restore_options.in_place or not restore_options.vm_size:
            if dest_vm.vm_size != vm_schedule.vm_size:
                self.log.info('VM Size validation failed. Source Size : {0}, Destination Size : {1}'.format(
                    str(vm_schedule.vm_size), str(dest_vm.vm_size)))
                return False
        else:
            if dest_vm.vm_size != self.vm_restore_options.vm_size:
                self.log.info('VM Size validation failed. Source Size : {0}, Destination Size : {1}'.format(
                    str(vm_schedule.vm_size), str(dest_vm.vm_size)))
                return False

        return True

    def check_disk_type(self, source_vm, dest_vm, restore_options=None):
        """
        Checks if the disk typeof restored vm is as expected
        Args:
               source_vm  (object) : vm object for which disk type vm has to be validated
               restore_options  (object) : object of vm retore options
               dest_vm     (object) :destination VM object

        Returns:
            True if expected, otherwise False
        """
        if restore_options.in_place or not restore_options.restoreAsManagedVM :
            if dest_vm.managed_disk != source_vm.managed_disk :
                self.log.info('Disk Type validation failed')
                return False
        elif restore_options.restoreAsManagedVM :
            if not dest_vm.managed_disk :
                self.log.info('Disk Type validation failed')
                return False
        return True

    def check_storageacc_destination_vm(self, vm_schedule, dest_vm, restore_options=None, vm=False):
        """Checks if the destination vm uses the same storage account as mentioned in schedule

               Args:
               vm_schedule  (object) :schedule or vm object for which destination vm has ti be validated
               dest_vm     (object) :destination VM object
               restore_options  (object) : object of vm retore options
               vm          (bool) : whether we are checking for vm restore validation or live sync validation
               returns     (bool): True If destination VM uses storage account as in schedule else false

               """
        if not vm:
            vmdeatils = vm_schedule.virtualServerRstOptions['diskLevelVMRestoreOption']['advancedRestoreOptions']
            for vm in vmdeatils:
                if vm['name'] == self.vm_name:
                    if 'Datastore' in vm:
                        dest_stracc = None
                        bloburi = dest_vm.get_bloburi()
                        for disk in bloburi:
                            if bloburi[disk]:
                                dest_stracc = bloburi[disk].split('.')[0].split('//')[1]
                                break
                        if vm['Datastore'] != '' and vm['Datastore'] != dest_stracc:
                            self.log.info(
                                "Validation Failure : Storage Account mismatch Source : {0} Destination : {1}")
                            return False
        elif not dest_vm.managed_disk:
            uri = dest_vm.vm_info['properties']['storageProfile']['osDisk']['vhd']['uri']
            source_uri = vm_schedule.vm_info['properties']['storageProfile']['osDisk']['vhd']['uri']
            dest_sa = uri.split('//')[1].partition('.')[0]
            source_sa = source_uri.split('//')[1].partition('.')[0]
            if restore_options.in_place or not restore_options.Storage_account:
                if dest_sa != source_sa:
                    self.log.info('Storage Account validation failed. Expected SA : {0}, Actual SA : {1}'.format(
                        vm_schedule.restore_storage_acc, dest_sa))
                    return False
            else:
                if dest_sa != restore_options.Storage_account:
                    self.log.info('Storage Account validation failed. Expected SA : {0}, Actual SA : {1}'.format(
                        restore_options.Storage_account, dest_sa))
                    return False
        return True

    def check_snap_exist_intimeinterval(self, list_of_snapshots, start_time, end_time):
        """Checks if list of snapshot has snapshot created between start_time and end_time

        Args:
             list_of_snapshots   (list): list containing string of snapshot creation time

             start_time          (str): start time for interval between which snapshot existence  verified

             end_time           (str):  end time for interval between which snapshot existance has to be verified

        Returns                (bool): True if snapshot exist between time interval esle False

        """
        if not list_of_snapshots:
            return False
        elif len(list_of_snapshots) == 0:
            return False
        else:
            start_time = datetime.datetime.strptime(start_time, '%Y-%m-%d %H:%M:%S')
            end_time = datetime.datetime.strptime(end_time, '%Y-%m-%d %H:%M:%S')
            for snap in list_of_snapshots:
                snap_time = datetime.datetime.strptime(snap, '%Y-%m-%d %H:%M:%S')
                if start_time < snap_time < end_time:
                    self.log.info("Snapshot exists with creation"
                                  " time : {0}".format(snap))
                    return True
        return False

    def check_gen_of_vm(self, dest_vm):
        """Checks if destination vm and source vm have same generation

        args:
            dest_vm    (obj): Destination vm object
        returns:
            Bool           : True if both source and destination vm have same generation
                            else false

        """
        try:
            if self.get_vm_generation() != dest_vm.get_vm_generation():
                return False
            return True
        except Exception as err:
            self.log.exception("Exception in Checking Vm Generation")
            raise Exception(err)

    def check_regionof_destinationvm(self, vm_schedule, dest_vm, restore_options=None, vm=False):
        """Checks if the destination vm is in region  as mentioned in schedule

               Args:
               vm_schedule  (object) :schedule or vm object for which destination vm has to be validated
               dest_vm     (object) :destination VM object
               restore_options  (object) : object of vm retore options
               vm           (bool) : whether we are checking for vm restore validation or live sync validation
               returns     (bool): If destination VM has region as in schedule else false

               """
        if not vm:
            vmdeatils = vm_schedule.virtualServerRstOptions['diskLevelVMRestoreOption']['advancedRestoreOptions']
            for vm in vmdeatils:
                if vm['name'] == self.vm_name:
                    if 'datacenter' in vm:
                        if vm['datacenter'] and vm['datacenter'] != dest_vm.vm_info['location']:
                            return False
        elif vm_schedule.vm_info['location'] != dest_vm.vm_info['location']:
            self.log.info('Region validation failed. Expected region : {0}, Actual region : {1}'.format(
                vm_schedule.vm_info['location'], dest_vm.vm_info['location']))
            return False
        elif restore_options.in_place or not restore_options.region:
            if vm_schedule.vm_info['location'] != dest_vm.vm_info['location']:
                self.log.info('Region validation failed. Expected region : {0}, Actual region : {1}'.format(
                    vm_schedule.vm_info['location'], dest_vm.vm_info['location']))
                return False
        else:
            if dest_vm.vm_info['location'] != restore_options.region:
                self.log.info('Region validation failed. Expected region : {0}, Actual region : {1}'.format(
                    vm_schedule.vm_info['location'], dest_vm.vm_info['location']))
                return False
        return True

    def check_publicipof_destinationvm(self, vm_schedule, dest_vm, restore_options=None, vm=False):
        """Checks if the destination vm  has the public ip or not if   mentioned in schedule

               Args:
               vm_schedule  (object) :schedule or vm object for which destination vm has ti be validated
               dest_vm     (object) :destination VM object
               restore_options  (object) : object of vm retore options
               returns     (bool): If destination VM has publicip created if mentioned  in schedule else false

               """
        if not vm:
            vmdeatils = vm_schedule.virtualServerRstOptions['diskLevelVMRestoreOption']['advancedRestoreOptions']
            for vm in vmdeatils:
                if vm['name'] == self.vm_name:
                    if "createPublicIP" in vm:
                        if vm["createPublicIP"] and not (dest_vm.get_publicipid()):
                            self.log.info('Public IP validation failed')
                            return False
        elif restore_options.in_place or not restore_options.createPublicIP:
            if (vm_schedule.get_publicipid()) and not (dest_vm.get_publicipid()) or \
                    (dest_vm.get_publicipid()) and not (vm_schedule.get_publicipid()):
                self.log.info('Public IP validation failed')
                return False
        else:
            if not dest_vm.get_publicipid():
                self.log.info('Public IP validation failed')
                return False

        return True

    def get_publicipid(self):
        """Gets ID of pubclic IP associated with VM

        Returns : (str)  ID of public IP if present else None

         Raises:
            Exception:
                When getting Public IP ID failed
        """
        try:
            azure_url = "https://management.azure.com/"
            networkinfo = self.vm_info['properties']['networkProfile']['networkInterfaces']
            networkinfourl = None
            for network in networkinfo:
                if network.get('properties', {}).get('primary'):
                    networkinfourl = network['id']
            publicipurl = None
            if networkinfourl:
                azure_diskurl = azure_url + networkinfourl + "?api-version=2018-10-01"
                response = self.azure_session.get(azure_diskurl, headers=self.default_headers, verify=False)
                if response.status_code == 200:
                    data = response.json()
                    networkconfigs = data['properties']['ipConfigurations']
                    dataip = None
                    for networkconfig in networkconfigs:
                        if networkconfig['properties']['primary']:
                            dataip = networkconfig
                            break
                    if dataip:
                        if 'publicIPAddress' in dataip['properties']:
                            publicipurl = dataip['properties']['publicIPAddress']['id']
            return publicipurl
        except Exception as exp:
            self.log.exception("Exception in getting  public ip ID")
            raise Exception("Exception in getting public ip ID:" + str(exp))

    def get_snapshots(self):
        """Gets snapshots associated with the bolbs and disks of VM

            Returns : (dict) dictionary with all snapshots on blobs and disks
        """

        blobs = self.get_snapshotsonblobs()
        disks = self.get_disk_snapshots()
        return {'blobs': blobs, 'disk': disks}

    def get_bloburi(self):
        """ Gets uri of blob associated with all disks of VM

            Returns: (dict)  dictionary with key as disk name and value as blob uri if present else None

             Raises:
            Exception:
                When getting blobruri failed

        """
        try:
            disks = self.disk_dict
            azure_url = "https://management.azure.com/"
            self.blobs = {}
            for disk in disks:
                if 'blob.core' in disks[disk]:
                    diskname = disks[disk].split('/')[-1]
                    self.blobs[diskname] = disks[disk]
                    continue
                azure_diskurl = azure_url + disks[disk] + "?api-version=2017-03-30"
                response = self.azure_session.get(azure_diskurl, headers=self.default_headers, verify=False)
                blob_uri = None
                if response.status_code == 200:
                    data = response.json()
                    if 'sourceUri' in data['properties']['creationData']:
                        blob_uri = data['properties']['creationData']['sourceUri']
                    self.blobs[data['name']] = blob_uri

            return self.blobs

        except Exception as exp:
            self.log.exception("Exception in getting bloburi")
            raise Exception("Exception in getting bloburi:" + str(exp))

    def get_disk_snapshots(self):
        """ Gets snapshots details associated with disks of a VM

        Returns : (dict) with key as disk name and value as list of dict with with containing name
                of snapshot, time of creation , jobid of job which
                created the snapshot


         Raises:
            Exception:
                When getting snapshot details failed
        """

        try:
            disks = self.disk_dict
            snapshots = {}
            for disk in disks:
                diskname = disks[disk].split('/')[-1]
                snapshots[diskname] = None
                if 'blob.core' in disks[disk]:
                    continue
                diskrsg = disks[disk].split('/')[4]
                azure_vmurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                              "/%s/providers/Microsoft.Compute/snapshots" % (self.subscription_id, diskrsg)
                api_version = '?api-version=2019-07-01'
                azure_snapurl = azure_vmurl + api_version
                response = self.azure_session.get(azure_snapurl, headers=self.default_headers, verify=False)
                if response.status_code == 200:
                    data = response.json()['value']
                    for snap in data:
                        if disks[disk] in snap['properties']['creationData']['sourceResourceId']:
                            if not snapshots[diskname]:
                                snapshots[diskname] = []
                            stamp = snap['properties']['timeCreated']
                            timeofsnap = stamp.split('T')[0] + " " + stamp.split('T')[1].split('.')[0]
                            snapname = snap['name']
                            jobid = self.get_snapshot_jobid(snap['id'])
                            snapshots[diskname].append([{'name': snapname, 'timeCreated': timeofsnap, 'JobId': jobid}])

            return snapshots
        except Exception as exp:
            self.log.exception("Exception in getting snapshots on disks")
            raise Exception("Exception in getting snapshots on disks:" + str(exp))

    def check_disk_snapshots_by_jobid(self, job_obj, all_snap=False):
        """ Gets snapshots details associated with disks and a job id
        Args:
            job_obj (basestring): job obj for which snapshot has to be checked.
            all_snap (bool): Whether return all snap details or beak if one snap exist

        Return:
            snapshot_exists (boolean) : Whether snapshot exists or not
            snapshots (dictionary): dict of snapshots for that particular job

         Raises:
            Exception:
                When getting snapshot details failed
        """

        try:

            self.log.info("Getting snapshot details based on job id for job {0}".format(job_obj.job_id))
            disks = self.disk_dict
            snapshot_exists = False
            snapshots = {}
            if self.managed_disk:
                self.log.info("Managed Disk...")
                for disk in disks:
                    # diskname = disks[disk].split('/')[-1]
                    snapshots[disk] = None
                    if 'blob.core' in disks[disk]:
                        continue
                    diskrsg = disks[disk].split('/')[4]
                    azure_vmurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                                  "/%s/providers/Microsoft.Compute/snapshots" % (self.subscription_id, diskrsg)
                    api_version = '?api-version=2019-07-01'
                    azure_snapurl = azure_vmurl + api_version
                    response = self.azure_session.get(azure_snapurl, headers=self.default_headers, verify=False)
                    if response.status_code == 200:
                        data = response.json()['value']
                        for snap in data:
                            if disks[disk] in snap['properties']['creationData']['sourceResourceId']:
                                if not snapshots[disk]:
                                    snapshots[disk] = False
                                snap_jobid = self.get_snapshot_jobid(snap['id'])
                                if int(job_obj.job_id) == snap_jobid:
                                    snapshots[disk] = True
                                    snapshot_exists = True
                                    self.log.info("Snapshot on disk {0} : {1}".format(disk, snap['id']))
                                    if not all_snap:
                                        break
            else:
                self.log.info("Unmanaged Disk...")
                destsnaps = self.get_snapshotsonblobs()
                startime = job_obj.start_time
                endtime = job_obj.end_time
                for disk in destsnaps:
                    for each_disk, value in self.disk_dict.items():
                        if re.match(disk, value.split('/')[-1]):
                            diskname = each_disk
                            snapshots[diskname] = None
                            break
                    snapshot_exists = self.check_snap_exist_intimeinterval(destsnaps[disk],
                                                                           startime, endtime)
                    if snapshot_exists:
                        self.log.info("snap exists for "
                                      "job {0} on Blob {1} ".format(job_obj.job_id, disk))

                    if not snapshot_exists:
                        self.log.info("Snapshot for Job {0} does not exist on blob {1} ".format(job_obj.job_id, disk))
                    if not all_snap and snapshot_exists:
                        return snapshot_exists, snapshots
                    elif snapshot_exists:
                        snapshots[diskname] = True
            if all_snap:
                snapshot_exists = True
                for disk in snapshots:
                    if not snapshots[disk]:
                        snapshot_exists = False
                        break
            return snapshot_exists, snapshots
        except Exception as exp:
            self.log.exception("Exception in getting snapshots on disks")
            raise Exception("Exception in getting snapshots on disks:" + str(exp))

    def get_snapshotsonblobs(self):
        """Gets time of creation of all snapshots on the blobs associated with VM

            Returns:  (dict)  dictionary with key as disk name and value as list containing creation time
                              of all snapshots associated with disk


        """
        try:
            bloburi = self.get_bloburi()
            blobsnapshots = {}
            for disk in bloburi:
                blobsnapshots[disk] = None
                if bloburi[disk]:
                    blobname = bloburi[disk].split('/')[-1]
                    blob_list = bloburi[disk].split('/')
                    uri = ""
                    for i in blob_list[0:3]:
                        uri = uri + i + '/'
                    uri = uri + blob_list[3] + '?restype=container&comp=list&include=snapshots&marker='
                    blobsnapshots[disk] = []
                    marker = ""
                    search_finished = False
                    while not search_finished:
                        header = self.get_storage_header('2017-11-09')
                        url = uri + marker
                        response = self.azure_session.get(url, headers=header, verify=False)
                        if response.status_code == 200:
                            res_dict = xmltodict.parse(response.text)
                            marker = res_dict['EnumerationResults']['NextMarker']
                            if res_dict['EnumerationResults']['Blobs']:
                                blobs = res_dict['EnumerationResults']['Blobs']['Blob']
                                if type(blobs) != list:
                                    blobs = [blobs]
                            for eachblob in blobs:
                                if eachblob['Name'] == blobname and 'Snapshot' in eachblob.keys():
                                    stamp = eachblob['Snapshot']
                                    timeofsnap = stamp.split('T')[0] + " " + stamp.split('T')[1].split('.')[0]
                                    if timeofsnap not in blobsnapshots[disk]:
                                        blobsnapshots[disk].append(timeofsnap)
                            if not marker:
                                search_finished = True
                                break
                        else:
                            self.log.error("Error in getting snapshots response "
                                           "not success : status code {0}".format(response.status_code))
                else:
                    blobsnapshots[disk] = None

            return blobsnapshots

        except Exception as exp:
            raise Exception("Execption in getting blob snapshots %s" % exp)

    def get_snapshot_jobid(self, snap_id):
        """Gets the jobid of job that created disk snapshot if snapshot

          Args:
             snap_id    (str): id of the snapshot

         Returns   (int ): JobID of job that created snapshot


        """
        try:
            azure_vmurl = "https://management.azure.com"
            api_version = '?api-version=2019-07-01'
            azure_snapuri = azure_vmurl + snap_id + api_version
            response = self.azure_session.get(azure_snapuri, headers=self.default_headers, verify=False)
            jobid = None
            if response.status_code == 200:
                snap_info = response.json()
                if 'tags' in snap_info and 'CreatedBy' in snap_info['tags'] and 'Description' in snap_info['tags']:
                    if snap_info['tags']['CreatedBy'] == 'Commvault':
                        if len(snap_info['tags']['Description'].split('_')) > 8:
                            jobid = int(snap_info['tags']['Description'].split('_')[3])
                        elif len(snap_info['tags']['Description'].split(' ')) > 7:
                            jobid = int(snap_info['tags']['Description'].split(' ')[3][1:-1])
                return jobid
        except Exception as err:
            self.log.info("Execption in getting jobID of snapshot")

    def get_vm_generation(self):
        """Gets the generation of VM
           Retruns   (str):  generation of vm

        """
        try:
            api_version = "?api-version=2019-12-01"
            azure_vmurl = "https://management.azure.com/subscriptions/%s/resourceGroups" \
                          "/%s/providers/Microsoft.Compute/virtualMachines" \
                          "/%s/instanceView" % (self.subscription_id, self.resource_group_name, self.vm_name)
            vm_instance_url = azure_vmurl + api_version
            response = self.azure_session.get(vm_instance_url, headers=self.default_headers, verify=False)
            if response.status_code == 200:
                instance_data = response.json()
                if 'hyperVGeneration' in instance_data.keys():
                    return instance_data['hyperVGeneration']
            else:
                raise Exception("Error in getting Genenration of VM response was not successful ")
        except Exception as err:
            raise Exception("Execption in getting vm generation %s" % err)

    def get_vm_guid(self):
        """
        Get GUID of particular VM

        Raises:
            Exception:
                if failed to fetch GUID of VM
        """
        try:
            data = self.vm_info
            self.guid = data["properties"]["vmId"]
            setattr(self, "guid", self.guid)

        except Exception as err:
            self.log.exception("Exception in get_vm_guid")
            raise Exception(err)

    def get_nic_info(self):
        """
        Get all network attached to that VM

        Raises:
            Exception:
                    if failed to get network information of VM
        """
        try:

            data = self.vm_info
            nic_names_id = data["properties"]["networkProfile"]["networkInterfaces"]
            for eachname in nic_names_id:
                nic_id = eachname["id"]
                nic_name = eachname["id"].split("/")[-1]
                self.nic.append(nic_name)
                self.nic_id.append(nic_id)

            self.nic_count = len(nic_names_id)
            setattr(self, "nic_count", self.nic_count)

        except Exception as err:
            self.log.exception("Exception in get_nic_info")
            raise Exception(err)

    def get_nic_details(self):
        """Gets  the type of IP allocation of all the network interfaces of VM

        """
        try:
            azure_vmurl = "https://management.azure.com"
            api_version = '?api-version=2018-07-01'
            all_nics = self.nic_id
            self.nic_details = []
            for nic_id in all_nics:
                azure_nicuri = azure_vmurl + nic_id + api_version
                response = self.azure_session.get(azure_nicuri, headers=self.default_headers, verify=False)
                if response.status_code == 200:
                    nic_detail = {}
                    data = response.json()
                    nic_detail['name'] = data['name']
                    nic_detail['allocation'] = data['properties']['ipConfigurations'][0]['properties'][
                        'privateIPAllocationMethod']
                    self.nic_details.append(nic_detail)
        except Exception:
            self.log.info('Failed to get nic details')

    def get_VM_size(self):
        """
        Get instance size for the VM

        Raises:
            Exception:
                    if failed to get instance size of VM
        """
        try:

            data = self.vm_info
            self.vm_size = data["properties"]["hardwareProfile"]["vmSize"]

        except Exception as err:
            self.log.exception("Exception in get_vm_size")
            raise Exception(err)

    def get_cores(self):
        """
        Get number of CPU, memory of VM

        Raises:
            Exception:
                    if failed to get CPU, memory information of VM
        """
        try:
            if self.vm_size:
                if self.instance_type == "azure resource manager":
                    self.api_version = "?api-version=2018-10-01"
                vm_sizeurl = self.azure_vmurl + "/vmSizes" + self.api_version
                response = self.azure_session.get(vm_sizeurl, headers=self.default_headers, verify=False)
                data = response.json()
                if response.status_code == 200:
                    for eachsize in data["value"]:
                        if eachsize["name"] == self.vm_size:
                            self.no_of_cpu = eachsize["numberOfCores"]
                            _memory = ((eachsize["memoryInMB"]) / 1024)
                            self.memory = _memory

                elif response.status_code == 404:
                    self.log.info("There was No VM in Name %s ,check the VM name" % self.vm_name)
                    self.size_info = False
                    # No VMs found

                else:
                    raise Exception("VM  data cannot be collected")

        except Exception as err:
            self.log.exception("Exception in get_vm_size")
            raise Exception(err)

    def get_Disk_info(self):
        """
        Get disk properties of both OS and data disks of VM

        Raises:
            Exception:
                    if failed to disk information of VM
        """
        try:

            data = self.vm_info
            os_disk_details = data["properties"]["storageProfile"]["osDisk"]
            if 'managedDisk' in os_disk_details:
                if "storageAccountType" not in os_disk_details["managedDisk"]:
                    self.power_on()
                    time.sleep(180)
                    self._get_vm_info()
                    data = self.vm_info
                    os_disk_details = data["properties"]["storageProfile"]["osDisk"]
                self.disk_info["OsDisk"] = os_disk_details["managedDisk"]["id"]
                if "storageAccountType" in os_disk_details["managedDisk"]:
                    self.storageaccount_type = os_disk_details["managedDisk"]["storageAccountType"]
                    self.disk_sku_dict['osDisk'] = {'storageAccountType': os_disk_details["managedDisk"]["storageAccountType"]}
                data_disk_details = data["properties"]["storageProfile"]["dataDisks"]
                for each in data_disk_details:
                    self.disk_info[each["name"]] = each["managedDisk"]["id"]
                    self.disk_sku_dict[each["lun"]] = {'storageAccountType': each['managedDisk']['storageAccountType']}
            else:
                self.managed_disk = False
                self.disk_info["OsDisk"] = os_disk_details["vhd"]["uri"]
                data_disk_details = data["properties"]["storageProfile"]["dataDisks"]
                for each in data_disk_details:
                    self.disk_info[each["name"]] = each["vhd"]["uri"]

            self.disk_dict = self.disk_info
            self.disk_count = len(self.disk_info.keys())

        except Exception as err:
            self.log.exception("Exception in get_disk_info")
            raise Exception(err)

    def get_data_disk_info(self):
        """
        Get data disks details of VM

        Raises:
            Exception:
                    if failed to disk information of VM
        """
        try:

            if self.instance_type == "azure resource manager":
                self.api_version = "?api-version=2017-12-01"
            vm_infourl = self.azure_vmurl + self.api_version
            response = self.azure_session.get(vm_infourl, headers=self.default_headers,
                                              verify=False)
            data = response.json()
            data_disk_details = []

            if response.status_code == 200:
                data_disk_details = data["properties"]["storageProfile"]["dataDisks"]

            elif response.status_code == 404:
                self.log.info("There was No VM in Name %s , please check the VM name")

            return data_disk_details

        except Exception as err:
            self.log.exception("Exception in get_data_disk_info")
            raise Exception(err)

    def get_status_of_vm(self):
        """
        Get the status of VM like started.stopped

        Raises:
            Exception:
                    if failed to get status of VM
        """
        try:

            body = {}
            vmurl = self.azure_vmurl + "/InstanceView" + self.api_version
            data = self.azure_session.get(vmurl, headers=self.default_headers, verify=False)
            if data.status_code == 200:
                status_data = data.json()
                if "vmAgent" in status_data.keys():
                    self.vm_state = status_data["vmAgent"]["statuses"][0]["displayStatus"]

            elif data.status_code == 404:
                self.log.info("VM Not found")


            else:
                raise Exception("Cannot get the status of VM")

        except Exception as err:
            self.log.exception("Exception in getStatusofVM")
            raise Exception(err)

    def get_OS_type(self):
        """
        Update the OS Type of VM

        Raises:
            Exception:
                    if failed to find OS type of VM
        """

        try:
            data = self.vm_info
            guest_os = data['properties']["storageProfile"]["osDisk"]["osType"]
            setattr(self, "guest_os", guest_os)
            self.log.info("OS type is : %s" % self.guest_os)

        except Exception as err:
            self.log.exception("Exception in GetOSType")
            raise Exception(err)

    def get_subnet_ID(self):
        """
        Update the subnet_ID for VM

        Raises:
            Exception:
                    if failed to find subnet information of VM
        """

        try:

            if self.instance_type == "azure resource manager":
                self.api_version = "?api-version=2018-01-01"
            azure_list_nwurl = self.azure_baseURL + "/subscriptions/" + self.subscription_id \
                               + "/providers/Microsoft.Network/networkInterfaces" + self.api_version
            data = self.azure_session.get(azure_list_nwurl, headers=self.default_headers, verify=False)
            if data.status_code == 200:
                _all_nw_data = data.json()
                for each_nw in _all_nw_data["value"]:
                    if each_nw["name"] == self.network_name:
                        ip_config_info = each_nw["properties"]["ipConfigurations"]
                        for each_ip_info in ip_config_info:
                            self.subnetId = each_ip_info["properties"]["subnet"]["id"]
                            break
            else:
                raise Exception("Failed to get network details for the VM")


        except Exception as err:
            self.log.exception("Exception in GetSubnetID")
            raise Exception(err)

    def get_IP_address(self):
        """
        Get the Ip address of the VM

        Raises:
            Exception:
                    if failed to get IP address of VM
        """
        try:
            data = self.vm_info
            self.log.info("VM data : %s" % data)

            nw_interfaces = data['properties']["networkProfile"]["networkInterfaces"]
            for each_network in nw_interfaces:
                nw_interface_value = each_network["id"]

            if self.instance_type == 'azure resource manager':
                self.api_version = "?api-version=2018-01-01"

            nw_interface_url = self.azure_baseURL + nw_interface_value + self.api_version
            response = self.azure_session.get(nw_interface_url, headers=self.default_headers, verify=False)
            nic_interface_info = response.json()
            # Setting network security group
            nsg = ''
            if "networkSecurityGroup" in nic_interface_info["properties"]:
                nsg = nic_interface_info["properties"]["networkSecurityGroup"]["id"]
            setattr(self, "nsg", nsg)

            ip_config_info = nic_interface_info["properties"]["ipConfigurations"]
            for each_ip_info in ip_config_info:
                vm_vnet_props = each_ip_info["properties"]
                break

            if self.instance_type == 'azure resource manager':
                self.api_version = "?api-version=2018-01-01"
            else:
                self.api_version = "?api-version=2017-10-01"

            vm_ip = vm_vnet_props["privateIPAddress"]
            setattr(self, "ip", vm_ip)

            self.subnet_id = vm_vnet_props["subnet"]["id"]
            nw_name = nic_interface_info["id"]
            self.network_name = nw_name.split("/")[-1]

            setattr(self, "host_name", vm_ip)

        except Exception as err:
            self.log.exception("Exception in get_vm_ip_address")
            raise Exception(err)

    ##Chirag's code
    def get_new_network_interface(self, vm, subnet_resource, subnet_network, subnet_name):
        """

        Creates a new network interface on Azure in the current resource group
        Args :
            vm: Name for the NIC
            subnet_resource: Name of the subnet resource
            subnet_network: Name of the subnet network
            subnet_name: Name of the subnet
        Raises:
            Exception:
                If failed to create new network interface
        """
        nic_url = self.azure_baseURL
        request_url = "/subscriptions/{}/resourceGroups/{}/providers/Microsoft.Network/networkInterfaces" \
                      "/{}{}".format(self.subscription_id, self.resource_group_name, vm,
                                     self.api_version)
        nic_url += request_url

        subnet_id = "/subscriptions/{}/resourceGroups/{}/providers/Microsoft.Network/virtualNetworks" \
                    "/{}/subnets/{}".format(self.subscription_id, subnet_resource, subnet_network,
                                            subnet_name)

        data = {
            "properties": {
                "enableAcceleratedNetworking": "false",
                "ipConfigurations": [
                    {
                        "name": "ipconfig1",
                        "properties": {
                            "subnet": {
                                "id": "%s" % subnet_id
                            }
                        }
                    }
                ]
            },
            "location": "East US2"
        }

        data = json.dumps(data)

        try:
            response = self.azure_session.put(nic_url, data=data, headers=self.default_headers)

            status_code = response.status_code
            if status_code != 201 and status_code != 200:
                raise Exception("Error: %s" % status_code)

            return response.url.split("azure.com")[1]

        except Exception as err:
            self.log.exception("Failed to create a network interface")
            raise Exception(err)

    # Chirag's code
    def create_vm(self, subnet_resource, subnet_network, subnet_name):
        """
        Creates a new VM on Azure int the current resource group
        subnet_resource: Name of the subnet resource
        subnet_network: Name of the subnet network
        subnet_name: Name of the subnet
        Raises:
            Exception:
                If failed to create VM
        """
        new_vm_name = self.vm_name + str(time.time())[-4:]
        api_version = "?api-version=2019-12-01"
        network_interface = self.get_new_network_interface(new_vm_name, subnet_resource,
                                                           subnet_network, subnet_name).split('?')[0]

        create_url = "/subscriptions/{}/resourceGroups/{}/providers/Microsoft.Compute" \
                     "/virtualMachines/{}{}".format(self.subscription_id, self.resource_group_name,
                                                    new_vm_name, api_version)

        request_url = self.azure_baseURL + create_url

        data = {
            "location": "East US2",
            "name": "%s" % new_vm_name,
            "properties": {
                "hardwareProfile": {
                    "vmSize": "Standard_B1ls"
                },
                "storageProfile": {
                    "imageReference": {
                        "sku": "18.04-LTS",
                        "publisher": "Canonical",
                        "version": "latest",
                        "offer": "UbuntuServer"
                    },
                    "osDisk": {
                        "caching": "ReadWrite",
                        "managedDisk": {
                            "storageAccountType": "Standard_LRS"
                        },
                        "name": "%s-od" % new_vm_name,
                        "createOption": "FromImage"
                    }
                },
                "osProfile": {
                    "adminUsername": "cvuser",
                    "computerName": "test-crg",
                    "adminPassword": "commvault!12"
                },
                "networkProfile": {
                    "networkInterfaces": [
                        {
                            "id": "%s" % network_interface,
                            "properties": {
                                "primary": "true"
                            }
                        }
                    ]
                }
            }
        }

        data = json.dumps(data)

        try:
            response = self.azure_session.put(request_url, data=data, headers=self.default_headers)

            status_code = response.status_code
            if status_code != 201 and status_code != 200:
                raise Exception("Error: %s" % status_code)

            self.log.info("Successfully created new VM: %s" % new_vm_name)

        except Exception as err:
            self.log.exception("Failed to create a new VM")
            raise Exception(err)

    def update_vm_info(self, prop='Basic', os_info=False, force_update=False):
        """
         fetches all the properties of the VM

         Args:
                 should have code for two possibilties

                 Basic - Basic properties of VM like cores,GUID,disk

                 All   - All the possible properties of the VM

                 Set the property VMGuestOS for creating OS Object

                 fetches some particular specified property

                 force_update - to refresh all the properties always
                    True : ALways collect  properties
                    False: refresh only if properties are not initialized

         Raises:
            Exception:
                 if failed to get all the properties of the VM
         """
        try:
            if self.vm_info:
                self.log.info("VM Info is : {0}".format(self.vm_info))
                self.power_on()

                if not self._basic_props_initialized or force_update:
                    self.get_vm_guid()
                    self.get_VM_size()
                    self.get_cores()
                    self.get_Disk_info()
                    self.get_status_of_vm()
                    self.get_IP_address()
                    self._basic_props_initialized = True

                if prop == 'All':
                    self.get_nic_info()
                    self.get_subnet_ID()
                    self.get_OS_type()
                    self.vm_guest_os = self.guest_os
                    self.get_drive_list()
                    self.get_nic_details()

                elif hasattr(self, prop):
                    return getattr(self, prop, None)

            else:
                self.log.info("VM Info is Empty. Fetching it again")
                self._get_vm_info()
                self._basic_props_initialized = False

        except Exception as err:
            self.log.exception("Failed to Get the VM Properties of the VM")
            raise Exception(err)
