# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that does all operations on VM

classes defined:
    Base class:
        HypervisorVM- Act as base class for all VM operations

    Inherited class:
        HyperVVM - Does all operations on Hyper-V VM

    Methods:

        get_drive_list()    - get all the drive list associated with VM

        power_off()            - power off the VM

        power_on()            -power on the VM

        delete_vm()            - delete the VM

        update_vm_info()    - updates the VM info
    """

import re
from abc import ABCMeta, abstractmethod
from AutomationUtils import logger
from AutomationUtils import machine
from AutomationUtils.pyping import ping
from . import VirtualServerUtils, VirtualServerConstants
from importlib import import_module
from inspect import getmembers, isclass, isabstract
import os
import time
from VirtualServer.VSAUtils.VirtualServerUtils import validate_ip

class HypervisorVM(object):
    """
    Main class for performing operations on Hyper-V VM
    """
    __metaclass__ = ABCMeta

    def __new__(cls, hvobj, vm_name):
        """
        Initialize VM object based on the Hypervisor of the VM
        """
        instance_type = hvobj.instance_type.lower()
        vm_helper = VirtualServerConstants.instance_vmhelper(instance_type)
        hh_module = import_module("VirtualServer.VSAUtils.VMHelpers.{}".format(vm_helper))
        classes = getmembers(hh_module, lambda m: isclass(m) and not isabstract(m))
        for name, _class in classes:
            if issubclass(_class, HypervisorVM) and _class.__module__.rsplit(".", 1)[-1] == vm_helper:
                return object.__new__(_class)

    def __init__(self, hvobj, vm_name):
        """
        Initialize the VM initialization properties
        """
        self.vm_name = vm_name
        self.hvobj = hvobj
        self.commcell = self.hvobj.commcell
        self.server_name = hvobj.server_host_name
        self.host_user_name = hvobj.user_name
        self.host_password = hvobj.password
        self.log = logger.get_log()
        self.instance_type = hvobj.instance_type
        self.utils_path = VirtualServerUtils.UTILS_PATH
        self.host_machine = self.hvobj.machine
        self.guest_os = None
        self.ip = None
        self._DriveList = None
        self._user_name = None
        self._password = None
        self._drives = None
        self.machine = None
        self._preserve_level = 1
        self.DiskType = None
        self.power_state = ''
        self.DiskList = []
        self.validate_name_tag = None  # attribute used for AWS validation
        self.backup_job = None  # attribute needed  for openstackValidation validation
        self.vm_exist = True  # used for conversion purpose

    class VmValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.restore_job = self.vm_restore_options.restore_job
            self.log = logger.get_log()

        def __eq__(self, other):
            """compares the source vm and restored vm"""
            return True

    class VmConversionValidation(object):
        def __init__(self, vmobj, vm_restore_options):
            self.vm = vmobj
            self.vm_restore_options = vm_restore_options
            self.log = logger.get_log()

        def __eq__(self, other):
            return True

    @property
    def drive_list(self):
        """
        Returns the drive list for the VM. This is read only property
        """
        if self._drives is None:
            self.get_drive_list()
        return self._drives

    @property
    def user_name(self):
        """gets the user name of the Vm prefixed with the vm name to avoid conflict.
        It is a read only attribute"""

        return self._user_name

    @user_name.setter
    def user_name(self, value):
        """sets the username of the vm"""

        self._user_name = value

    @property
    def password(self):
        """gets the user name of the Vm . It is a read only attribute"""

        return self._password

    @password.setter
    def password(self, value):
        """sets the password of the vm"""

        self._password = value

    @property
    def vm_hostname(self):
        """gets the vm hostname as IP (if available or vm name). It is a read only attribute"""

        if self.ip:
            return self.ip

        return self.vm_name

    @property
    def vm_guest_os(self):
        """gets the VM Guest OS . it is read only attribute"""
        return self.machine

    @vm_guest_os.setter
    def vm_guest_os(self, value):
        self._set_credentials(value)

    @property
    def preserve_level(self):
        """gets the default preserve level of Guest OS. it is read only attribute"""
        if self.guest_os.lower() == "windows":
            self._preserve_level = 0
        else:
            self._preserve_level = 1

        return self._preserve_level

    # just to reduce redirection

    def get_os_name(self, vm_name=None):
        """
        Get the OS Name from the machine name by ping as Hypervisor API gives wrong information
        Returns:
            os_name (str) - os of the VM
        """
        if not vm_name:
            vm_name = self.vm_hostname

        self.log.info("Pinging the vm : {0}".format(vm_name))
        response = ping(vm_name)

        # Extract TTL value form the response.output string.
        try:
            ttl = int(re.match(r"(.*)ttl=(\d*) .*",
                               response.output[2]).group(2))
        except AttributeError:
            raise Exception(
                'Failed to connect to the machine.\nError: "{0}"'.format(
                    response.output)
            )

        if ttl <= 64:
            return "Linux"
        elif ttl <= 128:
            return "Windows"
        elif ttl <= 255:
            return "Linux"
        else:
            raise Exception(
                'Got unexpected TTL value.\nTTL value: "{0}"'.format(ttl))

    def _set_credentials(self, os_name):
        """
        set the credentials for VM by reading the config INI file
        """

        os_name = self.get_os_name(self.vm_hostname)
        if self.user_name and self.password:
            try:
                vm_machine = machine.Machine(self.vm_hostname,
                                             username=self.user_name,
                                             password=self.password)
                if vm_machine:
                    self.machine = vm_machine
                    return
            except:
                raise Exception("Could not create Machine object! The existing username and "
                                "password are incorrect")

        self.guest_os = os_name
        sections = VirtualServerUtils.get_details_from_config_file(os_name.lower())
        user_list = sections.split(",")
        attempt = 0

        while attempt < 5:
            incorrect_usernames = []

            for each_user in user_list:
                user_name = each_user.split(":")[0]
                password = VirtualServerUtils.decode_password(each_user.split(":")[1])
                try:
                    vm_machine = machine.Machine(self.vm_hostname,
                                                         username=user_name,
                                                         password=password)
                    if vm_machine:
                        self.machine = vm_machine
                        self.user_name = user_name
                        self.password = password
                        return
                except:
                    incorrect_usernames.append(each_user.split(":")[0])
            attempt = attempt + 1
        self.log.exception("Could not create Machine object! The following user names are "
                           "incorrect: {0}".format(incorrect_usernames))

    def get_drive_list(self, drives=None):
        """
        Returns the drive list for the VM
        """
        try:
            _temp_drive_letter = {}
            storage_details = self.machine.get_storage_details()

            if self.guest_os.lower() == "windows":
                _drive_regex = "^[a-zA-Z]$"
                for _drive, _size in storage_details.items():
                    if re.match(_drive_regex, _drive):
                        if drives is None and _size['available'] < 900:
                            continue
                        _drive = _drive + ":"
                        _temp_drive_letter[_drive.split(":")[0]] = _drive

            else:
                temp_dict = {}
                fstab = self.machine.execute_command('cat /etc/fstab')
                if fstab.exception:
                    self.log.error("Exception:{}".format(fstab.exception))
                    raise Exception("Error in getting Mounted drives of the vm: {}".format(self.vm_name))
                self.log.info("complete fstab for vm {0}: {1}".format(self.vm_name, fstab.output))
                for mount in fstab.formatted_output:
                    if (re.match('/', mount[0])
                        or re.match('UUID=', mount[0], re.I)
                        or re.match('LABEL=', mount[0], re.I)) and not (
                            re.match('/boot', mount[1], re.I)
                            or re.match('swap', mount[1], re.I)):
                        temp_dict[mount[0]] = mount[1]
                self.log.info("Mount points applicable: {}".format(temp_dict))
                _temp_storage = {}
                for _detail in storage_details.values():
                    if isinstance(_detail, dict):
                        _temp_storage[_detail['mountpoint']] = _detail['available']
                self.log.info("Storage of VM {0}: {1}".format(self.vm_name, _temp_storage))
                _index = 0
                for key, val in temp_dict.items():
                    if _temp_storage[val] > 900:
                        if re.match('/dev/sd', key, re.I) or re.match('/dev/xvd', key, re.I):
                            _command = 'blkid ' + key
                            blkid = self.machine.execute_command(_command)
                            if blkid.exception:
                                raise Exception("Error in getting UUID of the vm: {}".format(self.vm_name))
                            _temp_drive_letter[
                                '/cvlostandfound/' + re.findall(r'\w+(?:-\w+)+', blkid.formatted_output)[0]] = val
                        elif re.match('/dev/mapper', key, re.I):
                            _temp_drive_letter[key.split('dev/mapper/')[1]] = val
                        elif re.match('/dev/VolGroup', key, re.I):
                            _temp_drive_letter['-'.join(key.split("/")[2:])] = val
                        elif re.match('UUID=', key, re.I) and val == "/":
                            _temp_drive_letter["MountDir-2"] = val
                            _index = 3
                        elif re.match('LABEL=', key, re.I) and val == "/":
                            _temp_drive_letter["MountDir-1"] = val
                            _index = 2
                        elif re.match('LABEL=', key, re.I):
                            _temp_drive_letter["MountDir-" + str(_index)] = val
                            _index += 1
                        else:
                            _temp_drive_letter[val] = val
                self.log.info("Disk and mount point for vm {0}: {1}".format(self.vm_name, _temp_drive_letter))
                del _temp_storage, temp_dict
            self._drives = _temp_drive_letter
            if not self._drives:
                raise Exception("Failed to Get Volume Details for the VM")

        except Exception as err:
            self.log.exception(
                "An Exception Occurred in Getting the Volume Info for the VM {0}".format(err))
            return False

    @abstractmethod
    def power_off(self):
        """
        power off the VM.

        return:
                True - when power off is successfull

        Exception:
                When power off failed

        """
        self.log.info("Power off the VM")

    @abstractmethod
    def power_on(self):
        """
        power on the VM.

        return:
                True - when power on is successful

        Exception:
                When power on failed

        """
        self.log.info("Power on the VM")

    @abstractmethod
    def delete_vm(self):
        """
        power on the VM.

        return:
                True - when delete is successful

                False - when delete failed
        """
        self.log.info("Delete the VM")

    def clean_up(self):
        """
        Clean up the VM resources post restore

        Raises:
             Exception:
                If unable to clean up VM and its resources

        """
        self.log.info("Powering off VMs/Instances after restore")
        self.power_off()

    @abstractmethod
    def update_vm_info(self):
        """
        fetches all the properties of the VM

        Args:
                should have code for two possibilties

                Basic - Basic properties of VM like HostName,GUID,Nic
                        especially the properties with which VM can be added as dynamic content

                All   - All the possible properties of the VM

                Set the property VMGuestOS for creating OS Object

                all the property need to be set as class variable

        exception:
                if failed to get all the properties of the VM
        """
        self.log.info("Update the VMinfo of the VM")

    def wait_for_vm_to_boot(self):
        """
        Waits for a VM to start booting by pinging it to see if an IP has been successfully assigned.

        Raise Exception:
                If IP assigned within 10 minutes
        """
        # Wait for IP to be generated
        wait = 10

        try:
            while wait:
                self.log.info(
                    'Waiting for 60 seconds for the IP to be generated')
                time.sleep(60)

                try:
                    self.update_vm_info(force_update=True)
                except Exception as exp:
                    self.log.info(exp)

                if self.ip and validate_ip(self.ip):
                    break
                wait -= 1

            else:
                raise Exception(
                    f'Valid IP for VM: {self.vm_name} not generated within 10 minutes')

            self.log.info(f'IP is generated for VM: {self.vm_name}')

        except Exception as err:
            self.log.exception("An error occurred in fetching VM IP")
            raise Exception(err)

    def compare_disks(self, num_threads, multiplier):
        """
            Copies over an executabel to the target machine which will
            do a disk level comparison, and generate logs on the target machine

            Raise Exception:
                    if executable not executed successfully
                    if Disk Comparison fails
        """
        try:
            disk_compare_py = os.path.join(
                self.utils_path, "DiskCompare.py")
            command = '(Get-WmiObject Win32_OperatingSystem).SystemDrive'
            output = self.machine.execute_command(command)

            target_path = output.output.strip() + "\\"

            exe_file = self.host_machine.generate_executable(disk_compare_py)
            self.machine.copy_from_local(exe_file, target_path)
            remote_exe_file_path = None

            retry = 10
            while retry:
                self.log.info(
                    'Waiting for 2 minutes to copy the extent exe folder')
                time.sleep(120)
                _exe_file = os.path.join(target_path, os.path.split(exe_file)[1:][0])

                if self.machine.check_file_exists(_exe_file):
                    remote_exe_file_path = _exe_file
                    break
                else:
                    self.machine.copy_from_local(exe_file, target_path)

                retry -= 1
            else:
                raise Exception("Failed to copy the extent exe folder within 10 tries")

            self.log.info("Waiting 2 minutes before running executable")
            time.sleep(120)

            cmd = "iex " + "\"" + remote_exe_file_path + " -t " \
                  + str(num_threads) + " -m " + str(multiplier) + "\""
            try:
                self.log.info("Executing command {0} on MA Machine".format(cmd))
                output = self.machine.execute_command(cmd)
            except Exception as err:
                self.log.error("Failed to run executable on remote machine")

            # Delete remote executable
            try:
                self.machine.delete_file(remote_exe_file_path)
            except Exception as err:
                self.log.error("An error occurred in deleting file %s", str(err))

            # Delete local executable
            try:
                self.host_machine.delete_file(exe_file)
            except Exception as err:
                self.log.error("An error occurred in deleting file %s", str(err))

            return output.output

        except Exception as err:
            self.log.exception(str(err))
            raise Exception(err)

    def set_disk_props(self, props_dict):
        """
        Runs a PowerShell script which sets disks properties 
        such as offline and read-only, can be specified via props_dict

        Raise Exception:
                if script was not run successfully
        """
        try:
            ps_path = os.path.join(self.utils_path, "ToggleDiskReadOnlyOffline.ps1")
            self.update_vm_info('All', True, True)
            output = self.machine.execute_script(ps_path, props_dict)

        except Exception as err:
            self.log.exception("An exception occurred in setting disk properties")
            raise Exception(err)

    def get_os_disk_number(self):
        """
            Gets the disk number of the disk which has the OS installed
        """
        try:
            command = 'gwmi -query "Select * from Win32_DiskPartition WHERE Bootable = True" | foreach { ' \
                      '$_.DiskIndex} '
            self.update_vm_info('All', True, True)
            output = self.machine.execute_command(command)
            os_disk = int(output.output)
            return os_disk

        except Exception as err:
            self.log.exception("An exception occurred in trying to get OS disk number")
            raise Exception(err)
