# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for performing operations relating to live sync

classes defined:

    Base class:

        LiveSyncUtils                   -   Act as base class for all Live Sync operations

    Methods:

        validate_live_sync()            -   Validate Live Sync

        get_recent_replication_job()    -   Get latest replication job corresponding to the Live Sync Schedule

        cleanup_live_sync()             -   Delete replicated VM and Live Sync Schedule
"""

from time import sleep

from AutomationUtils import logger
from Server.Scheduler.schedulerhelper import SchedulerHelper
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.VirtualServerHelper import AutoVSAVSClient
from VirtualServer.VSAUtils.VirtualServerHelper import AutoVSAVSInstance


class LiveSyncUtils:
    """

    Base Class for performing Live Sync related functions.
    """

    def __init__(self, auto_subclient, schedule_name):
        """
        Initialize common variables for LiveSyncUtils

        Args:

            auto_subclient          -       (AutoVSASubclient)  Auto VSA subclient

            schedule_name:          -       (str)   schedule name of live sync
        """
        self.log = logger.get_log()
        self.schedule_name = schedule_name
        self._auto_subclient = auto_subclient
        auto_subclient.subclient.live_sync.refresh()
        self._live_sync_pair = auto_subclient.subclient.live_sync.get(schedule_name)
        self._vm_pairs = self._live_sync_pair.vm_pairs
        self._vm_pair = None
        self._destination_vms = None
        self._destination_client = None
        self._agent = None
        self._instance = None
        self._dest_auto_client = None
        self._dest_auto_vsa_instance = None
        self.dest_region = None

    @property
    def vm_pair(self):
        """

        First vm pair from vm_pairs

        Returns:
            (LiveSyncVMPair) object for vm pair

        """
        if not self._vm_pair:
            self._auto_subclient.subclient.live_sync.refresh()
            self._vm_pair = self._live_sync_pair.get(next(iter(self._vm_pairs)))
        return self._vm_pair

    @property
    def live_sync_pair(self):
        """

        Get Live Sync Pair

        Returns:
            live_sync_pair  -   (LiveSyncPair) Object for Live Sync Pair

        """
        return self._live_sync_pair

    @property
    def vm_pairs(self):
        """

        Get Live Sync VM pairs

        Returns:
            vm_pairs    -   (LiveSyncVMPairs) Object for Live Sync VM pairs

        """
        return self._vm_pairs

    @property
    def destination_client(self):
        """

        Get destination virtualization client for live sync

        Returns:
            destination_client   -   (Client)  Virtualization client object

        """
        if not self._destination_client:
            self._destination_client = self._auto_subclient.auto_commcell.commcell \
                .clients.get(self.vm_pair.destination_client)
        return self._destination_client

    @property
    def agent(self):
        """

        Get Agent object for Live Sync

        Returns:
            agent       -   (Agent) Agent Object

        """
        if not self._agent:
            self._agent = self.destination_client.agents.get('virtual server')
        return self._agent

    @property
    def instance(self):
        """

        Get Instance object for the Live sync Pair

        Returns:
            instance    -   (Instance) Instance Object

        """
        if not self._instance:
            self._instance = self.agent.instances.get(self.vm_pair.destination_instance)
        return self._instance

    @property
    def dest_auto_client(self):
        """

        Get destination auto client

        Returns:
            dest_auto_client    -   (AutoVSAVSClient) Destination Auto client

        """
        if not self._dest_auto_client:
            self._dest_auto_client = AutoVSAVSClient(self._auto_subclient.auto_commcell, self.destination_client)
        return self._dest_auto_client

    @property
    def destination_vms(self):
        """

        Get list of destination VMs in Live Sync

        Returns:
            destination_vms     -   (list) List of destination VMs

        """
        if not self._destination_vms:
            self._destination_vms = [self.live_sync_pair.get(vm_pair).destination_vm for vm_pair in self._vm_pairs]
        return self._destination_vms

    @property
    def dest_auto_vsa_instance(self):
        """

        Destination Auto VSA instance

        Returns:
            dest_auto_vsa_instance  -   (AutoVSAVSInstance) Destination Auto VSA Instance

        """
        if not self._dest_auto_vsa_instance:
            self.log.info("Initializing destination AutoVSAVSInstance")
            self._dest_auto_vsa_instance = AutoVSAVSInstance(self.dest_auto_client, self.agent, self.instance,
                                                             region=self.dest_region)

            for dest_vm_name in self.destination_vms:
                self._dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
        return self._dest_auto_vsa_instance

    def get_recent_replication_job(self, backup_jobid=None):
        """
        Args:
            backup_jobid (int): will get the replication job id that ran after backup job

        Returns latest replication job, for live sync schedule

        Returns:
            replication_job (Job) : Job Object for latest replication job

        """
        self._auto_subclient.auto_vsaclient.vsa_client.schedules.refresh()
        schedule = self._auto_subclient.auto_vsaclient.vsa_client.schedules.get(self.schedule_name)
        schedule_helper = SchedulerHelper(schedule, self._auto_subclient.auto_commcell.commcell)
        if not backup_jobid:
            replication_job = schedule_helper.get_jobid_from_taskid()
        else:
            for index in range(15): #replication thread is triggered once in 15 minutes
                # Get latest replication job from schedule helper
                temp = schedule_helper.get_jobid_from_taskid()
                if temp.job_id > backup_jobid:
                    replication_job = temp
                    break
                if index < 15:
                    self.log.info(
                        f"New Replication job not yet triggered after backup job {backup_jobid} "
                        f"recheck after 1 minute"
                    )
                    sleep(60)
            else:
                raise Exception('New Replication job not triggered after backup job {backup_jobid}')

        return replication_job

    def validate_live_sync(self, replication_run=True, check_replication_size=True, schedule=None):
        """To validate VSA live sync

        Args:

            replication_run (bool)          -       Set to True to check if replication job is triggered
            for the backup,else False, default: True

            check_replication_size (bool)   -       Set to False if incremental job is coverted to full replication

             schedule(object) -- schedule object for replication schedule to be validated

        Raises:
            Exception

                - If validation fails

        """

        for vm_pair in self.vm_pairs:
            self.log.info('validating VM pair: "%s"', vm_pair)

            source_vm = self._auto_subclient.hvobj.VMs[vm_pair]
            source_vm.update_vm_info('All', os_info=True, force_update=True)
            self.log.info("Vm Pair Props")
            self._vm_pair = self.vm_pair
            if self._vm_pair._properties["VMReplInfoProperties"]:
                for vm_property in self._vm_pair._properties["VMReplInfoProperties"]:
                    if vm_property["propertyId"] == 2207:
                        self.dest_region = vm_property["propertyValue"]
                        self.dest_region = self.dest_region[:-1]
                        break
            dest_vm_name = self.live_sync_pair.get(vm_pair).destination_vm
            self.dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
            dest_vm = self.dest_auto_vsa_instance.hvobj.VMs[dest_vm_name]

            dest_vm.power_on()
            self.log.info('Successfully powered on VM: "%s"', dest_vm_name)

            # Wait for IP to be generated
            wait = 10

            while wait:
                self.log.info('Waiting for 60 seconds for the IP to be generated')
                sleep(60)
                try:
                    dest_vm.update_vm_info('All', os_info=True, force_update=True)
                except Exception as exp:
                    self.log.info(exp)

                if dest_vm.ip and VirtualServerUtils.validate_ip(dest_vm.ip):
                    break
                wait -= 1
            else:
                self.log.error('Valid IP not generated within 10 minutes')
                raise Exception(f'Valid IP for VM: {dest_vm_name} not generated within 5 minutes')
            self.log.info('IP is generated')

            vm_pair_obj = self.live_sync_pair.get(vm_pair)

            replication_job = self._auto_subclient.auto_commcell.commcell.job_controller.get(
                vm_pair_obj.latest_replication_job)
            if replication_run:
                # To validate if replication job completed successfully
                if(self._auto_subclient.auto_commcell.check_v2_indexing(
                        self._auto_subclient.auto_vsaclient.vsa_client.client_name)):
                    self.log.info(f"Last synced Backup Job : {vm_pair_obj.last_synced_backup_job}")
                    child_jobs = self._auto_subclient.auto_commcell.get_child_jobs(self._auto_subclient.backup_job.job_id)
                    assert str(vm_pair_obj.last_synced_backup_job) in child_jobs, \
                        f"Replication job failed to sync latest backup job {self._auto_subclient.backup_job.job_id}"
                    self.log.info('Backup job sync successful')

                    assert self._auto_subclient.auto_commcell.get_backup_pending_jobs_to_replicate(
                        source_vm.vm_name) == [''], f"Pending backup jobs to sync for VM: {source_vm.vm_name}"
                else:
                    self.log.info(f"Last synced Backup Job : {vm_pair_obj.last_synced_backup_job}")
                    assert str(vm_pair_obj.last_synced_backup_job) == str(self._auto_subclient.backup_job.job_id), \
                        f"Replication job failed to sync latest backup job {self._auto_subclient.backup_job.job_id}"
                    self.log.info('Backup job sync successful')
                backup_job = self._auto_subclient.auto_commcell.commcell.job_controller.get(
                    self._auto_subclient.backup_job.job_id)
                if backup_job.backup_level == 'Incremental' and check_replication_size:
                    assert replication_job.size_of_application < (backup_job.size_of_application + 10485760), \
                        "Replication job has replicated more data than expected for Incremental backup"
                    self.log.info('Data replicated for incremental job validation successful')
            else:
                # To validate if replication job never started
                assert str(vm_pair_obj.last_synced_backup_job) != str(self._auto_subclient.backup_job.job_id), \
                    f"Replication Job started for Synthetic full, failing case"
                self.log.info('Replication run not started for Synthetic, validation successful')

            # To validate sync status
            assert vm_pair_obj.status == 'IN_SYNC', \
                f'VM pair : "{vm_pair}" \n Status: "{vm_pair_obj.status} \n Validation failed"'
            self.log.info('Sync status validation successful')

            _livesyncsource = self._auto_subclient.LiveSyncVmValidation(source_vm, schedule, replication_job)
            _livesyncdest = self._auto_subclient.LiveSyncVmValidation(dest_vm, schedule, replication_job)

            # hypervisor specific validation
            assert _livesyncsource == _livesyncdest, "Error while validation"


            # To validate test data between source and destination
            if replication_run:
                for label, drive in dest_vm.drive_list.items():
                    dest_path = source_vm.machine.join_path(
                        drive, self._auto_subclient.backup_folder_name, "TestData", self._auto_subclient.timestamp)
                    self._auto_subclient.fs_testdata_validation(dest_vm.machine, dest_path)
                self.log.info('Testdata validation successful')

            dest_vm.power_off()
            self.log.info('Successfully powered off VM: "%s"', dest_vm_name)

            self.log.info('Validation successful for VM pair: "%s"', vm_pair)

    def cleanup_live_sync(self):
        """

        To clean up live sync operations

        Raises:
            Exception

                - If cleanup operation fails

        """

        for vm_pair in self.vm_pairs:
            dest_vm_name = self.live_sync_pair.get(vm_pair).destination_vm
            # Checking Existence of destination VM before deleting, in case the replication was killed
            if self.dest_auto_vsa_instance.hvobj.check_vms_exist([dest_vm_name]):
                self.dest_auto_vsa_instance.hvobj.VMs = dest_vm_name
                dest_vm = self.dest_auto_vsa_instance.hvobj.VMs[dest_vm_name]
                dest_vm.power_off()

                # To delete the replicated VM
                output = dest_vm.delete_vm()
                if output:
                    self.log.info('Successfully deleted the replicated VM : "%s"', dest_vm_name)
                else:
                    raise Exception(f'Failed to delete the VM {dest_vm_name} please check the logs')

        # To delete the created live sync configuration
        self._auto_subclient.subclient._client_object.schedules.delete(self.schedule_name)
        self.log.info('Successfully deleted the Live sync configuration schedule %s', self.schedule_name)

        self.log.info('Live sync cleanup operation is successful')
