# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that act as wrapper for testcase and SDK

classes defined:

    KubernetesHelper    -- Kubernetes Helper class


"""

import os
import re
import time
from abc import ABCMeta
import json, yaml
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import logger
from AutomationUtils import machine
from past.builtins import basestring
from AutomationUtils.database_helper import get_csdb
from VirtualServer.VSAUtils import OptionsHelper
from VirtualServer.VSAUtils.VirtualServerHelper import (
    AutoVSACommcell,
    AutoVSAVSClient,
    AutoVSAVSInstance,
    AutoVSABackupset,
    AutoVSASubclient
)
from Database.MongoDBUtils.mongodbhelper import MongoDBHelper
from Web.AdminConsole.Helper.cassandra_helper import CassandraHelper


class KubernetesHelper(object):
    __metaclass__ = ABCMeta
    """
    class that act as wrapper for SDK and Testcase

    Methods:
            populate_tc_inputs()           - populate tc inputs

            populate tc inputs for cassandra/mongo db test

            source_vm_object_creation()    - Initialise Kuberenetes VM objects

            backup()                       - Kubernetes backup

            restore_out_of_place()         - Kubernetes out of place restore

            get_pods_services()            - get list of pods and services in namespace

            create_name_space()            - Create namespace of kubernetes

            delete_and_recreate_namespace()- Delete and Recreate Namespace

            populate_data()                - Populate Pod Data

            upload_pod_data()              - Upload Pod data

            download_pod_data()            - Download Pod Data

            get_pods()                     - get list of pods in namespace
                             
            move_pod_content()              - Move pod content in pod for validation

            create_pod_yaml()               - create pod yaml file for deployment

            create_pvc_yaml()                - create pvc yaml file for deployment
                    
            create_svc_yaml()               - create Service yaml file for deployment
            
            create_secret_pod_yaml()        - create Secret pod yaml file for deployment
            
            create_secret_yaml()            - create Secret yaml file for deployment
            
            create_configmap_yaml()         - create Configmap  yaml file for deployment
            
            create_blocklevel_yaml()        - Create config yaml for blocklevel
             
            validate_block_data()           - Validate block data
            
            add_block_data()                - Add data to block device
            
            download_block_file()           - Download data from block device

            validate_restored_manifest()    - validate restored manifest data

            get_pvcs()                      - get list of pvcs in namespace

            create_db_pvc_yaml()            - create pvc yaml file for cassandra/mongo deployment

            create_cassandra_statefulset    - create statefulset yaml file for cassandra deployment

            create_mongo_statefulset        - create statefulset yaml for mongo deployment

            create_cassandra_svc_yaml       - create cassandra service yaml file for deployment

            create_mongo_svc_yaml           - create mongo service yaml file for deployment

            get_db_nodeinfo                 - get worker node and port informatoin for DB

            drop_cassandra_data             - drop cassandra db data

            populate_cassandra_mongo_db     - populate cassandra and mongo db test data

            validate_db_restore_data        - validate db restored data

            populate_db_tc_inputs           - populate db tc inputs

            generate_imagepull_secret       - generate imagePullSecret
    """

    def __init__(self, testcase):
        """
        Initialize common variables for hypervior
            testcase (obj)  -- testcase object

        """
        self.testcase = testcase
        self.log = logger.get_log()
        self._csdb = get_csdb()
        self._VMs = {}

    def populate_tc_inputs(self, cv_testcase):
        """Initializes all the test case inputs after validation

        Args:
            cv_testcase    (obj)    --    Object of CVTestCase
            mandatory      (bool)   --    whether to check for mandatory inputs
            and raise exception if not found

        Returns:
            None

        Raises:
            Exception:
                if a valid CVTestCase object is not passed.
                if CVTestCase object doesn't have agent initialized
        """

        if not isinstance(cv_testcase, CVTestCase):
            raise Exception(
                "Valid test case object must be passed as argument"
            )
        self.agent_name = cv_testcase.tcinputs.get('AgentName', None)
        self.instance_name = cv_testcase.tcinputs.get('InstanceName', None)
        self.backupset_name = cv_testcase.tcinputs.get('BackupsetName', None)
        self.master_node = cv_testcase.tcinputs.get('MasterNode', None)
        self.username = cv_testcase.tcinputs.get("Username", None)
        self.password = cv_testcase.tcinputs.get('Password', None)
        self.scripts_file = cv_testcase.tcinputs.get('ScriptsFile', None)
        self.secret_name = cv_testcase.tcinputs.get('secretName', None)
        self.secret_key = cv_testcase.tcinputs.get('secretKey', None)
        self.vsa_client = cv_testcase.tcinputs.get('VSAClient', None)
        self.plan_name = cv_testcase.tcinputs.get('PlanName', None)
        self.secret_key = cv_testcase.tcinputs.get('secretKey', None)
        self.storage_type = cv_testcase.tcinputs.get('StorageType', False)
        self.subclient_name = cv_testcase.tcinputs.get('SubclientName', None)
        self.client_name = cv_testcase.tcinputs.get('ClientName', None)
        self.dest_client = cv_testcase.tcinputs.get('DestinationClient', False)
        self.namespace = cv_testcase.tcinputs.get('Namespace', False)
        self.restore_namespace = cv_testcase.tcinputs.get('RestoreNamespace', False)
        self.master_machine = machine.Machine(cv_testcase.tcinputs.get('MasterNode'),
                                              username=cv_testcase.tcinputs.get('Username'),
                                              password=cv_testcase.tcinputs.get('Password'))

    def source_vm_object_creation(self, cv_testcase):
        """To create basic VSA SDK objects
        Args:
            cv_testcase    (obj)    --    Object of CVTestCase
        Returns:
            None
        Raises:
            Exception:
                if a valid CVTestCase object is not passed.
                if CVTestCase object doesn't have agent initialized
        """
        self.commcell = AutoVSACommcell(cv_testcase.commcell, cv_testcase.csdb)
        self.client = AutoVSAVSClient(self.commcell, cv_testcase.client)
        self.instance = AutoVSAVSInstance(self.client, cv_testcase.agent, cv_testcase.instance)
        self.backupset = AutoVSABackupset(self.instance, cv_testcase.backupset)
        self.auto_subclient = AutoVSASubclient(self.backupset, cv_testcase.subclient)
        self.subclient = cv_testcase.subclient

    def backup(self, backup_type):
        """ To Run Kubernetes backup
        Args:
            backup_type    (str)    --    backup type
        Returns:
            None
        Raises:
            Exception:
                if a valid CVTestCase object is not passed.
                if CVTestCase object doesn't have agent initialized
        """
        backup_options = OptionsHelper.BackupOptions(self.auto_subclient)
        if backup_type == "FULL":
            backup_options.backup_type = 'FULL'
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")
        elif backup_type == "INCREMENTAL":
            backup_options.backup_type = 'INCREMENTAL'
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")
        else:
            backup_options.backup_type = 'SYNTHETIC_FULL'
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")

        self.log.info("Running Backup")
        self.auto_subclient.backup(backup_options)

    def restore_out_of_place(self, client_name, storage_type, datacenter):
        """
        perform  Kubernetes pod/ VM level restore

        Args:
                client_name             (String):   client

                storage_type            (string):  Storage type to used with restore

                datacenter              (string): namespace / destination to restore to

        Exception:
                        if job fails
                        if validation fails
        """
        try:
            browse_output = self.subclient.browse()
            source_paths = []
            for path in browse_output[0]:
                source_paths.append(path.strip("\\"))
            self.log.info("sourcepaths are %s", str(source_paths))
            for source_path in source_paths:
                job = self.subclient.full_vm_restore_out_of_place(vm_to_restore=source_path,
                                                                  restored_vm_name=source_path,
                                                                  vcenter_client=client_name,
                                                                  kubernetes_host=client_name,
                                                                  datastore=storage_type,
                                                                  datacenter=datacenter)
                self.log.info("Started restore out of place job with job id: %s", str(job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run restore out of place job with error: {0}".format(
                            job.delay_reason))

                if not job.status.lower() == "completed":
                    raise Exception(
                        "Job status is not Completed, job has status: {0}".format(
                            job.status))

                self.log.info("Successfully finished restore out of place job")
                time.sleep(100)
            return job
        except Exception as exp:
            logger.get_log().error('Kubernetes Out of place restore failed : %s ', str(exp))
            return False


    def restore_in_place(self, client_name, storage_type, datacenter):
        """
        perform  Kubernetes pod/ VM level restore

        Args:
                client_name             (String):   client

                storage_type            (string):  Storage type to used with restore

                datacenter              (string): namespace / destination to restore to

        Exception:
                        if job fails
                        if validation fails
        """
        try:
            browse_output = self.subclient.browse()
            source_paths = []
            for path in browse_output[0]:
                source_paths.append(path.strip("\\"))
            self.log.info("sourcepaths are %s", str(source_paths))
            for source_path in source_paths:
                job = self.subclient.full_vm_restore_in_place(kubernetes_host=client_name,
                                                              datastore=storage_type,
                                                              datacenter=datacenter)
                self.log.info("Started IN-PLACE restore  job with job id: %s", str(job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run IN-PLACE restore  job with error: {0}".format(
                            job.delay_reason))

                if not job.status.lower() == "completed":
                    raise Exception(
                        "Job status is not Completed, job has status: {0}".format(
                            job.status))

                self.log.info("Successfully finished restore IN-PLACE job")
                time.sleep(100)
            return job
        except Exception as exp:
            logger.get_log().error('Kubernetes inplace restore failed : %s ', str(exp))
            return False


    def disk_restore(self,
                     vm_name,
                     destination_path,
                     disk_name=None):
        """
        perform  Kubernetes pod/ VM level restore

        Args:
                vm_name                 (String):   VM for restore.

                destination_path        (string):    Destination path to restore disk data to.

                disk_name              (string):     Diskname to be restored if we have anything specific or else
                                                     will enumerate what VM used


        Exception:
                        if job fails
                        if validation fails
        """
        try:
            browse_output = self.subclient.browse()
            source_paths = []
            for path in browse_output[0]:
                source_paths.append(path.strip("\\"))
            self.log.info("sourcepaths are %s", str(source_paths))
            for source_path in source_paths:
                job = self.subclient.disk_restore(vm_name =source_path, destination_path=destination_path)
                self.log.info("Started IN-PLACE restore  job with job id: %s", str(job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run IN-PLACE restore  job with error: {0}".format(
                            job.delay_reason))

                if not job.status.lower() == "completed":
                    raise Exception(
                        "Job status is not Completed, job has status: {0}".format(
                            job.status))

                self.log.info("Successfully finished restore IN-PLACE job")
                time.sleep(100)
            return job
        except Exception as exp:
            logger.get_log().error('Disk restore error: %s ', str(exp))
            return False

    def get_pods_services(self, namespace, host=None):
        """
        Function to get list of pods and service on mentioned namespace.
        Args:
            host                (obj):  Machine class object of kubernetes Master/Kubectl machine.
            namespace           (str):  kubernetes namespace
        Returns:
            element_list       (list): list of namespace entities

        Raises:
                Exception:
                    if it fails to create get pod information.

        """
        try:
            log = logger.get_log()
            cmd = "kubectl get secrets,services,pods,pvc,configmaps -n {}".format(namespace)
            output = self.master_machine.execute_command(cmd)
            log.info("Getting pod information by running command {} ".format(cmd))
            elements_list = []
            for element in output.formatted_output:
                for i in element:
                    if 'secret/' in i:
                        elements_list.append(i)
                    elif 'service/' in i:
                        elements_list.append(i)
                    elif 'pod/' in i and 'Running' in element:
                        elements_list.append(i)
                    elif 'persistentvolumeclaim/' in i:
                        elements_list.append(i)
                    elif 'configmap/' in i:
                        elements_list.append(i)
                    else:
                        pass
            return elements_list

        except Exception as exp:
            logger.get_log().error('get_pods_services_error: %s ', str(exp))
            return False

    def create_name_space(self, delete=False):
        """
        Function to create namespace and setup things for Automation.
        Args:
            delete          (bool): Switch to delete existing /not.

        Raises:
                Exception:
                    if it fails to create namespace

        """
        try:
            log = logger.get_log()
            cmd = "kubectl get namespace "
            output = self.master_machine.execute_command(cmd)
            log.info("Getting pod information by running command %s ", str(cmd))
            list_namespaces = []
            for element in output.formatted_output:
                list_namespaces.append(element[0].lower())
            for i in list_namespaces:
                if self.namespace in i:
                    log.info("Found namespace already exists Removing it %s", str(i))
                    cmd = "kubectl delete namespace {}".format(self.namespace)
                    output = self.master_machine.execute_command(cmd)
                    log.info("running command %s ", str(cmd))
                    if output.exit_code != 0:
                        log.exception("cmd failed to execute")
                        raise Exception('cmd failed to execute')
                if self.restore_namespace in i:
                    log.info("Found namespace already exists Removing it %s", str(i))
                    cmd = "kubectl delete namespace {}".format(str(self.restore_namespace))
                    output = self.master_machine.execute_command(cmd)
                    log.info("running command %s ", str(cmd))
                    if output.exit_code != 0:
                        log.exception("cmd failed to execute")
                        raise Exception('cmd failed to execute')
            if delete:
                return

            cmd = "kubectl create namespace {}".format(self.namespace)
            output = self.master_machine.execute_command(cmd)
            cmd = "kubectl create namespace {}".format(self.restore_namespace)
            output = self.master_machine.execute_command(cmd)
            log.info("Getting pod information by running command %s ", str(cmd))
            if output.exit_code != 0:
                log.exception("cmd failed to execute")
                raise Exception('cmd failed to execute')

        except Exception as exp:
            logger.get_log().error('get_pods_services_error: %s ', (exp))
            return False

    def delete_and_recreate_namespace(self, namespace=None):
        """
        Function to refresh namespace.
        Args:
            namespace          (bool): namespace information to refresh
                                      default: restore_namespace
        Raises:
                Exception:
                    if it fails to create namespace
        """
        try:
            if namespace:
                ns = namespace
            else:
                ns = self.restore_namespace
            log = logger.get_log()
            cmd = "kubectl get namespace "
            output = self.master_machine.execute_command(cmd)
            log.info("Getting namespace by running command %s ", str(cmd))
            list_namespaces = []
            for element in output.formatted_output:
                list_namespaces.append(element[0].lower())
            for i in list_namespaces:
                if ns in i:
                    log.info("Found namespace already exists Removing it %s ", str(i))
                    cmd = "kubectl delete namespace  {}".format(ns)
                    output = self.master_machine.execute_command(cmd)
                    log.info("running command %s ", str(cmd))
                    if output.exit_code != 0:
                        log.exception("cmd failed to execute")
                        raise Exception("cmd failed to execute")
            cmd = "kubectl create namespace {}".format(ns)
            output = self.master_machine.execute_command(cmd)
            log.info("Getting pod information by running command %s ", str(cmd))
            if output.exit_code != 0:
                log.exception("cmd failed to execute")
                raise Exception("cmd failed to execute")
        except Exception as exp:
            logger.get_log().error('get_pods_services_error: %s', str(exp))
            return False

    def populate_data(self, scripts_path):
        """
        Function to create scripts for data population.
        Args:
            self        (obj):  self object

            scripts_path   (str):  scripts path on controller.

        Raises:
                Exception:
                    if it fails to create namespace

        """
        log = logger.get_log()
        remote_path = '/tmp'
        head_tail = os.path.split(scripts_path)
        log.info("copying deployment file to controller")
        output = self.master_machine.copy_from_local(scripts_path, remote_path)
        if not output:
            log.exception("File failed to copy to controller")
            raise Exception('File failed to copy to controller')
        file_path = remote_path + "/" + head_tail[1]
        cmd = "kubectl apply -f {0} -n {1}".format(file_path, self.namespace)
        output = self.master_machine.execute_command(cmd)
        log.info("Running Scripts %s ", str(cmd))
        if output.exit_code != 0:
            log.exception("cmd failed to execute")
            raise Exception('cmd failed to execute')

    def upload_pod_data(self, pod_name, data_path, tcid, job_type='FULL',
                        upload_pod_path=None):
        """
        Function to create scripts for data population.
        Args:
            self        (obj):  self object

            pod_name        (str):  pod to upload data to

            data_path       (str):  Data path on controller to be uploaded.

            upload_pod_path  (str) : path to upload data on pod

            job_type        (str):  backup job type.
                                    default: FULL
        Raises:
                Exception:
                    if it fails to create namespace
        """
        log = logger.get_log()
        dir_path = data_path
        if upload_pod_path is None:
            upload_path = "/tmp/automation_{0}".format(tcid)
        if job_type == 'FULL':
            data_path = dir_path + "/{0}".format(str(job_type))
            upload_pod_path = upload_path + "/{0}".format(str(job_type))
        else:
            data_path = dir_path + "/{0}".format(str(job_type))
            upload_pod_path = upload_path + "/{0}".format(str(job_type))
        if self.master_machine.check_directory_exists(dir_path):
            self.master_machine.remove_directory(dir_path)
        output = self.master_machine.execute_command("mkdir -p {0}".format(str(data_path)))
        if output.exit_code != 0:
            log.exception("create directory %s failed to create", str(data_path))
            raise Exception('create directory failed')
        else:
            log.info("Directory created %s ", str(data_path))

        log.info("Creating test data at: %s ", str(data_path))
        self.master_machine.generate_test_data(
            data_path
        )
        log.info("Creating Directory {0} on pod {1} ".format(upload_pod_path, pod_name))
        cmd = "kubectl exec -it  '{0}'  -n '{1}' -- bash -c 'mkdir -p {2} '" \
            .format(pod_name, self.namespace, upload_pod_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            log.exception("cmd failed to execute")
            raise Exception('cmd failed to execute')
        else:
            log.info("Creating Directory {0} on pod {1}  Successful "
                     .format(upload_pod_path, pod_name))
        log.info("Uploading data to pod {0} at location {1}".format(data_path, upload_pod_path))
        cmd = "kubectl cp '{0}' '{1}'/'{2}':{3}".format(data_path, self.namespace, pod_name,
                                                        upload_pod_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            log.exception("cmd failed to execute")
            raise Exception('cmd failed to execute')
        else:
            log.info("Uploaded Data {0} to {1} on pod {2}"
                     .format(upload_pod_path, self.namespace, pod_name))

    def download_pod_data(self,
                          pod_name,
                          namespace,
                          tcid,
                          upload_pod_path=None,
                          restore_path=None):
        """
        Function to create scripts for data population.
        Args:
            self        (obj):  self object

            pod_name        (str):  pod to upload data to

            namespace       (str):  namespace of pod to download from

            upload_pod_path  (str) : path to upload data on pod

            restore_path        (str): path to restore data
        Raises:
                Exception:
                    if it fails to create namespace
        """
        log = logger.get_log()
        if upload_pod_path == None:
            upload_pod_path = "/tmp/automation_{0}".format(tcid)
        if restore_path == None:
            restore_path = "/tmp/automation_restore_{0}_{1}".format(tcid, namespace)

        if self.master_machine.check_directory_exists(restore_path):
            log.info("Directory Exists %s cleaning it ", str(restore_path))
            self.master_machine.remove_directory(restore_path)
        output = self.master_machine.execute_command("mkdir -p {0}".format(str(restore_path)))
        if output.exit_code != 0:
            log.exception("create directory %s failed to create", str(restore_path))
            raise Exception('Create directory failed')
        else:
            log.info("Directory created %s", str(restore_path))
        # Downloading data from pod
        cmd = "kubectl cp  '{0}'/'{1}':{2} {3}".format(namespace, pod_name,
                                                       upload_pod_path, restore_path)
        output = self.master_machine.execute_command(cmd)
        if self.master_machine.check_directory_exists(restore_path):
            log.info("Download done to path %s", str(restore_path))
        else:
            log.execption("Download failed to path {0}".format(restore_path))
            raise Exception('Download Failed')
        if output.exit_code != 0:
            log.exception("cmd failed to execute")
            raise Exception('Download Failed')
        else:
            log.info("Downloaded  Data {0} to {1} on pod {2}".
                     format(upload_pod_path, namespace, pod_name))
        return restore_path

    def get_pods(self, namespace, host=None):
        """
        Function to get list of pods and service on mentioned namespace.
        Args:
            host                (obj):  Machine class object of kubernetes Master/Kubectl machine.
            namespace           (str):  kubernetes namespace
        Returns:
            pods_list           (list): List of pods

        Raises:
                Exception:
                    if it fails to create get pod information.

        """
        try:
            log = logger.get_log()
            if host is None:
                host = self.master_machine
            cmd = "kubectl get pods -n {}".format(namespace)
            output = host.execute_command(cmd)
            log.info("Getting pod information by running command %s ", str(cmd))
            pods_list = []
            if len(output.formatted_output) > 1:
                for i in range(1, len(output.formatted_output)):
                    pods_list.append(output.formatted_output[i][0])
            else:
                log.info("No Pods Found in Namespace.")
            return pods_list

        except Exception as exp:
            logger.get_log().error('get_pods_services_error: %s ', str(exp))
            return False

    def move_pod_content(self, pod_name, src_path, dest_path):
        """
        Function to create scripts for data population.
        Args:
            self        (obj):  self object

            pod_name   (str):  name of the pod you want to move data on

            src_path        (str): source path

            data_path    (str) : Path to populate data


        Raises:
                Exception:
                    if it fails to create namespace
        """
        log = logger.get_log()
        log.info("Moving files on  pod {0}  from {1} to {2}".format(pod_name, src_path, dest_path))
        cmd = "kubectl exec -it  '{0}'  -n '{1}' -- bash -c 'mkdir -p {2} && mv {3} {4}'" \
            .format(pod_name, self.namespace, dest_path, src_path, dest_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            log.exception("cmd failed to execute")
            raise Exception("cmd failed to execute")
        else:
            log.info("Moved folder on pod {0} from {1} to {2}"
                     .format(pod_name, src_path, dest_path))

    def validate_data(self, srclist, rstlist):
        """
        Function to validate list from both namespaces.
        Args:
            srclist   (list): List of pods before backup.

            rstlist   (list):  List of pods after restore.

        Raises:
                Exception:
                    if it fails if validation fails.

        """
        log = logger.get_log()
        final_source, final_rst = [], []

        log.info("Validating lists src list: {0}"
                 " Destination list :{1} ".format(srclist, rstlist))
        for before_bkp in srclist:
            if 'persistentvolumeclaim/' in before_bkp:
                continue
            ss_before_bkp = before_bkp.rsplit("-", 1)
            final_source.append(ss_before_bkp[0])
        for after_bkp in rstlist:
            if 'persistentvolumeclaim/' in after_bkp:
                continue
            ss_after_bkp = after_bkp.rsplit("-", 1)
            final_rst.append(ss_after_bkp[0])
        final_source.sort()
        final_rst.sort()
        if final_source == final_rst:
            log.info("Lists {0} and {1} are identical".format(final_source, final_rst))
        else:
            log.exception("Lists {0} and {1} are not  identical failing testcase "
                          .format(final_source, final_rst))
            raise Exception

    def add_block_data(self, pod_name, backup_type, device_path=None,
                       upload_file=None):

        """
        Function to add_block data to blockdevice.
        Args:
            pod_name   (str): Name of the pod which has block device attached.

            backup_type   (str):  Backup type

            device_path  (str): mount poath of device attached to pod

            upload_file (str) : FIle to upload  to block device



        Raises:
                Exception:
                    if it fails if uploading fails.

        """

        if device_path is None:
            device_path = "/dev/blk001"
        if upload_file is None:
            upload_file = "/var/log/dpkg.log"

        if backup_type is "FULL":
            count = 35745
        if backup_type is "INCREMENTAL":
            count = 70000
        self.log.info("Uploading data to block device %s".format(device_path))
        cmd = "kubectl exec -it  '{0}'  -n '{1}' dd if={2} of={3} bs=1 count={4}" \
            .format(pod_name, self.namespace, upload_file, device_path, count)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            self.log.exception("cmd failed to upload data to block device ")
            raise Exception('cmd failed to execute')
        else:
            self.log.info("Uploading %s on pod %s  Successful"
                          .format(upload_file, device_path))

    def download_block_file(self, pod_name, namespace, restore_path, device_path=None,
                            upload_file=None, download_file=None):

        """
        Function to add_block data to blockdevice.
        Args:
            pod_name   (str): Name of the pod which has block device attached.

            namespace  (str) : pod namespace

            restore_path   (str):  path to restore data

            device_path  (str): mount path of device attached to pod

            upload_file (str) : File to upload  to block device

            download_file(str): File to download


        Raises:
                Exception:
                    if it fails if validation fails.

        """

        if device_path is None:
            device_path = "/dev/blk001"
        if upload_file is None:
            upload_file = "/var/log/"
        if download_file is None:
            download_file = "/tmp/dpkg.log"
        self.log.info("Downloading  data to block device %s".format(device_path))
        cmd = "kubectl exec -it  '{0}'  -n '{1}' dd of={2} if={3} bs=1 count=35745" \
            .format(pod_name, namespace, download_file, device_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            self.log.exception("cmd failed to upload data to block device ")
            raise Exception('cmd failed to execute')
        else:
            self.log.info("Download  %s on pod %s  Successful "
                          , (download_file, device_path))
        restore_path = restore_path + "//" + namespace
        if not self.master_machine.check_directory_exists(restore_path):
            self.master_machine.create_directory(restore_path)
            self.log.info("Download done to path %s", str(restore_path))
        # Downloading data from pod
        cmd = "kubectl cp  '{0}'/'{1}':{2} {3}".format(namespace, pod_name,
                                                       upload_file, restore_path)
        output = self.master_machine.execute_command(cmd)
        if self.master_machine.check_directory_exists(restore_path):
            self.log.info("Download done to path %s", str(restore_path))
        else:
            self.log.execption("Download failed to path {0}".format(restore_path))
            raise Exception('Download Failed')
        if output.exit_code != 0:
            self.log.exception("cmd failed to execute")
            raise Exception('Download Failed')
        else:
            self.log.info("Downloaded  Data {0} to {1} on pod {2}".
                          format(upload_file, self.restore_namespace, pod_name))
        return restore_path

    def create_pvc_yaml(self, dest_path, storage_class=None, type="File"):
        """
        Function to create pvc on namespace and setup things for Automation.
        Args:
            dest_path          (str): Destination to create yaml
            storage_class      (str): Kubernetes Storage class
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_pvc_json = {
            "apiVersion": "v1",
            "kind": "PersistentVolumeClaim",
            "metadata": {
                "name": "automation-pvc",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "spec": {
                "accessModes": [
                    "ReadWriteOnce"
                ],
                "resources": {
                    "requests": {
                        "storage": "10Gi"
                    }
                },
                "storageClassName": "rook-ceph-block"
            }
        }
        template_pvc_json['spec']['storageClassName'] = storage_class
        if type == "Block":
            template_pvc_json['spec']['volumeMode'] = "Block"
            template_pvc_json['metadata']['name'] = "automation-block-pvc"
            template_pvc_json['spec']['resources']['requests']['storage'] = "2Gi"

        with open(dest_path, 'w') as file:
            yaml.dump(template_pvc_json, file)

    def create_pod_yaml(self, dest_path):
        """
        Function to create pod yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_pod_json = {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {
                "name": "automation-pod",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "spec": {
                "containers": [
                    {
                        "name": "web-server",
                        "image": "nginx:latest",
                        "volumeMounts": [
                            {
                                "name": "mypvc",
                                "mountPath": "/tmp"
                            }
                        ]
                    }
                ],
                "volumes": [
                    {
                        "name": "mypvc",
                        "persistentVolumeClaim": {
                            "claimName": "automation-pvc",
                            "readOnly": False
                        }
                    }
                ]
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_pod_json, file)

    def create_svc_yaml(self, dest_path):
        """
        Function to create Service yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_svc_json = {
            "apiVersion": "v1",
            "kind": "Service",
            "metadata": {
                "name": "automation-svc",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "spec": {
                "ports": [
                    {
                        "port": 80,
                        "name": "nginx-test"
                    }
                ],
                "clusterIP": "None",
                "selector": {
                    "app": "nginx-test"
                }
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_svc_json, file)

    def create_secret_pod_yaml(self, dest_path):
        """
        Function to create Secret pod yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_secret_pod_json = {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {
                "name": "automation-pod",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "spec": {
                "containers": [
                    {
                        "name": "web-server",
                        "image": "nginx:latest",
                        "env": [
                            {
                                "name": "POD_ADMIN_PASSWORD",
                                "valueFrom": {
                                    "secretKeyRef": {
                                        "name": "podpassword",
                                        "key": "password"
                                    }
                                }
                            },
                            {
                                "name": "POD_CONFIG",
                                "valueFrom": {
                                    "configMapKeyRef": {
                                        "name": "specialconfig",
                                        "key": "SPECIAL_TYPE"
                                    }
                                }
                            }
                        ],
                        "volumeMounts": [
                            {
                                "name": "mypvc",
                                "mountPath": "/tmp"
                            }
                        ]
                    }
                ],
                "volumes": [
                    {
                        "name": "mypvc",
                        "persistentVolumeClaim": {
                            "claimName": "automation-pvc",
                            "readOnly": False
                        }
                    }
                ]
            }
        }
        with open(dest_path, 'w') as file:
            yaml.dump(template_secret_pod_json, file)

    def create_secret_yaml(self, dest_path):
        """
        Function to secret  yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_secret_json = {
            "apiVersion": "v1",
            "data": {
                "password": "ZWtzLWNvdXJzZS1teXNxbC1wdw=="
            },
            "kind": "Secret",
            "metadata": {
                "name": "podpassword",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "type": "Opaque"
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_secret_json, file)

    def create_configmap_yaml(self, dest_path):
        """
        Function to config map yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_configmap_json = {
            "apiVersion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": "specialconfig",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "data": {
                "SPECIAL_LEVEL": "very",
                "SPECIAL_TYPE": "charm"
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_configmap_json, file)

    def create_configmap_yaml(self, dest_path):
        """
        Function to config map yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_configmap_json = {
            "apiVersion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": "specialconfig",
                "labels": {
                    "app": "nginx-test"
                }
            },
            "data": {
                "SPECIAL_LEVEL": "very",
                "SPECIAL_TYPE": "charm"
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_configmap_json, file)

    def create_blocklevel_yaml(self, dest_path):
        """
        Function to config map yaml
        Args:
            dest_path          (str): Destination to create yamls
        Raises:
                Exception:
                    if it fails to create namespace

        """
        template_blocklevel_json = {
            "apiVersion": "apps/v1",
            "kind": "Deployment",
            "metadata": {
                "name": "auto-deployment-storage",
                "labels": {
                    "app": "nginx-storage"
                }
            },
            "spec": {
                "replicas": 1,
                "selector": {
                    "matchLabels": {
                        "app": "nginx-storage"
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "app": "nginx-storage"
                        }
                    },
                    "spec": {
                        "containers": [
                            {
                                "name": "auto-storage",
                                "image": "nginx:latest",
                                "ports": [
                                    {
                                        "containerPort": 88
                                    }
                                ],
                                "volumeDevices": [
                                    {
                                        "name": "my-nginx-pv",
                                        "devicePath": "/dev/blk001"
                                    }
                                ]
                            }
                        ],
                        "volumes": [
                            {
                                "name": "my-nginx-pv",
                                "persistentVolumeClaim": {
                                    "claimName": "automation-block-pvc",
                                    "readOnly": False
                                }
                            }
                        ]
                    }
                }
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_blocklevel_json, file)

    def validate_block_data(self, source_path, dest_path):
        """
        Function to validate block device data
        Args:
            source_path        (str): Source content path
            dest_path          (str): Destination path to restore
        Raises:
                Exception:
                    if it fails to create namespace

        """
        self.log.info("Diff comparision")
        cmd = "diff  {0}/dpkg.log {1}/dpkg.log".format(source_path, dest_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            self.log.exception("cmd failed to upload data to block device ")
            raise Exception('cmd failed to execute')
        else:
            if output.columns == '':
                self.log.info("Both folders {0} {1} are identical"
                              .format(source_path, dest_path))
            else:
                self.log.info("Mismatch found between folders {0} {1} "
                              .format(source_path, dest_path))
                raise Exception('Mismatch found')
            self.log.info("Diff comparision  done")

        self.log.info("Checksum  comparision")
        cmd = "cksum  {0}/dpkg.log {1}/dpkg.log".format(source_path, dest_path)
        output = self.master_machine.execute_command(cmd)
        if output.exit_code != 0:
            self.log.exception("cmd failed to upload data to block device ")
            raise Exception('cmd failed to execute')
        else:
            self.log.info("Diff comparision  done")

    def validate_restored_manifest(self, manifest_path):
        """
        Function to Compares the current state of the namespace/pod/pvc against the
        state that the namespace/pod/pvc would be in if the manifest was applied.

        Args:
            manifest_path   (String): path to the manifest files.
        Raises:
                Exception:
                    if it fails if validation fails.

        """
        log = logger.get_log()
        cmd = "ls {}".format(manifest_path)
        log.info("Getting list of yaml files by running command %s ", str(cmd))
        output = self.master_machine.execute_command(cmd)

        for yaml in output.formatted_output:
            yaml_name = yaml[0].replace(r"`", r"\`")
            cmd = "kubectl diff -f {0}/{1}".format(manifest_path,yaml_name)
            log.info("running command:  %s ", str(cmd))
            output = self.master_machine.execute_command(cmd)
            if output.exit_code == 0:
                log.info("Yaml {0} and current cluster comparison succesful ".format(yaml[0]))
            elif 'Service' not in yaml_name:
                raise Exception('cmd failed to execute')
            else:
                for element in output.formatted_output:
                    testline = ""
                    for i in element:
                        testline = testline + i
                    #for db application, the service nodeport will be different from source pod
                    #for "kubectl diff" result, if only nodePort is different, then mark result
                    #as pass
                    if 'Service' not in testline:
                        if testline[0] == '+' and 'nodePort' not in testline:
                            raise Exception('cmd failed to execute')
                        elif testline[0] == '-' and 'nodePort' not in testline:
                            raise Exception('cmd failed to execute')
                log.info("Except nodePort, Yaml {0} and current cluster comparison successful".format(yaml[0]))

    def get_pvcs(self, namespace, host=None):
        """
        Function to get list of pvcs on mentioned namespace.
        Args:
            host                (obj):  Machine class object of kubernetes Master/Kubectl machine.
            namespace           (str):  kubernetes namespace
        Returns:
            pvcs_list           (list): List of pvc

        Raises:
                Exception:
                    if it fails to create get pvc information.

        """
        try:
            log = logger.get_log()
            if host is None:
                host = self.master_machine
            cmd = "kubectl get pvc -n {}".format(namespace)
            output = host.execute_command(cmd)
            log.info("Getting pvc information by running command %s ", str(cmd))
            pvcs_list = []
            if len(output.formatted_output) > 1:
                for i in range(1, len(output.formatted_output)):
                    pvcs_list.append(output.formatted_output[i][0])
            else:
                log.info("No pvcs Found in Namespace.")
            return pvcs_list
        except Exception as exp:
            logger.get_log().error('get_pvc_services_error: %s ', str(exp))
            return False

    def populate_db_tc_inputs(self, cv_testcase):
        """Initializes all the test case inputs after cassandra/mongo validation

        Args:
            cv_testcase    (obj)    --    Object of CVTestCase

        Returns:
            None

        """
        self.populate_tc_inputs(cv_testcase)
        self.cassandra_namespace = cv_testcase.tcinputs.get(
            "cassandra_namespace")
        self.cassandra_restore_namespace = cv_testcase.tcinputs.get(
            "cassandra_restore_namespace")
        self.cassandra_keyspace = cv_testcase.tcinputs.get(
            "cassandra_keyspace")
        self.cassandra_tablename = cv_testcase.tcinputs.get(
            "cassandra_tablename")
        self.mongo_namespace = cv_testcase.tcinputs.get("mongo_namespace")
        self.mongo_restore_namespace = cv_testcase.tcinputs.get(
            "mongo_restore_namespace")
        self.mongo_datalist = {}
        self.cassandra_row = 10

    def create_cassandra_svc_yaml(self, dest_path):
        """
        Function to create cassandra service yaml
        Args:
            dest_path          (str): Destination to create cassandra yamls
        """
        template_svc_json = {
            "apiVersion": "v1",
            "kind": "Service",
            "metadata": {
                "name": "cassandra",
                "labels": {
                    "app": "cassandra"
                }
            },
            "spec": {
                "type": "NodePort",
                "selector": {
                    "app": "cassandra",
                },
                "ports": [
                    {
                        "port": 9042,
                        "targetPort": 9042,
                        "nodePort": 30007
                    }
                ]
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_svc_json, file)

    def create_mongo_svc_yaml(self, dest_path):
        """
        Function to create mongo Service yaml
        Args:
            dest_path          (str): Destination to create mongo yamls

        """
        template_svc_json = {
            "apiVersion": "v1",
            "kind": "Service",
            "metadata": {
                "name": "mongo",
                "labels": {
                    "app": "mongo"
                }
            },
            "spec": {
                "type": "NodePort",
                "selector": {
                    "role": "mongo"
                },
                "ports": [
                    {
                        "port": 27017,
                        "targetPort": 27017,
                        "nodePort": 30008
                    }
                ]
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_svc_json, file)

    def create_db_pvc_yaml(
            self,
            dest_path,
            storage_class=None,
            type="Cassandra"):
        """
        Function to create pvc yaml for cassandra/mongo namespace
        Args:
            dest_path          (str): Destination to create yaml
            storage_class      (str): Kubernetes Storage class

        """
        template_db_pvc_json = {
            "apiVersion": "v1",
            "kind": "PersistentVolumeClaim",
            "metadata": {
                "name": "cassandra-data",
            },
            "spec": {
                "accessModes": [
                    "ReadWriteOnce"
                ],
                "resources": {
                    "requests": {
                        "storage": "8Gi"
                    }
                },
                "storageClassName": "rook-ceph-block"
            }
        }

        template_db_pvc_json['spec']['storageClassName'] = storage_class
        if type == "Mongo":
            template_db_pvc_json['metadata']['name'] = "mongo-persistent-storage"
            template_db_pvc_json['spec']['resources']['requests']['storage'] = "11Gi"

        with open(dest_path, 'w') as file:
            yaml.dump(template_db_pvc_json, file)

    def create_cassandra_statefulset(self, dest_path):
        """
        Function to cassandra statefullset yaml
        Args:
            dest_path          (str): Destination to create cassandra yamls

        """
        template_cassandra_json = {
            "apiVersion": "apps/v1",
            "kind": "StatefulSet",
            "metadata": {
                "name": "cassandra",
                "labels": {
                    "app": "cassandra"
                }
            },
            "spec": {
                "serviceName": "cassandra",
                "replicas": 1,
                "selector": {
                    "matchLabels": {
                        "app": "cassandra"
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "app": "cassandra"
                         }
                    },
                    "spec": {
                        "containers": [
                            {
                                "name": "cassandra",
                                "image": "cassandra",
                                "ports": [
                                    {
                                        "containerPort": 7000,
                                             "name": "intra-node"
                                    },
                                    {
                                        "containerPort": 7001,
                                        "name": "tls-intra-node"
                                    },
                                    {
                                        "containerPort": 7199,
                                        "name": "jmx"
                                    },
                                    {
                                        "containerPort": 9042,
                                        "name": "cql"
                                    }
                                ],
                                "resources": {
                                    "limits": {
                                        "cpu": "500m",
                                        "memory": "1Gi"
                                    },
                                    "requests": {
                                        "cpu": "500m",
                                         "memory": "1Gi"
                                    }
                                },
                                "securityContext": {
                                    "capabilities": {
                                        "add": ["IPC_LOCK"]
                                    }
                                },
                                "lifecycle": {
                                    "preStop": {
                                        "exec": {
                                            "command": ['/bin/sh', '-c', 'nodetool drain']
                                        }
                                    }
                                },
                                "env": [
                                    {
                                        "name": "MAX_HEAP_SIZE",
                                        "value": "512M"
                                    },
                                    {
                                        "name": "HEAP_NEWSIZE",
                                        "value": "100M"
                                    },
                                    {
                                        "name": "CASSANDRA_CLUSTER_NAME",
                                        "value": "K8Demo"
                                    },
                                    {
                                        "name": "CASSANDRA_DC",
                                        "value": "DC1-K8Demo"
                                    },
                                    {
                                        "name": "CASSANDRA_RACK",
                                        "value": "Rack1-K8Demo"
                                    },
                                    {
                                        "name": "POD_IP",
                                        "valueFrom": {
                                            "fieldRef": {
                                                "fieldPath": "status.podIP"
                                            }
                                        }
                                    }
                                ],
                                "volumeMounts": [
                                    {
                                        "name": "cassandra-data",
                                        "mountPath": "/var/lib/cassandra"
                                    }
                                ]
                            }
                        ],
                        "imagePullSecrets": [{"name": "autoregcred"}],
                        "volumes": [
                            {
                                "name": "cassandra-data",
                                "persistentVolumeClaim": {
                                    "claimName": "cassandra-data",
                                    "readOnly": False
                                }
                            }
                        ]
                    }
                }
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_cassandra_json, file)

    def create_mongo_statefulset(self, dest_path):
        """
        Function to mongo statefullset yaml
        Args:
            dest_path          (str): Destination to create mongo yamls

        """
        template_mongo_json = {
            "apiVersion": "apps/v1",
            "kind": "StatefulSet",
            "metadata": {
                "name": "mongo",
                "labels": {
                    "environment": "test",
                    "role": "mongo"
                }
            },
            "spec": {
                "serviceName": "mongo",
                "replicas": 1,
                "selector": {
                    "matchLabels": {
                        "environment": "test",
                        "role": "mongo"
                    }
                },
                "template": {
                    "metadata": {
                        "labels": {
                            "environment": "test",
                            "role": "mongo"
                        }
                    },
                    "spec": {
                        "containers": [
                            {
                                "command": ['mongod', '--bind_ip', '0.0.0.0'],
                                "image": "mongo",
                                "name": "mongo",
                                "ports": [
                                    {
                                        "containerPort": 27017,
                                        "protocol": "TCP"
                                    }
                                ],
                                "resources": {},
                                "terminationMessagePath": "/dev/termination-log",
                                "terminationMessagePolicy": "File",
                                "volumeMounts": [
                                    {
                                        "mountPath": "/data/db",
                                        "name": "mongo-persistent-storage"
                                    }
                                ]
                            },
                            {
                                "env": [
                                    {
                                        "name": "MONGO_SIDECAR_POD_LABELS",
                                        "value": "role=mongo,environment=test"
                                    }
                                ],
                                "image": "cvallance/mongo-k8s-sidecar",
                                "name": "mongo-sidecar",
                                "resources": {},
                                "terminationMessagePath": "/dev/termination-log",
                                "terminationMessagePolicy": "File"
                            }
                        ],
                        "volumes": [
                            {
                                "name": "mongo-persistent-storage",
                                "persistentVolumeClaim": {
                                    "claimName": "mongo-persistent-storage",
                                    "readOnly": False
                                }
                            }
                        ],
                        "imagePullSecrets": [
                            {
                                "name": "autoregcred"
                            }
                        ],
                        "dnsPolicy": "ClusterFirst",
                        "restartPolicy": "Always",
                        "schedulerName": "default-scheduler",
                        "securityContext": {},
                        "terminationGracePeriodSeconds": 10
                    }
                },
                "updateStrategy": {
                    "type": "OnDelete"
                }
            }
        }

        with open(dest_path, 'w') as file:
            yaml.dump(template_mongo_json, file)

    def get_db_nodeinfo(self, namespace, app='cassandra'):
        """
        Function to get the worker node name and node port for accessing to the application
        Args:
            self        (obj):  self object

            namespace   (str):  namespace

            app         (str):  application name, 'cassandra' or 'mongo'

        Raises:
            Exception:
                    if it fails to get db worker node and port info
        """
        try:
            log = logger.get_log()
            nodeinfo = []
            cmd = "kubectl get pod -n " + namespace + " -o wide"
            log.info(cmd)
            output = self.master_machine.execute_command(cmd)
            log.info(
                "get worker node for application pod by running command %s ",
                str(cmd))
            for element in output.formatted_output:
                if app in element[0]:
                    nodeinfo.append(element[6])
                    log.info("node name: %s ", element[6])
            cmd = "kubectl get service -n  " + namespace
            log.info(cmd)
            output = self.master_machine.execute_command(cmd)
            log.info(
                "get node port for the service by running command %s ",
                str(cmd))
            for element in output.formatted_output:
                if app in element[0]:
                    ports = re.split('[:/]', element[4])
            nodeinfo.append(ports[1])
            log.info("service NodePort: %s ", ports[1])
            return nodeinfo
        except Exception as exp:
            logger.get_log().error('get db worker node and port error: %s ', str(exp))
            return False

    def drop_cassandra_data(self, cassandra_namespace):
        """
        Function to drop casandra db test data
        Args:
            cassandra_namesapce    (str):   namespace for cassandra pod

        """
        log = logger.get_log()
        log.info("drop keyspace from cassandra db")
        cassandra_nodeinfo = self.get_db_nodeinfo(
            cassandra_namespace, app='cassandra')
        cassandra = CassandraHelper(
            cassandra_nodeinfo[0],
            'cassandra',
            'cassandra',
            cassandra_nodeinfo[1])
        cassandra.dropkeyspace(self.cassandra_keyspace)
        cassandra.close_connection()

    def populate_cassandra_mongo_db(self, commcell, jobtype='FULL'):
        """populate cassandra/mongo db test data"""

        log = logger.get_log()
        log.info("Populating cassandra db")
        time.sleep(120)
        cassandra_nodeinfo = self.get_db_nodeinfo(
            self.cassandra_namespace, app='cassandra')

        cassandra = CassandraHelper(
            cassandra_nodeinfo[0],
            'cassandra',
            'cassandra',
            cassandra_nodeinfo[1])
        if jobtype == 'FULL':
            cassandra.createkeyspace(self.cassandra_keyspace)
            cassandra.createtable(
                self.cassandra_keyspace,
                self.cassandra_tablename)
        else:
            self.cassandra_row = 20

        cassandra.populate_test_data(
            self.cassandra_keyspace,
            self.cassandra_tablename,
            self.cassandra_row)
        cassandra.close_connection()

        log.info("Populating mongo db")
        pods = self.get_pods(self.mongo_namespace)
        test_pod = pods[0]
        time.sleep(120)
        mongo_nodeinfo = self.get_db_nodeinfo(
            self.mongo_namespace, app='mongo')
        mongo = MongoDBHelper(
            commcell, mongo_nodeinfo[0], int(
                mongo_nodeinfo[1]))
        if jobtype == 'FULL':
            self.mongo_datalist = mongo.generate_test_data()
        else:
            datalist2 = mongo.generate_test_data(
                database_prefix="auto_db2", num_dbs=1, num_col=1, num_docs=1)
            self.mongo_datalist.update(datalist2)
        mongo.close_connection()

    def validate_db_restore_data(
            self,
            commcell,
            cassandra_namespace,
            mongo_namespace):
        """
        Function to validate that cassanrda and mongo db data are restored correctly
        Args:
            cassadnra_namespace (string): the name of cassandra namespace
            mongo_namespace    (string):  the name of mongo namespace
        """
        log = logger.get_log()
        log.info("validate restored cassandra db data")
        cassandra_nodeinfo = self.get_db_nodeinfo(
            cassandra_namespace, app='cassandra')
        time.sleep(60)
        cassandra = CassandraHelper(
            cassandra_nodeinfo[0],
            'cassandra',
            'cassandra',
            cassandra_nodeinfo[1])

        size = self.cassandra_row + 1
        origids = list(range(1, size))
        results = cassandra.get_rows(
            self.cassandra_keyspace,
            self.cassandra_tablename)
        count = 0
        for result in results:
            count += 1
            if (result.id not in origids) or (
                    result.fname != 'test') or (result.lname != 'test'):
                raise Exception(
                    "restored cassandra data does not match with original data")
        if count == self.cassandra_row:
            log.info("cassandra db data are restored correctly")
        else:
            raise Exception(
                "number of rows in restored cassandra data does not match original data")
        cassandra.close_connection()

        log.info("validate restored mongo db data")
        mongo_nodeinfo = self.get_db_nodeinfo(mongo_namespace, app='mongo')
        mongo = MongoDBHelper(
            commcell, mongo_nodeinfo[0], int(
                mongo_nodeinfo[1]))
        db_connection = mongo.connection()
        databases = self.mongo_datalist.keys()
        database_names = db_connection.list_database_names()
        for dbs in databases:
            if dbs in database_names:
                dbcollectionlist = self.mongo_datalist[dbs]
                number_docs = dbcollectionlist.pop()
                newdbcon = db_connection[dbs]
                collection_names = newdbcon.list_collection_names()
                for collection in dbcollectionlist:
                    if collection in collection_names:
                        colcon = newdbcon[collection]
                        if number_docs == colcon.count():
                            log.info(
                                "Document count matched for collection : " +
                                collection +
                                " in database :" +
                                dbs)
                        else:
                            raise Exception(
                                "Restored document count did not match original data")
                    else:
                        raise Exception(
                            "Restored document count did not match original data")
            else:
                raise Exception("Database is not present in the server")
        log.info("mongo db data are restored successfully")
        mongo.close_connection()

    def generate_imagepull_secret(
            self,
            regkeyserver,
            dockerusername,
            dockerpassword,
            dockeremail):
        """
        Function to create imagePullSecret.
        Args:
            self        (obj):  self object

            regkeyserver   (str):  https://index.docker.io/v1/

            dockerusername    (str):  docker login username

            dockerpassword    (str):  docker login password

            dockeremail    (str):  email address for docker login

        Raises:
                Exception:
                    if it fails to create imagePullSecret

        """

        log = logger.get_log()

        cmd = "kubectl get secret "
        output = self.master_machine.execute_command(cmd)
        log.info("Getting pod information by running command %s ", str(cmd))
        list_secrets = []
        for element in output.formatted_output:
            list_secrets.append(element[0].lower())
        for i in list_secrets:
            if i == "autoregcred":
                log.info("Found secret already exists Removing it %s", str(i))
                cmd = "kubectl delete secret autoregcred"
                output = self.master_machine.execute_command(cmd)
                if output.exit_code != 0:
                    log.exception("cmd failed to execute")
                    raise Exception('cmd failed to execute')

        cmd = "kubectl create secret docker-registry autoregcred " + \
            "--docker-server=" + regkeyserver + \
            " --docker-username=" + dockerusername + \
            " --docker-password=" + dockerpassword + \
            " --docker-email=" + dockeremail
        self.log.info("execute command %s ", str(cmd))
        output = self.master_machine.execute_command(cmd)
        if output.exit_code == 0:
            log.info("successfully created the ImagePullSecret")
        else:
            raise Exception('cmd failed to execute')

