# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that does all operations on Vmware

    VmwareHelper:

        get_all_vms_in_hypervisor()		        - abstract -get all the VMs in HYper-V Host

        compute_free_resources()		        - compute the Vmware host and Datastore
                                                    for performing restores

        _vmware_login()                         - Login to vcenter 6.5 and above

        _make_request()                         - Rest api calls to vcenter

        _vmware_get_vms()                       - Get list of qualified vms

        _get_vcenter_version()                  - get the vcenter version

        get_all_vms_in_hypervisor()             - Get complete list of vms in the vcenter

        compute_free_resources()                - Calculate free resources

        _get_datastore_dict()                   - Get list of datastore with free space

        _get_host_memory()                      - Get list of esx with free ram and cpu

        _get_required_resource_for_restore()    - Get restore vm ram and storage requirement

        _get_datastore_priority_list()          - Get list of datastore in descending
                                                    order as per free space

        _get_host_priority_list()               -  Get list of esx in descending oder
                                                    as per free ram

        _get_datastore_tree_list()              - get datastore hierarchy

"""

import socket
import re
import time
import os
import threading
from collections import OrderedDict
from AutomationUtils import machine
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from pyVmomi import vim, vmodl


class VmwareHelper(Hypervisor):
    """
    Main class for performing all operations on Vmware Hypervisor
    """

    def __init__(self, server_host_name, user_name,
                 password, instance_type, auto_commcell,
                 host_machine=socket.gethostbyname_ex(socket.gethostname())[2][0],
                 **kwargs):
        """
        Initialize Vmware Helper class properties

        Args:
            server_host_name    (str):      Hostname of the Vcenter

            host_machine        (str):      Vcenter name

            user_name           (str):      Username of the Vcenter

            password            (str):      Password of the Vcenter

            instance_type       (str):      Instance type of the Vmware

            auto_commcell       (object):   Commcell object

            **kwargs            (dict):   Optional arguments
        """
        super(VmwareHelper, self).__init__(server_host_name,
                                           user_name, password, instance_type, auto_commcell, host_machine)
        self.operation_ps_file = "GetVmwareProps.ps1"
        self.vcenter_version = 0
        self.prop_dict = {
            "server_name": self.server_host_name,
            "user": self.user_name,
            "pwd": self.password,
            "vm_name": "$null",
            "extra_args": "$null"
        }
        self.disk_extension = [".vmdk"]
        self.connection = None
        self.vim = vim
        self.vmodl = vmodl
        self.keep_login = None
        self.controller = machine.Machine()
        self._vmware_login()

    class VmwareSession(object):
        """
        Class for Keeping the session for vmware active
        """
        def __init__(self, connected_session, log, update_interval=600):
            """Create daemon thread to keep connection alive
            Create the thread
            Args:
                connected_session             (object): connection object

                log                             (object): object for logging

                update_interval                 (int):      Interval between which thread keep the session active

            """
            super().__init__()
            self._testcase_running = True
            self.connected_session = connected_session
            self.update_interval = update_interval
            self.log = log
            thread = threading.Thread(target=self.run, args=())
            thread.daemon = True
            thread.start()

        def stop(self):
            self._testcase_running = False

        def run(self):
            while self._testcase_running:
                self.log.info('Keeping the session {} alive'.format(
                    self.connected_session.content.sessionManager.currentSession.key))
                time.sleep(self.update_interval)

    def _vmware_login(self):
        """
        Login to vcenter 6.0 and above
        Raises:
            Exception:
                Failed to login to vcenter server
        """
        try:
            from pyVim.connect import SmartConnectNoSSL
            self.connection = SmartConnectNoSSL(host=self.server_host_name,
                                                user=self.user_name,
                                                pwd=self.password)
            if self.connection:
                self.keep_login = self.VmwareSession(self.connection, self.log, 600)
                self.log.info("Connection successful to VC: {}".format(self.server_host_name))
                self.vcenter_version = self.connection.content.about.version
                self.vcenter_version = float('.'.join(self.vcenter_version.split('.')[:-1]))
        except Exception as err:
            self.log.exception("An exception occurred while logging in to the vcenter")
            raise Exception(err)

    def get_content(self, vimtype):
        """
        Get content object
        Args:
            vimtype             (object): vim type

        Returns:
            obj                 (object):   object of the content of vimtype

        Raises:
            Exception:
                Failed to get content
        """

        _try = 0
        obj = {}
        while _try < 3:
            try:
                try:
                    container = self.connection.content.viewManager.CreateContainerView(
                        self.connection.content.rootFolder, vimtype, True)
                except self.vim.fault.NotAuthenticated as err:
                    self.log.info("Session timed out. Reconnecting")
                    self._vmware_login()
                    container = self.connection.content.viewManager.CreateContainerView(
                        self.connection.content.rootFolder, vimtype, True)
                for managed_object_ref in container.view:
                    obj.update({managed_object_ref: managed_object_ref.name})
                container.Destroy()
                return obj
            except Exception as err:
                self.log.info("Error:{} Attempt {}".format(err, _try))
                time.sleep(10)
                _try += 1
        self.log.exception("Not able to get content from VC: {}".format(self.server_host_name))

    def _vmware_get_vms(self):
        """
        Get all vms of the vcenter

        Returns:
             vmlist         (dict): list of vms in the Vcenter

        Raises:
            Exception:
                Not able to get the vms from vcenter
        """
        try:
            vmlist = []
            get_all_vms = self.get_content([self.vim.VirtualMachine])
            for vm in get_all_vms:
                try:
                    vmlist.append(vm.name)
                except AttributeError:
                    self.log.warning("Issue with vm: {}. Skipping it".format(vm))
                    continue
            return vmlist
        except Exception as err:
            self.log.exception("An exception occurred while getting list of vms from vcenter")
            raise Exception(err)

    def get_all_vms_in_hypervisor(self, server="", pattern="", c_type=""):
        """
        Get all the vms for vmware

        Args:
            server          (str):  Vcenter for which all the VMs has to be fetched

            pattern         (str):  Pattern to fetch the vms

            c_type            (str):  Type of content

        Returns:
            _all_vm_list    (str):  List of VMs in the host of the Vcenter

        Raises:
            Exception:
                Not able to get the vms from vcenter
        """
        try:
            _all_vm_list = []
            if not c_type:
                get_all_vms = self._vmware_get_vms()
            else:
                if not pattern:
                    self.log.error("No pattern received")
                else:
                    rep = {'type:server': ':runtime.host.name',
                           'type:datastore': ':datastore[0].name',
                           'type:datacenter': ':parent.parent.name',
                           'type:cluster': ' :runtime.host.parent.name',
                           'type:resource_pool': ':resourcePool.name',
                           'type:vmpowerstate': ':runtime.powerState',
                           'type:vmguestos': ':config.guestFullName',
                           'type:vmguesthostname': ':guest.hostName',
                           'type:tag': ':get-tag',
                           'type:tagcategory': ':get-tagcategory'}
                rep = dict((re.escape(k), v) for k, v in rep.items())
                _pattern = re.compile("|".join(rep.keys()))
                pattern = _pattern.sub(lambda m: rep[re.escape(m.group(0))], pattern)
                if re.search('get-tag', pattern):
                    _ps_path = os.path.join(
                        self.utils_path, self.operation_ps_file)
                    self.prop_dict["property"] = c_type
                    self.prop_dict["extra_args"] = pattern
                    output = self.controller._execute_script(_ps_path, self.prop_dict)
                    if not output.exception:
                        get_all_vms = output.output.rsplit("=", 1)[1].strip().split(",")
                    else:
                        self.log.error("Error in getting tags/category vms: {}".
                                       format(output.exception))
                        raise Exception
                else:
                    if 'id:' in pattern:
                        pattern = pattern.replace('name', '_moId')
                    get_all_vms = self._filter_vms(pattern)

            for vm in get_all_vms:
                if re.match("^[()\sA-Za-z0-9_-]*", vm):
                    _all_vm_list.append(vm)
            return _all_vm_list
        except Exception as err:
            self.log.exception("An exception occurred while getting all VMs from Vcenter")
            raise Exception(err)

    def _filter_vms(self, pattern):
        """
        Filters teh vms from the vm list
        Args:
            pattern             (string):   Patten to be looked for in vms

        Returns:
            vms                 (list):     List of vms matching the pattern

        """
        vms = []
        _prop = pattern.split('\n:')[1]
        _value = pattern.split('\n')[0].split(':')[1]
        all_vms = self.get_content([self.vim.VirtualMachine])
        _pat = re.compile(_value)
        for vm in all_vms:
            if vm.runtime.connectionState == 'connected':
                if _pat.match(eval('vm.%s' % _prop)):
                    vms.append(vm.name)
        return vms

    def compute_free_resources(self, vm_list):
        """
        Compute the free Resource of the Vcenter based on free memory and cpu

        Args:
            vm_list         (list):  list of Vms to be restored

        Returns:
            Datastore	    (str):  The Datastore where restore can be performed

            ESX             (str):  ESX where restore has to be performed

            Cluster         (str):  Cluster where restore has to be performed

            Datacenter      (str):  DataCenter where restore has to be performed

            network         (str):  Netowrk which needs to be attached

        Raises:
            Exception:
                Not able to get Datastore or ESX or Cluster or Datacenter or all

        """
        try:
            _datastore_priority_dict = self._get_datastore_dict()
            _host_priority_dict = self._get_host_memory()
            network = None

            if vm_list:
                _total_vm_memory, _total_disk_space = self._get_required_resource_for_restore(vm_list)
            else:
                _total_vm_memory = 0
                _total_disk_space = 0

            for each_datastore in _datastore_priority_dict.items():
                if (each_datastore[1][0]) > _total_disk_space:
                    datastore_name = each_datastore[0]
                    self.log.info("The Datastore {} has more than total"
                                  "disk space in VM".format(datastore_name))
                    _tree = self._get_datastore_tree_list(datastore_name)
                    if _tree['ESX'] in _host_priority_dict.keys():
                        if _host_priority_dict[_tree['ESX']][0] > _total_vm_memory:
                            for _vm in self.VMs:
                                if _tree['ESX'] != self.VMs[_vm].esx_host:
                                    network = self._get_host_network(_tree['ESX'])
                                    if not network:
                                        break
                            if not network:
                                continue
                            self.log.info(
                                "the Host {} has higher "
                                "memory than the total VMs".format(_tree['ESX']))
                            break
                else:
                    continue

            return each_datastore[0], [_tree['ESX']], _tree['Cluster'], _tree['Datacenter'], network

        except Exception as err:
            self.log.exception("An exception occurred while getting "
                               "datastore or ESX or Cluster or Datacenter or all")
            raise Exception(err)

    def _get_datastore_dict(self, gx_backup=False):
        """
        Get the list of datastore in an ESX

        Args:
            gx_backup           (bool):  include gx_backup datastores

        Returns:
            _disk_size_dict     (dict): Datastores with name and free spaces

        Raises:
            Exception:
                Not able to get datastores from the Vcenter
        """
        try:
            _disk_size_dict = {}
            all_ds = self.get_content([self.vim.Datastore])
            for ds in all_ds:
                if ds.summary.accessible:
                    free_space = int(ds.info.freeSpace / 1024 / 1024 / 1024)
                    _disk_size_dict.setdefault(ds.name, []).append(free_space)
            _disk_size_dict = OrderedDict(sorted(
                _disk_size_dict.items(), key=lambda kv: (kv[1], kv[0]), reverse=True))
            if not gx_backup:
                _disk_size_dict = dict(
                    filter(lambda item: 'GX_BACKUP' not in item[0], _disk_size_dict.items()))
            return _disk_size_dict

        except Exception as err:
            self.log.exception("exception raised in _get_datastore_dict  ")
            raise err

    def _get_all_nics_in_host(self):
        """
        Get all the nics in the host

        Returns:
            nics_dict           (list):     list of nics in each esx host

        Raises:
            Exception:
                Not able to get nic cards from the Vcenter
        """
        try:
            _nics_dict = {}
            all_esx = self.get_content([self.vim.HostSystem])
            for esx in all_esx:
                for network in esx.network:
                    if isinstance(network.name, str):
                        _nics_dict[esx.name] = network.name
            return _nics_dict
        except Exception as err:
            self.log.exception("exception raised in _get_all_nics_in_host  ")
            raise err

    def _get_host_memory(self):
        """
        Get the free memory in the ESX

        Returns:
            _esx_dict           (dict): Dictionary of ESX and its free space

        Raises:
            Exception:
                Raise exception when failed to get Memory of the ESX

        """
        try:
            _esx_dict = {}
            all_esx = self.get_content([self.vim.HostSystem])
            for host in all_esx:
                if host.summary.runtime.connectionState == 'connected':
                    total_memory = host.summary.hardware.memorySize
                    used_memory = host.summary.quickStats.overallMemoryUsage
                    free_memory = int(total_memory / 1024 / 1024 / 1024) - int(used_memory / 1024)
                    _esx_dict.setdefault(host.name, []).append(free_memory)
            _esx_dict = OrderedDict(sorted(
                _esx_dict.items(), key=lambda kv: (kv[1], kv[0]), reverse=True))
            return _esx_dict
        except Exception as err:
            self.log.exception("exception raised in GetMemory  ")
            raise err

    def _get_host_network(self, host):
        """
        Get the network in the ESX

        Args:
            host            (basestring):   the esx host whose network is to be obtained

        Returns:
            network         (basestring):   the network in the ESX host

        Raises:
            Exception:
                Raise exception when failed to get network of the ESX

        """
        try:
            import ipaddress
            vms = []
            network_count = {}
            all_vms = self.get_content([self.vim.VirtualMachine])
            for vm in all_vms:
                try:
                    if vm.runtime.host.name == host and vm.runtime.powerState == 'poweredOn':
                        vms.append(vm)
                except AttributeError:
                    self.log.warning("Issue with vm: {}. Skipping it".format(vm.name))
                    continue
            if len(vms) < 3:
                return None
            for vm in vms:
                _ip = vm.summary.guest.ipAddress
                if _ip and len(vm.network) > 0:
                    if ipaddress.ip_address(_ip) and not re.search('^169\.254', _ip):
                        if vm.network[0].name in network_count:
                            network_count[vm.network[0].name] += 1
                        else:
                            network_count[vm.network[0].name] = 1
            for net_name, count in network_count.items():
                if count > 5:
                    return net_name

        except Exception as err:
            self.log.exception("exception raised in GetNetwork")
            raise err

    def _get_required_resource_for_restore(self, vm_list):
        """
        sums up all the memory of needs to be restores(passed as VM list)

        Args:
            vm_list             (list):  list of vm to be restored

        Returns:
            _vm_total_memory    (int):  Total memory required for restoring

            _vm_total_space     (int):  Total disk space required for restoring

        Raises:
            Exception:
                Raise exception when failed to get space and ram of the source vm
        """
        try:
            copy_vm_list = vm_list
            _vm_total_memory = _vm_total_space = 0
            get_all_vms = self.get_content([self.vim.VirtualMachine])
            vms = dict(filter(lambda elem: elem[1] in copy_vm_list, get_all_vms.items()))
            for vm in vms:
                space = 0
                memory = int(vm.summary.config.memorySizeMB / 1024)
                for _v in vm.storage.perDatastoreUsage:
                    space = space + int(_v.committed / 1024 / 1024 / 1024)
                _vm_total_memory = _vm_total_memory + memory
                _vm_total_space = _vm_total_space + space
            return _vm_total_memory, _vm_total_space

        except Exception as err:
            self.log.exception(
                "An Error occurred in  _get_required_memory_for_restore ")
            raise err

    def _get_datastore_tree_list(self, datastore):
        """
        Get the free host memory in proxy and arrange them with ascending order

        Args:
            datastore           (str):  Datastore for which we need need to find the hierarchy

        Returns:
            tree_list           (dict): Dict contains parent ESX, Cluster, Datacenter for the datastore

        Raises:
            Exception:
                Raise exception when failed to get tree structure for the datastore
        """
        try:
            all_ds = self.get_content([self.vim.Datastore])
            datastores = dict(filter(lambda elem: datastore == elem[1], all_ds.items()))
            for ds in datastores:
                tree_list = {'ESX': ds.host[0].key.name,
                             'Cluster': ds.host[0].key.parent.name,
                             'Datacenter': ds.parent.parent.name}
                return tree_list
            self.log.error("Failed to find datastore")

        except Exception as err:
            self.log.exception("An error occurred in  _get_datastore_tree_list ")
            raise err

    def deploy_ova(self, ova_path, vm_name, esx_host, datastore, network, vm_password):
        """
        Deploy the OVA in the vcenter ESX

        Args:
            ova_path        (basestring):   the path where the OVA file is present

            vm_name         (basestring):   the name with which the OVA needs to be deployed

            esx_host        (basestring):   the esx host where the VM is to be deployed

            datastore       (basestring):   datastore to store the VM

            network         (basestring):   network to attach the VM to

            vm_password     (basestring):   the password of the new VM

        Returns:

        """
        try:
            _ps_path = os.path.join(self.utils_path, "DeployOVA.ps1")
            prop_dict = {
                "server_name": self.server_host_name,
                "user": self.user_name,
                "pwd": self.password,
                "ova_path": ova_path,
                "vm_name": vm_name,
                "esx_host": esx_host,
                "datastore": datastore,
                "vm_network": network,
                "vm_pwd": vm_password
            }
            output = self.controller._execute_script(_ps_path, prop_dict)
            exception_list = ["Exception", "Cannot", "login"]
            if any(exp in output.formatted_output for exp in exception_list):
                raise Exception(output.formatted_output)
        except Exception as exp:
            self.log.exception("Something went wrong while deploying OVA. Please check logs.")
            raise Exception(exp)

    def find_vm(self, vm_name):
        """
            Finds the vm and returns its status, ESX and Datastore

            Args:
                vm_name             (str): Name of the VM to be searched

            Returns:
                vm_detail           (list): VM found status, ESX and DS

            Raises:
                Exception:
                    Raise exception when failed to get the status of the vm
            """
        try:
            get_all_vms = self.get_content([self.vim.VirtualMachine])
            vms = dict(filter(lambda elem: vm_name == elem[1], get_all_vms.items()))
            if len(vms) == 0:
                return False, None, None
            if len(vms) == 1:
                for vm in vms:
                    _ds = vm.config.files.logDirectory
                    _ds = _ds[_ds.find("[") + 1:_ds.find("]")]
                    return True, vm.runtime.host.name, _ds
            return 'Multiple', None, None

        except Exception as err:
            self.log.exception("exception raised in finding the vm details")
            raise err

    def find_esx_parent(self, vm_host):
        """
            Finds the parent of the ESX host

            Args:
               vm_host             (str): ESX host to be checked for

            Returns:
               vm_host_detail      (list): ParentId and Parent name

            Raises:
               Exception:
                   Raise exception when failed to get the parent of the vm
            """
        try:
            vm_host_detail = []
            all_esx = self.get_content([self.vim.HostSystem])
            esx_dict = dict(filter(lambda elem: vm_host == elem[1], all_esx.items()))
            for host in esx_dict.keys():
                vm_host_detail = [host.parent._moId, host.parent.name]
            return vm_host_detail
        except Exception as err:
            self.log.exception("exception raised in finding the vm details: {0}".format(err))
            raise err

    def wait_for_tasks(self, tasks):
        """
        Waits till a task if completed
        Args:
            tasks                   (object):   Task to be monitored

        """

        property_collector = self.connection.content.propertyCollector
        task_list = [str(task) for task in tasks]
        obj_specs = [self.vmodl.query.PropertyCollector.ObjectSpec(obj=task)
                     for task in tasks]
        property_spec = self.vmodl.query.PropertyCollector.PropertySpec(type=self.vim.Task,
                                                                        pathSet=[],
                                                                        all=True)
        filter_spec = self.vmodl.query.PropertyCollector.FilterSpec()
        filter_spec.objectSet = obj_specs
        filter_spec.propSet = [property_spec]
        pc_filter = property_collector.CreateFilter(filter_spec, True)
        try:
            version, state = None, None
            while len(task_list):
                update = property_collector.WaitForUpdates(version)
                for filter_set in update.filterSet:
                    for obj_set in filter_set.objectSet:
                        task = obj_set.obj
                        for change in obj_set.changeSet:
                            if change.name == 'info':
                                state = change.val.state
                            elif change.name == 'info.state':
                                state = change.val
                            else:
                                continue
                            if not str(task) in task_list:
                                continue
                            if state == self.vim.TaskInfo.State.success:
                                task_list.remove(str(task))
                            elif state == self.vim.TaskInfo.State.error:
                                raise task.info.error
                version = update.version
        finally:
            if pc_filter:
                pc_filter.Destroy()

    def data_store_for_attach_disk_restore(self, host):
        """
        Find the datastore to be used for attach disk restore
        Args:
            host                (string):   ESX where datastore has to be searched

        Returns:
            datastore_name      (string):   Datastore name where restore will happen

        Raises:
               Exception:
                   Raise exception when failed to get the DS

        """
        try:
            datastore_priority_dict = self._get_datastore_dict()
            for each_datastore in datastore_priority_dict.items():
                datastore_name = each_datastore[0]
                tree = self._get_datastore_tree_list(datastore_name)
                if tree['ESX'] == host:
                    self.log.info("Disk is restoring to datastore: {}".format(datastore_name))
                    return datastore_name
            self.log.exception("No Datastore found")
        except Exception as err:
            self.log.exception("exception raised in finding DS: {0}".format(err))
            raise err

    def power_off_all_vms(self):
        """

        Power off all VMs in VMs dictionary

        Raise:
            Exception:
                Raise Exception when fails to power off

        """
        try:
            for vm_name in self.VMs:
                self.VMs[vm_name].power_off()
        except Exception as exp:
            self.log.exception(f"Exception raised in powering off VMs: {exp}")
            raise exp

    def power_on_all_vms(self):
        """

        Power on all VMs in VMs dictionary

        Raise:
            Exception:
                Raise Exception when fails to power on

        """
        try:
            for vm_name in self.VMs:
                self.VMs[vm_name].power_on()
        except Exception as exp:
            self.log.exception(f"Exception raised in powering off VMs: {exp}")
            raise exp

    def check_vms_exist(self, vm_list):
        """

        Check each VM in vm_list exists in Hypervisor VMs Dict

        Args:
            vm_list (list): List of VMs to check

        Returns:
            True (bool): If All VMs are present

            False (bool): If any VM is absent

        """
        present_vms = self.get_all_vms_in_hypervisor()
        present_vms = set(present_vms)
        if (set(vm_list) & set(present_vms)) == set(vm_list):
            return True
        else:
            return False

    def check_vms_absence(self, vm_list):
        """

        Check VMs in vm_list should not be present in Hypervisor VMs Dict

        Args:

            vm_list (list): If all VMs are not present

        Returns:
            True (bool): If All VMs are absent

            False (bool): If any VM is present

        """
        present_vms = self.get_all_vms_in_hypervisor()
        present_vms = set(present_vms)
        if (set(vm_list) & set(present_vms)) == set():
            return True
        else:
            return False

    def check_ds_exist(self, ds_list):
        """

        Check datastrores if present in the vcenter

        Args:

            ds_list                 (list): List of datastores to be checked in the vcenter

        Returns:
            True (bool): If any of the DS is present in the vcenter

            False (bool): If all DS are not present

        """
        try:
            all_ds = self.get_content([self.vim.Datastore])
            if dict(filter(lambda elem: elem[1] in ds_list, all_ds.items())):
                return True
            return False
        except Exception as err:
            self.log.exception("An error occurred in checking DS list")
            raise err
