# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that does all operations on OCI """

import os
import oci
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor


class OciHelper(Hypervisor):

    def __init__(self, server_host_name,
                 user_name,
                 password,
                 instance_type,
                 commcell,
                 host_machine,
                 **kwargs):

        super(OciHelper, self).__init__(server_host_name, user_name, password,
                                        instance_type, commcell, host_machine)

        self.oci_dict = user_name
        self.server_host_name = server_host_name
        self.oci_vm_ip = None
        self.oci_vm_guid = None
        self.oci_vm_vmsize = None
        self.oci_vm_vcn = None
        self.vm_details_dict = dict()
        self.instance_type = instance_type
        self.oci = oci
        self.oci_config = {
            "user": self.oci_dict['oci_user_id'],
            "key_file": self.oci_dict['oci_private_file_name'],
            "fingerprint": self.oci_dict['oci_finger_print'],
            "tenancy": self.oci_dict['oci_tenancy_id'],
            "region": self.oci_dict['oci_region_name'],
            "pass_phrase": self.oci_dict['oci_private_key_password']
        }
        config_validator = self.oci.config.validate_config(self.oci_config)
        self.ComputeClient = self.oci.core.ComputeClient(self.oci_config)
        self.NetworkClient = self.oci.core.VirtualNetworkClient(self.oci_config)
        if config_validator not in (None, "None"):
            raise Exception
        self.identity = self.oci.identity.IdentityClient(self.oci_config)
        self.user = self.identity.get_user(self.oci_config["user"]).data
        self.root_compartment_id = self.user.compartment_id

    def get_vm_property(self, vm_name):
        """
        This method can get all properties of a particular vm
        as of now it gets all info and then filters out needed one.
        needs more research to figureo ut a way to get a particular vm info without
        getting all vm info

        :param vm_name: (str)   Name of the vm
        :return:        (dict)  Instance properties | False if no VM by that name
        """
        detailed_dict = self.get_all_compartments_detailed()
        for compartment in detailed_dict:
            for instance in detailed_dict[compartment]:
                if str(instance['name']).lower() == vm_name.lower():
                    self.log.info('Collected information for VM: %s' % vm_name)
                    return instance
        self.log.error('Could not find a VM by name: %s' % vm_name)
        return False

    def get_all_compartments_detailed(self):
        """
        method to get all information on all compartments and instances on the OCI

        :return:    (dict)  Dict format for all compartments and instances
        """

        op_dict = dict()
        response = self.oci.pagination.list_call_get_all_results(self.identity.list_compartments,
                                                                 self.root_compartment_id,
                                                                 compartment_id_in_subtree=True)
        compartments = response.data
        for compartment in compartments:
            compartment_name = compartment.name
            if compartment.lifecycle_state == compartment.LIFECYCLE_STATE_ACTIVE:

                compartment_id = compartment.id
                try:
                    response = \
                        self.oci.pagination.list_call_get_all_results(
                            self.ComputeClient.list_instances,
                            compartment_id=compartment_id)

                    if len(response.data) > 0:
                        out_p = self.cleanup_response(response.data, compartment_name, "Compute",
                                                      compartment_id)
                        if out_p != []:
                            op_dict[compartment_name] = out_p
                except Exception as err:
                    self.log.error('Exception occurred while getting compartment details')
                    self.log.error(str(err))

        return op_dict

    def cleanup_response(self, instances, compartment_name, instance_type,
                         parent_compartment_id):
        """
        method used by get_all_compartments_detailed to sort through the information
        received from oci

        :return:    (list) list of dictionary of all instances under a particular compartment
        """
        sub_op_list = []
        NoValueString = "n/a"  # what value should be used when no data is available
        for instance in instances:
            good_states = [instance.LIFECYCLE_STATE_RUNNING, instance.LIFECYCLE_STATE_STARTING,
                           instance.LIFECYCLE_STATE_STOPPED, instance.LIFECYCLE_STATE_STOPPING]
            if instance.lifecycle_state in good_states:
                sub_op = dict()
                sub_op['compartment_id'] = instance.compartment_id
                sub_op['instance_id'] = instance.id
                sub_op['name'] = instance.display_name
                sub_op['guid'] = instance.id
                sub_op['id'] = instance.id
                sub_op['power_state'] = instance.lifecycle_state
                sub_op['volumes'] = dict()
                sub_op['network'] = dict()
                sub_op['parent_compartment_id'] = parent_compartment_id
                sub_op['parent_compartment_name'] = compartment_name

                # get vcn details
                sub_op['vcn'] = dict()
                vcn_out = self.NetworkClient.list_vcns(instance.compartment_id).data
                for vcn in vcn_out:
                    if vcn.lifecycle_state == vcn.LIFECYCLE_STATE_AVAILABLE:
                        sub_op['vcn'][vcn.display_name] = {'display_name': vcn.display_name,
                                                           'network_id': vcn.id,
                                                           'vcn_domain_name': vcn.vcn_domain_name,
                                                           'dns_label': vcn.dns_label
                                                           }
                        sub_op['sourceNetwork'] = vcn.display_name
                        sub_op['networkName'] = vcn.display_name
                        sub_op['networkDisplayName'] = vcn.id
                        # sub_op['name'] = vcn.id

                # Handle details for Compute Instances
                if instance_type == "Compute":
                    # sub_op['OCPU'], sub_op['Memory'], sub_op['SSD'] =
                    # shapes.ComputeShape(instance.shape)
                    sub_op['vmSize'] = instance._shape
                    sub_op['instanceSize'] = instance._shape
                    sub_op['Datastore'] = instance.availability_domain

                    # get volume data
                    vol_op = self.ComputeClient.list_volume_attachments(instance.compartment_id)
                    # boot_vol = self.ComputeClient.list_boot_volume_attachments(instance.id)
                    for item in vol_op.data:
                        if item.lifecycle_state == item.LIFECYCLE_STATE_ATTACHED:
                            sub_op['volumes'][item.display_name] = item.volume_id

                    # get vnic
                    response = self.ComputeClient.list_vnic_attachments(
                        compartment_id=instance.compartment_id, instance_id=instance.id)
                    vnics = response.data
                    for vnic in vnics:
                        try:
                            responsenic = self.NetworkClient.get_vnic(vnic_id=vnic.vnic_id)
                            nicinfo = responsenic.data
                            sub_op['network'][nicinfo.id] = {'Public IP': nicinfo.public_ip,
                                                             'Private IP': nicinfo.private_ip,
                                                             'subnet_id': nicinfo.subnet_id,
                                                             'nic_mac': nicinfo.mac_address,
                                                             'nic_is_primary': nicinfo.is_primary,
                                                             'vnic_id': nicinfo.id
                                                             }

                            for key, val in sub_op['vcn'].items():
                                val['subnet_id'] = nicinfo.subnet_id
                            if nicinfo.is_primary in (True, 'True'):
                                self.oci_vm_ip = nicinfo.public_ip
                                sub_op['source_ip'] = nicinfo.public_ip
                                sub_op['ip'] = nicinfo.public_ip
                                sub_op['subnetId'] = nicinfo.subnet_id

                                sub_op['sourceNetworkId'] = nicinfo.subnet_id

                                sub_op['destinationNetwork'] = nicinfo.subnet_id

                        except Exception as err:
                            self.log.warning(str(err))
                            self.log.warning('Unable to get any network information for %s '
                                             % instance.display_name)

                    # Get OS Details
                    try:
                        response = self.ComputeClient.get_image(instance.source_details.image_id)
                        imagedetails = response.data
                        sub_op['OS'] = imagedetails.display_name
                    except:
                        sub_op['OS'] = NoValueString

                    prefix, AD = instance.availability_domain.split(":")
                    LicenseIncluded = "BYOL"

                sub_op_list.append(sub_op)
        return sub_op_list

    def get_vm_details(self, vm_name):
        """
        Gets the details of the VM based by its name
        :param vm_name: Name of the VM
        :return: Dict of the VM details
        """
        detailed_dict = self.get_all_compartments_detailed()
        for compartment in detailed_dict:
            for instance in detailed_dict[compartment]:
                if vm_name == str(instance['name']):
                    self.vm_details_dict = instance
                    return self.vm_details_dict

    def get_vm_ip(self, vm_name):
        """
        gets public IP of a VM by name

        :param vm_name: Name of the VM
        :return:
        """
        detailed_dict = self.get_all_compartments_detailed()
        for compartment in detailed_dict:
            for instance in detailed_dict[compartment]:
                if vm_name == str(instance['name']):
                    try:
                        for key, value in instance['network'].items():
                            if value['Public IP']:
                                return value['Public IP']
                    except Exception as err:
                        self.log.error(
                            'Could not find network information for the client: %s' % vm_name)
                        self.log.error(str(err))

    def get_vm_ids(self, vm_name):
        """
        This method can be used to get an instance id by its name
        :param vm_name: (str)   Name of the VM
        :return:        (list)   Reutns the ID of the VM
        """
        vm_list = []
        detailed_dict = self.get_all_compartments_detailed()
        for compartment in detailed_dict:
            for instance in detailed_dict[compartment]:
                if vm_name == str(instance['name']):
                    vm_list.append(instance['instance_id'])
        return vm_list

    def terminate_vm(self, vm_id):

        self.ComputeClient.terminate_instance(vm_id)
        self.oci.wait_until(
            self.ComputeClient,
            self.ComputeClient.get_instance(vm_id),
            'lifecycle_state',
            'TERMINATED',
            succeed_on_not_found=True
        )

    def get_all_vms_in_hypervisor(self):
        """
        Gets list of ALL names of the VM's in the OCI that user has access to
        :return:    (list)  Names of all vms in list format
        """
        vm_list = []
        detailed_dict = self.get_all_compartments_detailed()
        for compartment in detailed_dict:
            for instance in detailed_dict[compartment]:
                vm_list.append(str(instance['name']))
        return vm_list

    def update_hosts(self):
        """
        Update the VM data Information

        Raises:
            Exception:
                Failed to fetch information from cloud portal
        """
        try:
            self.get_all_compartments_detailed()

        except Exception as err:
            self.log.exception("An exception occurred in updating Host")
            raise err

    def copy_test_data_to_each_volume(self, vm_name, _drive, backup_folder, _test_data_path):
        """
        copy testdata to each volume in the vm provided


        Args:
                vm_name    (str)        - vm to which test data has to be copied

                _test_data_path(str) - path where testdata needs to be generated

                backup_folder(str)  - name of the folder to be backed up

        Exception:

                if fails to generate testdata

                if fails to copy testdata

        """

        try:

            self.log.info("creating test data directory %s" % _test_data_path)

            # initializing prerequisites
            _failed_file_list = []
            self.timestamp = _test_data_path.split("\\")[-1]
            # create Base dir
            _dest_base_path = self.VMs[vm_name].machine.join_path(_drive, backup_folder, "TestData", self.timestamp)
            if not self.VMs[vm_name].machine.check_directory_exists(_dest_base_path):
                _create_dir = self.VMs[vm_name].machine.create_directory(_dest_base_path)
                if not _create_dir:
                    _failed_file_list.append(_dest_base_path)

            self.log.info("Copying testdata to volume {0}".format(_drive))
            if self.machine.is_local_machine:
                self.VMs[vm_name].machine.copy_from_local(_test_data_path,
                                                          _dest_base_path)
            else:
                if self.VMs[vm_name].GuestOS == "Windows":
                    _dest_base_path = os.path.splitdrive(_dest_base_path)
                    network_path = "\\\\" + vm_name + "\\" + _dest_base_path[0].replace(
                        ":", "$") + _dest_base_path[-1]
                    self.VMs[vm_name].machine.copy_folder_to_network_share(
                        _test_data_path, network_path,
                        vm_name + "\\" +
                        self.VMs[vm_name].machine.username,
                        self.VMs[vm_name].machine.password)
        except Exception as err:
            self.log.exception(
                "An error occurred in  Copying test data to Vm  ")
            raise err
