# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that does all operations on Azure
    AzureHelper:

        get_access_token()		                - Get access token for any first authorization

        check_for_access_token_expiration()     - this function check if access_token is
                                                  expired if yes it will generate one

        copy_test_data_to_each_volume           - copy testdata to each volume in the vm

        update_hosts                            - Update the VM data Information

        collect_all_vm_data                     - Collect all VM Data

        collect_all_resource_group_data         - Collect All RG Info

        get_all_resource_group                  - get resource group info

        get_resourcegroup_name                  - gets the resource group of that VM

        get_resourcegroup_for_region            - get the Resource group for that particular
                                                  region

        compute_free_resources                  - compute all free resources required

         get_storage_access_token               - Gets storage access token

         check_for_storage_access_token_expiration - Checks if storage  access token has expired

"""

import requests
import datetime
import time
import adal
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from VirtualServer.VSAUtils.VirtualServerUtils import get_details_from_config_file


class AzureHelper(Hypervisor):
    """
        Main class for performing all operations on AzureRM Hypervisor
    """

    def __init__(self,
                 server_host_name,
                 user_name,
                 password,
                 instance_type,
                 commcell,
                 host_machine,
                 **kwargs):
        """
        Initialize AzureRM Helper class properties


            Args:
                server_host_name    (str):      server list at instance level

                host_machine        (str):      Co-ordinator at instance level

                user_name           (str):      App_ID of AzureRM subscription

                password            (tupple):   consist of password, subscriptionID,
                                                tenantID

                instance_type       (str):      Instance type of the AzureRM

                commcell            (object):   Commcell object

        """

        super(AzureHelper, self).__init__(server_host_name,
                                          user_name,
                                          password,
                                          instance_type,
                                          commcell,
                                          host_machine)

        self.disk_extension = [".vhd"]
        self.authentication_endpoint = 'https://login.microsoftonline.com/'
        self.azure_resourceURL = 'https://management.core.windows.net/'
        self.azure_baseURL = 'https://management.azure.com'
        self.azure_apiversion = "api-version="
        self._all_vmdata = {}
        self._all_rgdata = {}
        self.subscription_id = password[1]
        self.app_id = user_name
        self.tenant_id = password[2]
        self.app_password = password[0]
        self.azure_session = requests.Session()
        if instance_type == 'azure resource manager':
            self.get_access_token()
            self.collect_all_resource_group_data()
            self.collect_all_vm_data()

    @property
    def default_headers(self):
        """
        provide the default headers required

        """
        self._default_headers = {"Content-Type": "application/json",
                                 "Authorization": "Bearer %s" % self.access_token}
        return self._default_headers

    @property
    def all_vmdata(self):
        """
        provide info about all VM's
        """
        if self._all_vmdata is None:
            self.collect_all_vm_data()

        return self._all_vmdata

    @all_vmdata.setter
    def all_vmdata(self, value):
        """
        sets the VM related info
        """
        key, value = value
        self._all_vmdata[key] = value

    @property
    def all_rgdata(self):
        """
        collects all resource group data

        """
        if self._all_rgdata is None:
            self.collect_all_resource_group_data()

        return self._all_rgdata

    @all_rgdata.setter
    def all_rgdata(self, value):
        """
        sets the resource group data

        """
        self._all_rgdata = value

    def get_storage_access_token(self):
        """
        Get Storage access token


        Returns: (str) storage access token

        """
        try:
            if self.app_password:
                context = adal.AuthenticationContext(self.authentication_endpoint + self.tenant_id)
                token_response = context.acquire_token_with_client_credentials('https://storage.azure.com/',
                                                                               self.app_id,
                                                                               self.app_password)
                self.storage_access_token = token_response.get('accessToken')
                self.strtok_expire_time = token_response.get('expiresOn')
                return self.storage_access_token

        except Exception as err:
            self.log.info("An exception occurred in getting the storage Access token")

    def check_for_storage_access_token_expiration(self):
        """Check if the storage access is expired and return a valid storage access token
        """
        try:
            expiry_time = datetime.datetime.strptime(self.strtok_expire_time.split('.')[0],
                                                     '%Y-%m-%d %H:%M:%S')
            curr_loc_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
            curr_time = datetime.datetime.strptime(curr_loc_time, '%Y-%m-%d %H:%M:%S')
            if expiry_time > curr_time and (expiry_time - curr_time).seconds > 300:
                return self.storage_access_token
            return self.get_storage_access_token()
        except Exception:
            self.log.info("Error while checking storage access token validity")

    def get_access_token(self):
        """
        Get access token for any first authorization

        Raises:

            Exception:
                Failed to get access token while logging into Azure
        """
        try:
            self.log.info("Logging into Azure to get access token")
            if self.app_password == '':
                keys = get_details_from_config_file('azure')
                self.tenant_id = keys.split(":")[0]
                self.app_id = keys.split(":")[1]
                self.app_password = keys.split(":")[2]
            context = adal.AuthenticationContext(self.authentication_endpoint + self.tenant_id)
            token_response = context.acquire_token_with_client_credentials(self.azure_resourceURL,
                                                                           self.app_id,
                                                                           self.app_password)
            self.access_token = token_response.get('accessToken')
            self.log.info("Access Token is %s" % self.access_token)

        except Exception as err:
            self.log.exception("An exception occurred in getting the Access token")
            raise err

    def check_for_access_token_expiration(self):
        """
        This function check if access_token is expired if yes it will generate one

        Raises:

            Exception:
                Failed to get access token
        """
        try:

            is_sessionok = False
            count = 0
            while ((not is_sessionok) and (count < 3)):
                azure_list_subscriptions_url = self.azure_baseURL + "/subscriptions?" \
                                               + self.azure_apiversion + "2017-05-10"
                subscriptions = self.azure_session.get(azure_list_subscriptions_url,
                                                       headers=self.default_headers, verify=False)
                data = subscriptions.json()
                if subscriptions.status_code == 401:
                    if data["error"]["code"] == "ExpiredAuthenticationToken":
                        count = count + 1
                        self.get_access_token()
                    else:
                        self.log.info("The session is not expired , so there are credential issue")

                elif subscriptions.status_code == 200:
                    is_sessionok = True

                else:
                    self.log.info("There was error even after getting new token")
                    count = count + 1

        except Exception as err:
            self.log.exception("An exception occurred in getting the Access token")
            raise err

    def update_hosts(self):
        """
        Update the VM data Information

        Raises:
            Exception:
                Failed to fetch information from cloud portal
        """
        try:
            self.collect_all_vm_data()
            self.collect_all_resource_group_data()

        except Exception as err:
            self.log.exception("An exception occurred in updating Host")
            raise err

    def collect_all_vm_data(self):
        """
        Collect all VM Data from each resource group

         Raises:
            Exception:
                Failed to get VM information present in Resource group
        """
        try:
            _allrg = self.get_all_resource_group()
            if self.instance_type == 'azure resource manager':
                api_date = "2016-04-30-preview"
            else:
                api_date = "2015-06-15"
            for each_rg in _allrg:
                azure_list_vm_url = self.azure_baseURL + "/subscriptions/" + \
                                    self.subscription_id + "/resourceGroups/" \
                                    + each_rg + "/providers/Microsoft.Compute/virtualMachines?" \
                                    + self.azure_apiversion + api_date
                data = self.azure_session.get(azure_list_vm_url, headers=self.default_headers, verify=False)
                _all_data = data.json()
                self.all_vmdata = (each_rg, _all_data)

        except Exception as err:
            self.log.exception("An exception occurred in collect_all_vmdata")
            raise err

    def collect_all_resource_group_data(self):
        """
        Collect All Resource group data

        Raises:
            Exception:
                Failed to get information about resource group
        """
        try:
            if self.instance_type == 'azure resource manager':
                api_date = "2014-04-01"
            else:
                api_date = "2018-02-01"

            azure_resource_group_url = self.azure_baseURL + "/subscriptions/" \
                                       + self.subscription_id + "/resourceGroups?" \
                                       + self.azure_apiversion + api_date
            data = self.azure_session.get(azure_resource_group_url, headers=self.default_headers, verify=False)
            self.all_rgdata = data.json()

        except Exception as err:
            self.log.exception("An exception occurred in CollectAllResourceGroupData")
            raise err

    def get_all_resource_group(self):
        """
        Prepare list of resource groups and their corressponding resources

        Returns:
            resource_group     (str):  list of all resource groups

        Raises:
            Exception:
                Failed to sagregate resources groups and its data
        """

        try:
            _allrg_list = []
            if self.check_for_token_expiry(self.all_rgdata):
                self.log.info("Token Is expired, Renewing it")
                self.get_access_token()
                self.collect_all_resource_group_data()
            datadict = self.all_rgdata
            for eachkey in datadict["value"]:
                rg_name = eachkey["name"]
                _allrg_list.append(rg_name)

            return _allrg_list

        except Exception as err:
            self.log.exception("An exception occurred in collect_all_resource_group_data")
            raise err

    def get_all_vms_in_hypervisor(self):
        """
        This function fetches information about all VMs in particular resource group

        Returns:
            _all_vmlist     (list):  list of all VM's

        Raises:
            Exception:
                Failed to get VM information
        """
        try:

            _all_vmlist = []
            datadict = self.all_vmdata
            for eachdata, eachvalue in datadict.items():
                vm_info_value = eachvalue["value"]
                if vm_info_value != []:
                    for eachkey in vm_info_value:
                        vm_name = eachkey["name"]
                        _all_vmlist.append(vm_name)

            return _all_vmlist

        except Exception as err:
            self.log.exception("An exception occurred in getting the Access token")
            raise err

    def get_resourcegroup_name(self, vm_name):
        """
        Get resource group of particular VM

        Args:
            vm_name                 (str):  vm whose information needs to be fetched

        Returns:
            resource_group_name     (str):  Resource group where VM is found

        Raises:
            Exception:
                if it fails to find particular resource group for VM
        """
        try:

            resource_group_name = None
            if self.check_for_token_expiry(self.all_vmdata[next(iter(self.all_vmdata))]):
                self.collect_all_vm_data()
            datadict = self.all_vmdata
            for eachdata, each_value in datadict.items():
                if "value" in each_value:
                    vm_info_value = each_value["value"]
                    if vm_info_value != []:
                        for each_key in vm_info_value:
                            _vm_name = each_key["name"]
                            if _vm_name == vm_name:
                                vm_id = each_key["id"]
                                temp_str = vm_id.split("/")
                                resource_group_name = temp_str[temp_str.index("resourceGroups") + 1]
                                break

                else:
                    self.log.info("Cannot collect information for this VM")
            return resource_group_name

        except Exception as err:
            self.log.exception("An exception occurred in getting the Resource group")
            raise err

    def get_resourcegroup_for_region(self, region):
        """
        Get all resource groups of particular Region

        Args:
            region             (str):  region whose resource groups needs to be fetched

        Returns:
            resource_group     (str):  list of all resource groups

        Raises:
            Exception:
                if it fails to find resource groups of particular region
        """
        try:
            resource_group = []
            if not self.check_for_token_expiry(self.all_rgdata):
                self.collect_all_resource_group_data()
            rg_data = self.all_rgdata
            for eachrg in rg_data["value"]:
                rg_region = eachrg["location"]
                if rg_region == region:
                    resource_group.append(eachrg["name"])

            return resource_group

        except Exception as err:
            self.log.exception("An exception occurred in get_resourcegroup_for_region")
            raise err

    def compute_free_resources(self, vm_name, resource_group=None):
        """

        Compute the free Resource of the subscription based on region

        Args:
            vm_name             (str):  list of Vms to be restored

        Returns:
            resource_group	    (str):  The resource group where restore can be performed

            storage_account     (str):  Storage account where restore has to be performed

        Raises:
            Exception:
                Not able to get resource group or storage account or all

        """
        try:
            sa_name = None
            if resource_group is None:
                datadict = self.all_vmdata
                for eachdata, eachvalue in datadict.items():
                    vm_info_value = eachvalue["value"]
                    if vm_info_value != []:
                        for eachkey in vm_info_value:
                            _vm_name = eachkey["name"]
                            if _vm_name == vm_name[0]:
                                region = eachkey["location"]
                                break

                resource_group = self.get_resourcegroup_for_region(region)

            for each_rg in resource_group:
                storage_account_url = self.azure_baseURL + "/subscriptions/" + \
                                      self.subscription_id + "/resourceGroups/" + \
                                      each_rg + "/providers/Microsoft.Storage/storageAccounts?" \
                                      + self.azure_apiversion + "2017-06-01"
                data = self.azure_session.get(storage_account_url, headers=self.default_headers, verify=False)
                if data.status_code == 200:
                    storage_account_data = data.json()
                    sa_value = storage_account_data["value"]
                    if sa_value != []:
                        for each_sa in storage_account_data["value"]:
                            sa_name = each_sa["name"]
                            resource_group = each_rg
                            break
                    else:
                        self.log.info("Failed to get SA details for this Resource Group")
                else:
                    self.log.info("Failed to get SA details for this Resource Group: %s" % each_rg)
            location = self.get_storage_account_location(resource_group, sa_name)
            return resource_group, sa_name, location

        except Exception as err:
            self.log.exception("An exception occurred in ComputeFreeResources")
            raise err

    def get_storage_account_location(self, resource_group, sa_name):
        """
        Returns the location of the storage account

        Args:
            resource_group      (str):  Resource group os storage account

            sa_name             (str):  Storage Account name for which location has to be identified

        Returns:
            location            (str):  Returns the location of the storage account

        Raises:
            Exception:
                Not able to get location of storage account

        """
        storage_account_url = self.azure_baseURL + "/subscriptions/" + \
                              self.subscription_id + "/resourceGroups/" + \
                              resource_group + "/providers/Microsoft.Storage/storageAccounts/" + sa_name + "?" \
                              + self.azure_apiversion + "2017-06-01"
        data = self.azure_session.get(storage_account_url, headers=self.default_headers, verify=False)
        if data.status_code == 200:
            storage_account_data = data.json()
            location = storage_account_data['location']
            return location
        else:
            self.log.info("Failed to get location details for this Storage Account")

    def check_for_token_expiry(self, data):
        """
        Checks for the token expiry

        Args:
        data (basestring)  : Response  from api
        """
        if data.get('error', {}).get('code', 'noerror') == 'ExpiredAuthenticationToken':
            return True

    def get_storage_account(self, resource_group):
        """
        Returns the storage account name

        Args:
            resource_group (basestring) :  Resource group for which storage accound has to be looked for

        Returns:
            Storage Account for resource group
        """
        sa_value = ''
        storage_account_url = self.azure_baseURL + "/subscriptions/" + \
                              self.subscription_id + "/resourceGroups/" + \
                              resource_group + "/providers/Microsoft.Storage/storageAccounts?" \
                              + self.azure_apiversion + "2017-06-01"
        data = self.azure_session.get(storage_account_url, headers=self.default_headers, verify=False)
        if data.status_code == 200:
            storage_account_data = data.json()
            sa_value = storage_account_data["value"]
        return sa_value
