# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file that does all operations on Amazon """

import boto3
import botocore
import datetime
import time
from botocore.exceptions import ClientError
from dateutil.tz import tzlocal
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from VirtualServer.VSAUtils.VirtualServerUtils import get_details_from_config_file


class AmazonHelper(Hypervisor):
    """
        Main class for performing all operations on AWS
        """

    def __init__(self, server_host_name,
                 user_name,
                 password,
                 instance_type,
                 commcell,
                 host_machine,
                 **kwargs):

        super(AmazonHelper, self).__init__(server_host_name, user_name, password,
                                           instance_type, commcell, host_machine)

        if all(password):
            self.log.info("Using secret key and access key authentication")
            self._aws_access_key = password[0]
            self._aws_secret_key = password[1]
        elif self.password[0]:
            self.log.info("Using STS Assume Role")
            self._sts_role = password[0]
            self._aws_access_key = None
        else:
            self.log.info("Using IAM based authentication")
            keys = get_details_from_config_file('aws_login_creds', 'keys')
            self._aws_access_key = keys.split(":")[0]
            self._aws_secret_key = keys.split(":")[1]

        if kwargs.get('region') is None:
            self._aws_region = 'us-east-1'  # default region
        else:
            self._aws_region = kwargs.get('region')
        self.connection = None
        self.aws_session()

    def aws_session(self):
        """Create the session with aws """

        try:
            if self._aws_access_key:
                self.connection = boto3.Session(aws_access_key_id=self._aws_access_key,
                                                aws_secret_access_key=self._aws_secret_key,
                                                region_name=self.aws_region)
            else:
                self.connection = self.assumed_role_session(self._sts_role)
            self.log.info("Connection successful for AWS")
        except ClientError as err:
            self.log.exception("Unexpected error: %s" % err)
            raise err

    def power_on_proxies(self, proxy_ips):
        """
        power on the proxies

        Returns: None
        Raises:
            Exception:
                Not able to get instances in the the AWS
        """

        try:
            _resource = self.connection.resource('ec2')

            for instance in _resource.instances.all():
                power_state = instance.state['Code']
                if instance.state['Code'] != 48 and instance.tags:
                    if instance.private_ip_address in proxy_ips.values():
                        if power_state in (0, 32, 64):
                            time.sleep(120)
                            instance.start()
                        elif power_state == 80:
                            instance.start()
                        self.proxy_instances.append(instance)
                        # proxy_ips.remove(instance.private_ip_address)
        except Exception as err:
            self.log.exception("An exception occurred while powering on proxies")

    def power_off_proxies(self, proxy_ips):
        """
        power off the proxies

        """

        try:
            _resource = self.connection.resource('ec2')

            for instance in _resource.instances.all():
                if instance.private_ip_address in proxy_ips.values():
                    power_state = instance.state['Code']
                    if instance.state['Code'] != 48:
                        if power_state != 80:
                            instance.stop()
                            time.sleep(180)
                    else:
                        self.log.error("Power Off failed. Instance has been terminated already")

        except Exception as err:
            self.log.exception("An exception occurred while powering off proxies")

    def assumed_role_session(self, role_arn):
        """
        Login via assume role
        Args:
            role_arn                    (string):  assume role

        Returns:
            aws session

        Raises:
            Exception:
                Not able to get login via assume role

        """
        try:

            base_session = boto3.session.Session()._session
            fetcher = botocore.credentials.AssumeRoleCredentialFetcher(
                client_creator=base_session.create_client,
                source_credentials=base_session.get_credentials(),
                role_arn=role_arn
            )
            creds = botocore.credentials.DeferredRefreshableCredentials(
                method='assume-role',
                refresh_using=fetcher.fetch_credentials,
                time_fetcher=lambda: datetime.datetime.now(tzlocal())
            )
            botocore_session = botocore.session.Session()
            botocore_session._credentials = creds
            return boto3.Session(botocore_session=botocore_session, region_name=self.aws_region)
        except Exception as err:
            self.log.exception("Exception during login via assume role: {}".format(err))
            raise err

    def get_all_vms_in_hypervisor(self):
        """
        Get all the ec2 instances in aws

        Returns:
            _all_vm_list    (str):  List of instances in the the AWS

        Raises:
            Exception:
                Not able to get instances in the the AWS
        """
        try:
            _resource = self.connection.resource('ec2')
            _all_vm_list = []
            for instance in _resource.instances.all():
                if instance.state != 48 and instance.tags:
                    for tag in instance.tags:
                        if tag.get('Key') == 'Name':
                            _all_vm_list.append(tag['Value'])
            return _all_vm_list
        except Exception as err:
            self.log.exception("An exception occurred while getting all VMs from AWS")
            raise Exception(err)

    def compute_free_resources(self, proxy_vm):
        """
        If proxy is not in  AWS, only then bucket is needed
        Finds nearest bucket to proxy for restore

        Args:
            proxy_vm    (str)   -   AWS proxy vm

        Returns:
            bucket_id   (str)   -   bucket id

        Raises:
            Exception:
                if it fails to retreive bucket for non aws proxy

        """
        # TODO: this is only for non AWS proxy
        pass

