# -*- coding: utf-8 -*-

"""
This script helps fetch enums defined in Api.x, and generate a standalone
Python file containing two mappings. One is from type code to descrption
text, and the other is from enum to type code.
"""

import os
import sys
import re
import json
import datetime
import argparse
import collections


def setup():
    """sets up the parser for this script"""
    parser = argparse.ArgumentParser()
    parser.add_argument("-e", "--enum", required=True,
                        help="enum to be generated")
    parser.add_argument("-o", "--output", required=True,
                        help="output file path")
    parser.add_argument("-p", "--prefix", required=True,
                        help="prefix of generated definition")
    parser.add_argument("--api", help="path to the Api.x")
    return parser.parse_args()


def generate_top_msg():
    """generates the messages on top of output file"""
    msg_on_top = ""
    msg_on_top += r"# -*- coding: utf-8 -*-" + "\n"
    msg_on_top += r"#" + "\n"
    msg_on_top += r"# This is generated by" + "\n" + r"#     {}.".format(
        r"\vaultcx\Source\tools\Automation\VirtualServer\VSAUtils" +
        r"\GenerateEnumFromDotX.py") + "\n"
    msg_on_top += r"# Generated Time: {}.".format(
        str(datetime.datetime.now())) + "\n\n"

    msg_on_top += "\"\"\"This file provides dicts.\n" + \
        "One is from job phase code to text, and the other is from enum to code.\n\"\"\"\n\n"

    return msg_on_top


def parse_input(api_path, enum_to_parse):
    """parses input from the arguments"""
    api_path = os.path.abspath(api_path)
    sys.stderr.write("Open Api.x in " + api_path)

    input = ""
    with open(api_path, "r") as f:
        searching = False
        for line in f:
            if "enum" in line and enum_to_parse in line:
                searching = True

            if searching:
                input += line.strip()
                if '}' in line:
                    searching = False
                    break

    # extracts the part within "{}"
    input = re.findall(r'{(.*)}', input)[0]
    # separates the input by ','
    input = input.split(",")

    return input


def generate_output_dict(input):
    """generates the output dicts"""
    code_to_text = {}
    enum_to_code = {}

    for i in input:
        # uses regular expression to extract info we need
        regex_result = re.search(r'@Text\(text="(.*)"\)(.*)=[ ]*([0-9]*)', i)
        description_text = regex_result.group(1).strip()
        enum = regex_result.group(2).strip()
        type_code = regex_result.group(3).strip()

        code_to_text[type_code] = description_text
        enum_to_code[enum] = int(type_code)

    # make the `code_to_text` dict ordered by code value
    code_to_text = collections.OrderedDict(
        sorted(code_to_text.items(), key=lambda k: int(k[0])))

    # make the `enum_to_code` dict ordered by code value
    enum_to_code = collections.OrderedDict(
        sorted(enum_to_code.items(), key=lambda k: k[1]))
    return (code_to_text, enum_to_code)


def generate_output_str(prefix, code_to_text, enum_to_code):
    """converts dicts to json strings"""
    prefix = prefix.upper()
    output = ""
    output += generate_top_msg()
    output += prefix + r"_CODE_TO_TEXT = " + \
        json.dumps(code_to_text, indent=4) + "\n\n"
    output += prefix + r"_ENUM_TO_CODE = " + \
        json.dumps(enum_to_code, indent=4) + "\n"

    return output


def main():
    """main function of this script"""
    args = setup()
    enum_to_parse = args.enum

    # removes trailing ".py" (will be added at the end)
    args.output = args.output.replace(".py", "").strip()

    # determines whether api.x path is provided by user
    filepath_to_apix = args.api or (os.path.dirname(os.path.abspath(
        __file__)) + "\\..\\..\\..\\..\\Common\\XmlMessage\\Api.x")

    parsed_input = parse_input(filepath_to_apix, enum_to_parse)
    (code_to_text, enum_to_code) = generate_output_dict(parsed_input)

    output = generate_output_str(
        prefix=args.prefix, code_to_text=code_to_text,
        enum_to_code=enum_to_code)

    output_filename = args.output + ".py"
    with open(output_filename, "w") as f:
        f.write(output)


if __name__ == "__main__":
    main()
