# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# gitlab
# --------------------------------------------------------------------------

"""Template file for performing IntelliSnap operations for Virtual Server Agent

VSASNAPTemplates is the only class defined in this file

VSASNAPTemplates: Template class to perform IntelliSnap operations

VSASNAPTemplates:

    __init__()                   --  initializes VSASNAPTemplates object

    vsasnap_template_v1()        --  Template to Perform Snap backup, backup copy and Full VM Restore
                                     operations for VSA V1 client.
    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Out of place full vm restore from snap.
        4. In place full vm restore from backup copy

    vsasnap_template_v1_guestfile() -- Template to Perform Snapbackup, backup copy and Guest File
                                        Restore operations for VSA V1 client
    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Guest file restore from Snap (live Browse).
        4. Guest file restore from backup copy.

    vsasnap_template_v1_disk() -- Template to Perform Snapbackup, backup copy and Disk
                                        Restore operations for VSA V1 client

        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Disk restore from Snap.
        4. Disk restore from backup copy



    vsasnap_template_v2()        --  Template to Perform Snap backup, backup copy and Restore
                                     operations for VSA V2 client.

    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Out of place full vm restore from snap (parent job)
        4. In place full vm restore from backup copy

    vsasnap_template_v2_fullvm_client()        --  Template to Perform Snap backup, backup copy and Restore
                                     operations for VSA V2 client.

    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. FULL VM In Place restores : v2 Indexing from snap
        4. FULL VM out of Place restores - v2 Indexing from backup copy

    vsasnap_template_v2_guestfile()        --  Template to Perform Snap backup, backup copy and Restore
                                     operations for VSA V2 client.

    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Guest Files restores - v2 Indexing from snap
        4. Guest Files restores - v2 Indexing from backup copy

    vsasnap_template_v2_disk()        --  Template to Perform Snap backup, backup copy and Restore
                                     operations for VSA V2 client.

    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Backup copy of that VM.
        3. Disk restore from Snap.
        4. Disk restore from backup copy


    vsasnap_template_replica_v1()       -- Template to Perform Snap backup with replication,
                                        backup copy and Restore from replica for VSA V1 client
    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Run Aux copy
        3. Run backup copy
        4. Guest file restore from Replica copy
        5. Full VM out of place restore from Replica copy

    vsasnap_template_replica_v2()       -- Template to Perform Snap backup with replication,
                                        backup copy and Restore from replica for VSA V2 client
    Steps:
        1. Snap Backup of VM coming from a particular snap vendor array
        2. Run Aux copy
        3. Run backup copy
        4. Guest file restore from Replica copy
        5. Full VM out of place restore from Replica copy


"""


import time
from VirtualServer.VSAUtils import VirtualServerHelper, OptionsHelper, VirtualServerUtils
from cvpysdk.policy import Policies
from AutomationUtils.idautils import CommonUtils


class VSASNAPTemplates(object):
    """Template class to perform vsa snap operations"""

    def __init__(self):
        """Initializes VSASNAPTemplates object

        Args:

            tcinputs                        -- test case inputs

        Methods:

                vsasnap_template_v1()       --  Template for vsasnap v1 test cases

        """

    def vsasnap_template_v1(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V1 client
        """
        auto_subclient = VirtualServerUtils.subclient_initialize(testcase)
        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)

        vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
        try:
            VirtualServerUtils.decorative_log("Out of place full vm restore from snap")
            vm_restore_options.power_on_after_restore = True
            vm_restore_options.unconditional_overwrite = True
            vm_restore_options.browse_from_snap = True
            auto_subclient.virtual_machine_restore(vm_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        try:
            VirtualServerUtils.decorative_log("In place full vm restore from backup copy")
            vm_restore_options.in_place_overwrite = True
            vm_restore_options.browse_from_backup_copy = True
            auto_subclient.virtual_machine_restore(vm_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)
        return (auto_subclient, backup_options, vm_restore_options, testcase.test_individual_status,
                testcase.test_individual_failure_message)

    def vsasnap_template_v1_guestfile(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V1 client
        """

        auto_subclient = VirtualServerUtils.subclient_initialize(testcase)
        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)

        VirtualServerUtils.decorative_log("Restore from Snap")
        vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
        file_restore_options = OptionsHelper.FileLevelRestoreOptions(auto_subclient)

        try:
            VirtualServerUtils.decorative_log("Guest file restore from Snap")
            file_restore_options.browse_from_snap = True
            if "browse_ma" in testcase.tcinputs:
                file_restore_options.browse_ma = testcase.tcinputs["browse_ma"]
            if "fbr_ma" in testcase.tcinputs:
                file_restore_options.fbr_ma = testcase.tcinputs["fbr_ma"]
            auto_subclient.guest_file_restore(file_restore_options)
        except Exception as exp:
            testcase.log.error("sleeping 12 minutes for cleanup of mounted snaps")
            time.sleep(720)
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        VirtualServerUtils.decorative_log("Restores from backup copy")

        try:
            VirtualServerUtils.decorative_log("Guest file restore from backup copy")
            file_restore_options.browse_from_snap = False
            file_restore_options.browse_from_backup_copy = True
            auto_subclient.guest_file_restore(file_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        return (auto_subclient, backup_options, testcase.test_individual_status,
            testcase.test_individual_failure_message)

    def vsasnap_template_v1_disk(self, testcase):
            """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V1 client
            """
            auto_subclient = VirtualServerUtils.subclient_initialize(testcase)
            VirtualServerUtils.decorative_log("Backup")
            backup_options = OptionsHelper.BackupOptions(auto_subclient)
            _adv = {"create_backup_copy_immediately": True}
            backup_options.advance_options = _adv
            backup_options.backup_method = "SNAP"
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")
            auto_subclient.backup(backup_options)

            disk_restore_options = OptionsHelper.DiskRestoreOptions(auto_subclient)
            try:
                VirtualServerUtils.decorative_log("Disk restore from Snap")
                disk_restore_options.browse_from_snap = True
                auto_subclient.disk_restore(disk_restore_options)
            except Exception as exp:
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

            try:
                VirtualServerUtils.decorative_log("Disk restore from backup copy")
                disk_restore_options.browse_from_backup_copy = True
                auto_subclient.disk_restore(disk_restore_options)
            except Exception as exp:
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

            return (auto_subclient, backup_options, testcase.test_individual_status, testcase.test_individual_failure_message)

    def vsasnap_template_v2(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V2 client
        """

        auto_subclient = backup_options = None
        testcase.log.info("Started executing %s testcase", testcase.id)
        VirtualServerUtils.decorative_log("Initialize helper objects")
        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,
                                                              testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)

        try:
            VirtualServerUtils.decorative_log("FULL VM out of Place restores from snap")
            vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
            vm_restore_options.power_on_after_restore = True
            vm_restore_options.unconditional_overwrite = True
            vm_restore_options.browse_from_snap = True
            if "SourceIP" and "DestinationIP" in testcase.tcinputs:
                vm_restore_options.source_ip = testcase.tcinputs["SourceIP"]
                vm_restore_options.destination_ip = testcase.tcinputs["DestinationIP"]
            auto_subclient.virtual_machine_restore(vm_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        try:
            VirtualServerUtils.decorative_log("In place full vm restore from backup copy")
            vm_restore_options.in_place_overwrite = True
            vm_restore_options.browse_from_backup_copy = True
            auto_subclient.virtual_machine_restore(vm_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)
        return (auto_subclient, backup_options, vm_restore_options, testcase.test_individual_status,
                testcase.test_individual_failure_message)

    def vsasnap_template_v2_fullvm_client(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V2 client
        """

        auto_subclient = backup_options = None
        testcase.log.info("Started executing %s testcase", testcase.id)
        VirtualServerUtils.decorative_log("Initialize helper objects")
        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,
                                                              testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)
        vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
        vm_restore_options.power_on_after_restore = True
        vm_restore_options.unconditional_overwrite = True

        try:
            VirtualServerUtils.decorative_log("FULL VM out of Place restores"
                                              " - v2 Indexing from backup copy")
            vm_restore_options.browse_from_snap = False
            vm_restore_options.browse_from_backup_copy = True
            vm_restore_options.unconditional_overwrite = True
            for vm in auto_subclient.vm_list:
                auto_subclient.virtual_machine_restore(vm_restore_options, discovered_client=vm)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        try:
            VirtualServerUtils.decorative_log("FULL VM In Place restores : "
                                              "v2 Indexing from snap")
            vm_restore_options.browse_from_backup_copy = False
            vm_restore_options.browse_from_snap = True
            vm_restore_options.in_place_overwrite = True
            for vm in auto_subclient.vm_list:
                auto_subclient.virtual_machine_restore(vm_restore_options, discovered_client=vm)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)


        return (auto_subclient, backup_options, vm_restore_options, testcase.test_individual_status, testcase.test_individual_failure_message)

    def vsasnap_template_v2_guestfile(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V2 client
        """

        auto_subclient = backup_options = None
        testcase.log.info("Started executing %s testcase", testcase.id)
        VirtualServerUtils.decorative_log("Initialize helper objects")
        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,
                                                              testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)

        try:
            VirtualServerUtils.decorative_log("Files restores - v2 Indexing from snap")
            file_restore_options = OptionsHelper.FileLevelRestoreOptions(auto_subclient)
            VirtualServerUtils.set_inputs(testcase.tcinputs, file_restore_options)
            file_restore_options.browse_from_backup_copy = False
            file_restore_options.browse_from_snap = True
            for vm in auto_subclient.vm_list:
                auto_subclient.guest_file_restore(file_restore_options, discovered_client=vm)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        try:
            VirtualServerUtils.decorative_log("Files restores - v2 Indexing from backup copy")
            file_restore_options = OptionsHelper.FileLevelRestoreOptions(auto_subclient)
            VirtualServerUtils.set_inputs(testcase.tcinputs, file_restore_options)
            file_restore_options.browse_from_snap = False
            file_restore_options.browse_from_backup_copy = True
            for vm in auto_subclient.vm_list:
                auto_subclient.guest_file_restore(file_restore_options, discovered_client=vm)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        return (auto_subclient, backup_options, testcase.test_individual_status, testcase.test_individual_failure_message)

    def vsasnap_template_v2_disk(self, testcase):
        """ Template to Perform Snap backup, backup copy and Restore opertaions for VSA V2 client
        """

        auto_subclient = backup_options = None
        testcase.log.info("Started executing %s testcase", testcase.id)
        VirtualServerUtils.decorative_log("Initialize helper objects")
        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,
                                                              testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        _adv = {"create_backup_copy_immediately": True}
        backup_options.advance_options = _adv
        backup_options.backup_method = "SNAP"
        if backup_options.collect_metadata:
            raise Exception("Metadata collection is enabled")
        auto_subclient.backup(backup_options)

        try:
            VirtualServerUtils.decorative_log("Disk restore from Snap")
            disk_restore_options = OptionsHelper.DiskRestoreOptions(auto_subclient)
            disk_restore_options.browse_from_snap = True
            auto_subclient.disk_restore(disk_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        try:
            VirtualServerUtils.decorative_log("Disk restore from backup copy")
            disk_restore_options.browse_from_backup_copy = True
            auto_subclient.disk_restore(disk_restore_options)
        except Exception as exp:
            testcase.test_individual_status = False
            testcase.test_individual_failure_message = str(exp)

        return (auto_subclient, backup_options, testcase.test_individual_status,
                testcase.test_individual_failure_message)

    def vsasnap_template_replica_v1(self, testcase):
        """ Template to Perform Snap backup with replication, backup copy and Restore from replica for VSA V1 client
        """

        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client, testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)
        common_utils = CommonUtils(auto_commcell.commcell)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        backup_options.backup_method = "SNAP"
        _adv = {"create_backup_copy_immediately": False}
        backup_options.advance_options = _adv
        if backup_options.collect_metadata:
            raise Exception('Metadata collection is enabled')
        auto_subclient.backup(backup_options)

        subclient_storage_policy = Policies(testcase.commcell).storage_policies.get(
            auto_subclient.storage_policy)

        # if is_ocum input is set as true disable Use_scale
        if "is_ocum" in testcase.tcinputs and testcase.tcinputs["is_ocum"] == True:
            _is_ocum = True
            _use_scale = False
        else:
            _is_ocum = False
            _use_scale = True

        VirtualServerUtils.decorative_log('Aux Copy')
        _auxcopy_job = subclient_storage_policy.run_aux_copy(use_scale=_use_scale)
        testcase.log.info(f'Started Aux Copy Job {_auxcopy_job.job_id}')

        if not _auxcopy_job.wait_for_completion():
            raise Exception(
                "Failed to run Aux Copy with error: " +
                str(_auxcopy_job.delay_reason)
            )
        if _auxcopy_job.status != 'Completed':
            raise Exception(
                "Aux Copy Job completed with errors, Reason " +
                str(_auxcopy_job.delay_reason)
            )

        testcase.log.info(f'Aux Copy Job {_auxcopy_job.job_id} successfully completed')

        _secondary_copies = []
        for copy_name, copy_property in subclient_storage_policy.copies.items():
            for _property in copy_property:
                if _property == 'copyPrecedence' and copy_property[_property] not in [1, 2]:
                    _secondary_copies.append(copy_property[_property])
                    if copy_property[_property] == 3:
                        replica_copy = copy_name

        options = {
            'enable_backup_copy': True,
            'source_copy_for_snap_to_tape': replica_copy,
            'enable_snapshot_catalog': False,
            'source_copy_for_snapshot_catalog': replica_copy,
            'is_ocum': _is_ocum
        }

        subclient_storage_policy.update_snapshot_options(**options)

        VirtualServerUtils.decorative_log("Backup Copy from Replica")
        _backup_copy_job = subclient_storage_policy.run_backup_copy()
        testcase.log.info(f'Started Backup Copy Job {_backup_copy_job.job_id}')

        if not _backup_copy_job.wait_for_completion():
            raise Exception(
                "Failed to run Backup Copy with error : " +
                str(_backup_copy_job.delay_reason)
            )
        if _backup_copy_job.status != 'Completed':
            raise Exception(
                "Backup Copy Job completed with errors, Reason " +
                str(_backup_copy_job.delay_reason)
            )

        auto_subclient.backupcopy_job_id = common_utils.get_backup_copy_job_id(auto_subclient.backup_job.job_id)
        testcase.log.info(f'Backup Copy job  {_backup_copy_job} successfully completed')

        for _copy_precedence in _secondary_copies:
            VirtualServerUtils.decorative_log(f'Restore from Replica copy {_copy_precedence}')

            try:
                VirtualServerUtils.decorative_log(f'Guest file restore from Replica {_copy_precedence}')
                file_restore_options = OptionsHelper.FileLevelRestoreOptions(auto_subclient)
                file_restore_options.copy_precedence = _copy_precedence
                file_restore_options.browse_from_snap = True
                VirtualServerUtils.set_inputs(testcase.tcinputs, file_restore_options)
                auto_subclient.guest_file_restore(file_restore_options)
            except Exception as exp:
                testcase.log.error('sleeping 12 minutes for cleanup of mounted snaps')
                time.sleep(720)
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

            try:
                VirtualServerUtils.decorative_log(f'Out of place full vm restore from Replica {_copy_precedence}')
                vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
                vm_restore_options.power_on_after_restore = True
                vm_restore_options.unconditional_overwrite = True
                vm_restore_options.browse_from_snap = True
                vm_restore_options.copy_precedence = _copy_precedence
                auto_subclient.virtual_machine_restore(vm_restore_options)
            except Exception as exp:
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

        return (auto_subclient, backup_options, vm_restore_options, testcase.test_individual_status, testcase.test_individual_failure_message)

    def vsasnap_template_replica_v2(self, testcase):
        """ Template to Perform Snap backup with replication, backup copy and Restore from replica for VSA V2 client
        """

        auto_commcell = VirtualServerHelper.AutoVSACommcell(testcase.commcell, testcase.csdb)
        auto_client = VirtualServerHelper.AutoVSAVSClient(auto_commcell, testcase.client)
        auto_instance = VirtualServerHelper.AutoVSAVSInstance(auto_client,testcase.agent, testcase.instance)
        auto_backupset = VirtualServerHelper.AutoVSABackupset(auto_instance, testcase.backupset)
        auto_subclient = VirtualServerHelper.AutoVSASubclient(auto_backupset, testcase.subclient)
        common_utils = CommonUtils(auto_commcell.commcell)

        VirtualServerUtils.decorative_log("Backup")
        backup_options = OptionsHelper.BackupOptions(auto_subclient)
        backup_options.backup_method = "SNAP"
        _adv = {"create_backup_copy_immediately": False}
        backup_options.advance_options = _adv
        if backup_options.collect_metadata:
            raise Exception('Metadata collection is enabled')
        auto_subclient.backup(backup_options)

        subclient_storage_policy = Policies(testcase.commcell).storage_policies.get(
            auto_subclient.storage_policy)

        # if is_ocum input is set as true disable Use_scale
        if "is_ocum" in testcase.tcinputs and testcase.tcinputs["is_ocum"] == True:
            _is_ocum = True
            _use_scale = False
        else:
            _is_ocum = False
            _use_scale = True

        VirtualServerUtils.decorative_log('Aux Copy')
        _auxcopy_job = subclient_storage_policy.run_aux_copy(use_scale=_use_scale)
        testcase.log.info(f'Started Aux Copy Job {_auxcopy_job.job_id}')
        if not _auxcopy_job.wait_for_completion():
            raise Exception(
                "Failed to run Aux Copy with error: " +
                str(_auxcopy_job.delay_reason)
            )
        if _auxcopy_job.status != 'Completed':
            raise Exception(
                "Aux Copy Job completed with errors, Reason " +
                str(_auxcopy_job.delay_reason)
            )

        testcase.log.info(f'Aux Copy Job {_auxcopy_job.job_id} completed successfully')

        _secondary_copies = []
        for copy_name, copy_property in subclient_storage_policy.copies.items():
            for _property in copy_property:
                if _property == 'copyPrecedence' and copy_property[_property] not in [1, 2]:
                    _secondary_copies.append(copy_property[_property])
                    if copy_property[_property] == 3:
                        replica_copy = copy_name

        options = {
            'enable_backup_copy': True,
            'source_copy_for_snap_to_tape': replica_copy,
            'enable_snapshot_catalog': False,
            'source_copy_for_snapshot_catalog': replica_copy,
            'is_ocum': _is_ocum
        }

        subclient_storage_policy.update_snapshot_options(**options)

        VirtualServerUtils.decorative_log("Backup Copy from Replica")
        _backup_copy_job = subclient_storage_policy.run_backup_copy()
        testcase.log.info(f'Started Backup Copy Job {_backup_copy_job.job_id}')
        if not _backup_copy_job.wait_for_completion():
            raise Exception(
                "Failed to run Backup Copy with error : " +
                str(_backup_copy_job.delay_reason)
            )
        if _backup_copy_job.status != 'Completed':
            raise Exception(
                "Backup Copy Job completed with errors, Reason " +
                str(_backup_copy_job.delay_reason)
            )

        auto_subclient.backupcopy_job_id = common_utils.get_backup_copy_job_id(auto_subclient.backup_job.job_id)
        testcase.log.info(f'Backpup copy  {_backup_copy_job.job_id} completed successfully')

        for _copy_precedence in _secondary_copies:
            VirtualServerUtils.decorative_log(f'Restore from Replica copy {_copy_precedence}')
            try:
                VirtualServerUtils.decorative_log(f'Guest file restore from Replica {_copy_precedence}')
                file_restore_options = OptionsHelper.FileLevelRestoreOptions(auto_subclient)
                VirtualServerUtils.set_inputs(testcase.tcinputs, file_restore_options)
                file_restore_options.browse_from_snap = True
                file_restore_options.copy_precedence = _copy_precedence
                for vm in auto_subclient.vm_list:
                    auto_subclient.guest_file_restore(file_restore_options, discovered_client=vm)
            except Exception as exp:
                testcase.log.error('sleeping 12 minutes for cleanup of mounted snaps')
                time.sleep(720)
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

            try:
                VirtualServerUtils.decorative_log(f'Out of place full vm restore from Replica {_copy_precedence}')
                vm_restore_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, testcase)
                vm_restore_options.power_on_after_restore = True
                vm_restore_options.unconditional_overwrite = True
                vm_restore_options.browse_from_snap = True
                vm_restore_options.copy_precedence = _copy_precedence
                for vm in auto_subclient.vm_list:
                    auto_subclient.virtual_machine_restore(vm_restore_options, discovered_client=vm)
            except Exception as exp:
                testcase.test_individual_status = False
                testcase.test_individual_failure_message = str(exp)

        return (auto_subclient, backup_options, vm_restore_options, testcase.test_individual_status,
                testcase.test_individual_failure_message)


