# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file for performing deployment operations

DeploymentHelper is the only class defined in this file

DeploymentHelper: Helper class to perform deployment operations

DeploymentHelper:

    deploy_vmware_ova()             --  Deploys the OVA into the given VMware vCenter Server

    validate_vm()                   --  Validates the newly deployed VM is up or not

    validate_services()             --  Validates the Commvault services on the new VM

    check_machine_status()          --  Checks if the VM is up and accessible

"""

import time
import socket
from VirtualServer.VSAUtils.HypervisorHelper import Hypervisor
from VirtualServer.VSAUtils.VirtualServerUtils import decode_password


class DeploymentHelper:
    """Helper class to deploy VM and do setup"""

    def __init__(self, testcase):
        """
        constructor for install related files
        """
        self.log = testcase.log
        self.testcase = testcase
        self.tcinputs = testcase.tcinputs
        self.hvobj = None
        self.machine = None
        self.vm_name = None

    def deploy_vmware_ova(self):
        """
        Deploys the VMware OVA into a VM

        Raises:
            Exception:
                if a VM could not be properly deployed from the OVA

        """
        try:
            self.hvobj = Hypervisor([self.tcinputs['vCenter']],
                                    self.tcinputs['vCenterUsername'],
                                    self.tcinputs['vCenterPassword'],
                                    "vmware", self.testcase.commcell,
                                    socket.getfqdn())
            self.vm_name = self.tcinputs.get("VMName", "TestStoreOVA")

            ova_path = self.testcase.utils.get_temp_files()[0]
            password = self.tcinputs.get("VMPassword", None)
            if not password:
                password = decode_password(
                    self.testcase.config.MEDIAKIT.VirtualAppliance.VMware.password)

            self.hvobj.deploy_ova(ova_path, self.vm_name,
                                  self.tcinputs['esx'],
                                  self.tcinputs['datastore'],
                                  self.tcinputs['network'],
                                  password)

            self.log.info("Waiting for VM to finish setup and reboot")
            time.sleep(600)
            self.hvobj.VMs = self.vm_name
            self.hvobj.VMs[self.vm_name].update_vm_info()
            self.hvobj.VMs[self.vm_name].user_name = "Administrator"
            self.hvobj.VMs[self.vm_name].password = password
        except Exception as exp:
            self.log.exception("Exception occurred while deploying the OVA")
            raise exp

    def validate_vm(self):
        """
        Validates the deployed VM

        Raises:
            Exception:
                if the newly deployed VM could not be validated

        """
        try:
            self.hvobj.VMs[self.vm_name].update_vm_info("All", True, True)
            self.machine = self.hvobj.VMs[self.vm_name].machine
            self.check_machine_status()
            self.validate_services()
        except Exception as exp:
            self.log.exception("Exception occurred while validating the deployed VM."
                               " Please check the logs")
            raise exp

    def validate_services(self):
        """
        Validates all the services in the CS

        Raises:
            Exception:
                if atleast one of the service is not up

        """
        services = ['cvd', 'tomcat', 'AppMgrSvc', 'BlrSvc', 'ClMgrs', 'cvfwd', 'EvMgrs', 'JobMgr',
                    'MediaManager', 'QSDK', 'sqlservr']
        for service in services:
            if not self.machine.is_process_running(service):
                raise Exception(f"One of the expected process {service} is not running")

    def check_machine_status(self):
        """
        Checks if the VM is up after deployment is complete

        Raises:
            Exception:
                if the machine is not up after a wait time

        """
        wait_time = 0
        while wait_time < 120:
            wait_time += 30
            if self.machine.is_connected:
                return
            time.sleep(30)
        raise Exception("The machine is not up even after waiting")
