import threading
import importlib.abc
import os
import sys
import ast
import uuid
import time
import calendar
import random
import string
from urllib import request


def enum(**enums):
    return type('Enum', (), enums)


def download_url(url, download_path, download_timeout=120):
    """Download file from the given url..

        Args:
            url   (str)     -- valid url from where file will be downloaded

            download_path      (str)     -- existing directory where file will be downloaded

            download_timeout   (int)     -- timeout to be considered while download

        Returns:
            downloaded file path

        Raises:
            Exception:
                if any error occurs in the downloading url
    """
    if not os.path.exists(os.path.dirname(download_path)):
        raise Exception("download path %s doesn't exists".format(path=os.path.dirname(download_path)))

    try:
        res = request.urlopen(url=url, timeout=download_timeout)
    except BaseException as err:
        raise Exception("Following Exception was raised while downloading. Details: {err}\nExiting".format(err=err))

    if res.code != 200:
        raise Exception("Unsuccessful downloading attempt. Details: {info}\tExiting".format(info=res))

    try:
        with open(download_path, "wb") as fd_handle:
            for chunk in res:
                fd_handle.write(chunk)
        fd_handle.close()
    except Exception as error:
        raise Exception("Unable to download file from url %s. Error %s" % (url, error))


class MyThread(threading.Thread):
    def __init__(self, *args, **kwargs):
        self.parent = threading.current_thread().ident
        threading.Thread.__init__(self, *args, **kwargs)


def threadLauncher(tCount, q, target):
    for i in range(tCount):
        uniqueID = str(uuid.uuid1())
        theThread = MyThread(target=target, name=uniqueID, args=(uniqueID, q))
        theThread.daemon = True
        theThread.start()

    return True


def import_module(dir_path, module_name):
    filename = resolve_filename(dir_path, module_name)
    if module_name in sys.modules:
        return sys.modules[module_name]

    return Loader(module_name, filename).load_module(module_name)


def resolve_filename(dir_path, module_name):
    filename = os.path.join(dir_path, *module_name.split('.'))
    if os.path.isdir(filename):
        filename = os.path.join(filename, '__init__.py')
    else:
        filename += '.py'
    return filename


class Loader(importlib.abc.FileLoader, importlib.abc.SourceLoader):
    pass


def set_defaults(main, defaults):
    """Sets default value for the main dictionary using the defaults dictionary

        Args:
            main        (dict)      The dictionary to set default values
            defaults    (dict)      The dictionary which has default values

        Returns:
            Dictionary with default values applied

    """

    for key in defaults:
        if key not in main:
            main[key] = defaults[key]
        else:
            # If key is present, then make the value of same type
            if type(defaults[key]) is int:
                main[key] = get_int(main[key])

        if type(defaults[key]) == dict:
            set_defaults(main[key], defaults[key])


def get_dictionary_difference(dict1, dict2):
    """Gets the difference between the two dictionaries provided

        Args:
            dict1       (dict)      Dictionary 1 to compare
            dict2       (dict)      Dictionary 2 to compare

        Returns:
            added       (set)       Keys which are present in dict 1 but not in dict 2
            removed     (set)       Keys which are present in dict 2 but not in dict 1
            modified    (dict)      Keys which are present in both dict 1 and 2 but with
                                    modified values.

    """

    if not isinstance(dict1, dict) or not isinstance(dict2, dict):
        raise Exception('Cannot get difference as one of them is not a dictionary')

    d1_keys = set(dict1.keys())
    d2_keys = set(dict2.keys())
    intersect_keys = d1_keys.intersection(d2_keys)
    added = d1_keys - d2_keys
    removed = d2_keys - d1_keys
    modified = {}

    for o in intersect_keys:
        if dict1[o] != dict2[o]:
            modified[o] = (dict1[o], dict2[o])

    return added, removed, modified


def get_random_string(length=8, lowercase=True, uppercase=False,
                      digits=True, special_chars=False, custom_chars=None):
    """Generates a random string of the specified length

        Args:
             length         (int)      Length of the random string

             lowercase      (bool)     Include lowercase ASCII characters

             uppercae       (bool)     Include uppercase ASCII characters

             digits         (bool)     Include digits

             special_chars  (bool)     Include special characters

             custom_chars   (str)      Include custom characters like unicode

         Returns:
             (str)  --  A random string of specified length with the selected characters.
    """

    the_string = ''

    the_string += string.ascii_lowercase if lowercase else ''
    the_string += string.ascii_uppercase if uppercase else ''
    the_string += string.digits if digits else ''
    the_string += string.punctuation if special_chars else ''
    the_string += custom_chars if isinstance(custom_chars, str) else ''

    return ''.join(random.choice(the_string) for s in range(length))


def get_int(data, default=0):
    """Gets the int value of a string, returns default value on exception

        Args:
            data    (str/int)       The string to parse the integer from

            default (int)           The default integer value to return upon integer exception

        Returns:
            Parsed int from the string.
            0 if cannot parse int from string

    """

    try:
        return int(data)
    except ValueError:
        return default


def convert_to_timestamp(formatted_time):
    """Converts formatted time (%Y-%m-%d %H:%M:%S) to epoch timestamp in UTC timezone

        Args:
            formatted_time  (str)   Formatted time to convert to timestamp

        Returns:
            timestamp       (int)   Timestamp got from formatted time

    """

    # Using calendar.gmtime instead of time.mktime as time.mktime assumes its arg as local time,
    # whereas calendar.gmtime assumes the arg as epoch which is what we want

    return int(calendar.timegm(time.strptime(formatted_time, '%Y-%m-%d %H:%M:%S')))


def convert_to_formatted_time(timestamp):
    """Converts epoch timestamp to formatted time (%Y-%m-%d %H:%M:%S) in UTC timezone

        Args:
            timestamp   (int/str)   Timestamp to convert to formatted time

        Returns;
            Formatted time  (str)   Formatted time of the timestamp

    """

    return time.strftime('%Y-%m-%d %H:%M:%S', time.gmtime(timestamp))


def get_parent_path(path, separator):
    """Gets the parent directory for any given path

        Args:
            path        (str)       Path to get parent for
            separator   (str)       Separator present in the path

        Returns:
            Parent directory of the given path

        Example:
            C:\folder\file.txt -> C:\folder

    """

    if separator is None or separator == '':
        raise Exception('Invalid separator provided.')

    path_split = path.split(separator)
    path_split.pop()
    parent = separator.join(path_split)
    parent = separator if parent == '' else parent
    return parent


def add_prefix_sep(path, separator):
    """Adds a sep to the beginning of the path if not already prefixed

        Args:
            path        (str)       Path to prefix sep with
            separator   (str)       Separator of the path

        Returns:
            path with prefixed sep

        Example:
            C:\file.txt -> \C:\file.txt
            /home/file.txt -> /home/file.txt

    """

    if separator is None or separator == '':
        raise Exception('Invalid separator provided.')

    if path == '':
        return separator

    path = separator + path if path[0] != separator else path
    return path


def add_trailing_sep(path, separator=''):
    """Adds a trailing sep to the given path if present

        Args:
            path        (str)       Path to add the trailing sep
            separator   (str)       Separator of the path

        Returns:
            path with trailing sep added

        Example:
            C:\folder -> C:\folder\
            /home/folder/ -> /home/folder/

    """

    if separator is None or separator == '':
        raise Exception('Invalid separator provided.')

    if path == '':
        return separator

    path = path + separator if path[-1] != separator else path
    return path


def remove_prefix_sep(path, separator=''):
    """Removes the prefixed sep for the given path if present

        Args:
            path        (str)       Path to remove the prefixed sep
            separator   (str)       Separator of the path

        Returns:
            path with prefixed sep removed

        Example:
            \C:\file.txt -> C:\file.txt
            C:\file.txt -> C:\file.txt

    """

    if separator is None or separator == '':
        raise Exception('Invalid separator provided.')

    path = path[1:] if path[0] == separator else path
    return path


def remove_trailing_sep(path, separator=''):
    """Removes trailing sep for the given path

        Args:
            path        (str)       Path to remove the trailing sep
            separator   (str)       Separator of the path

        Returns:
            path with trailing sep removed

        Example:
            C:\folder\ -> C:\folder
            /home/folder/ -> /home/folder

    """

    if separator is None or separator == '':
        raise Exception('Invalid separator provided.')

    path = path[:-1] if path[-1] == separator else path
    return path