# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case
(HyperV Replication group: Verify Creation and edit operation)
TestCase is the only class defined in this file.

TestCase: Class for executing this test case
Sample JSON: {
        "hypervisor": "HyperV_name",
        "source_vm1": "Auto_HypervVM1",
        "source_vm2": "Auto_HypervVM2",
        "source_vm3" : "Auto_HypervVM3" ,
        "recovery_target": "Auto_HyperV",
        "storage": "storage1",
        "network" : "network1",
        "network_2" : "network2"
}
"""
from time import sleep
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.virtualization_replication import _Target, SOURCE_HYPERVISOR_HYPERV
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets
from Web.AdminConsole.VSAPages.vm_groups import VMGroups
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the replication"""
    test_step = TestStep()
    _MICROSOFT_VENDOR_NAME = "Microsoft Hyper-V"
    _VM_TYPE = 'Virtual Server'
    _REPLICATION_TYPE = "Periodic"
    _TARGET_FREQ_NUMBER = 4
    _TARGET_FREQ_UNIT = _Target.FREQUENCY_HOURS

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = " HyperV Replication group: Verify Creation and edit operation"
        self.tcinputs = {
            "hypervisor": None,
            "source_vm1": None,
            "source_vm2": None,
            "source_vm3": None,
            "recovery_target": None,
            "storage": None,
            "network": None,
            "network_2": None
        }
        self.utils = None
        self.browser = None
        self.admin_console = None
        self.replication_group = None
        self.group_details = None
        self.vm_group = None
        self.target_details = None
        self.edit_vm = None
        self.vm_noedit_details = None

    @property
    def group_name(self):
        """Returns the name for the replication group"""
        return f"Group_TC_{self.id}"

    def login(self):
        """Logs in to admin console"""
        self.admin_console = AdminConsole(
            self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname']
        )
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)
        self.vm_group = VMGroups(self.admin_console)

    def logout(self):
        """Logs out from the admin console"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def get_recovery_target_details(self):
        """Gets the recovery target details for verification"""
        self.admin_console.navigator.navigate_to_replication_targets()
        recovery_target = RecoveryTargets(self.admin_console).access_target(
            self.tcinputs['recovery_target']
        )
        summary = recovery_target.get_target_summary()
        self.target_details = summary['Network adapter']

    @test_step
    def delete_replication_group(self):
        """Deletes the replication group if it exists
        Returns: True if replication group was present and deleted, false if it didn't exist before
        """
        self.admin_console.navigator.navigate_to_replication_groups()
        if not self.replication_group.has_group(self.group_name):
            return False
        self.replication_group.delete_group(self.group_name)
        return True

    @test_step
    def create_replication_group(self):
        """Creates and configures a new Hyper V replication group """
        configure = self.replication_group.configure_hyperv()
        configure.content.set_name(self.group_name)

        configure.content.select_vm_from_browse_tree(self.tcinputs['hypervisor'],
                                                     {"VMs": [self.tcinputs['source_vm1']]})
        configure.content.select_vm_from_browse_tree(self.tcinputs['hypervisor'],
                                                     {"VMs": [self.tcinputs['source_vm2']]})
        configure.next()

        configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        configure.target.unconditionally_overwrite_vm(True)
        configure.next()

        configure.storage_cache.select_storage(self.tcinputs['storage'])
        configure.next()

        override_options = configure.override_options.override_vms(self.tcinputs['source_vm1'])
        override_options.select_network(self.tcinputs['network'])
        self.admin_console.click_button('Save')

        self.vm_noedit_details = configure.override_options.get_hyperv_details(
            self.tcinputs['source_vm2']
        )
        self.utils.assert_comparison(self.vm_noedit_details, self.target_details)

        configure.next()

        sleep(5)
        configure.finish()

    @test_step
    def verify_group_creation(self):
        """Verify whether the group creation was successful with a
        visible entry in Replication groups table"""
        if not self.replication_group.has_group(self.group_name):
            self.log.error("The replication group %s was not successfully created", self.group_name)
        table_content = (self.replication_group.
                         get_replication_group_details_by_name(self.group_name))
        expected_content = {
            'Group name': [self.group_name],
            'Source': [self.tcinputs['hypervisor']],
            'Destination': [self.tcinputs['recovery_target']],
            'Type': [self._VM_TYPE],
            'Replication type': [self._REPLICATION_TYPE],
            'State': ['Enabled'],
            'Actions': ['']
        }
        self.utils.assert_comparison(table_content, expected_content)

    @test_step
    def verify_vm_group(self):
        """Verifies whether the VM group exists or not"""
        self.admin_console.navigator.navigate_to_vm_groups()
        self.admin_console.refresh_page()
        table_content = self.vm_group.get_details_by_vm_group(self.group_name)
        self.utils.assert_includes(self.group_name, table_content['Name'][0])
        self.utils.assert_comparison(table_content['Vendor'][0], self._MICROSOFT_VENDOR_NAME)
        self.utils.assert_comparison(table_content['Hypervisor'][0], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(
            table_content['Plan'][0], f"{self.group_name}_ReplicationPlan")

    @test_step
    def verify_overview(self):
        """Verifies the details of the replication group in the overview tab"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)

        summary = self.group_details.overview.get_summary_details()
        self.utils.assert_comparison(summary['Source'], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(summary['Recovery target'], self.tcinputs['recovery_target'])
        self.utils.assert_comparison(summary['Destination vendor'], 'Hyper-V')
        self.utils.assert_comparison(summary['Replication type'], self._REPLICATION_TYPE)
        self.utils.assert_comparison(summary['State'], "Enabled")

    @test_step
    def disable_replication_group(self):
        """Disables the replication group and re-enables it to verify the group status"""
        self.group_details.disable_replication_group()
        summary = self.group_details.overview.get_summary_details()
        self.utils.assert_comparison(summary['State'], 'Disabled')
        self.admin_console.refresh_page()
        self.group_details.enable_replication_group()
        self.utils.assert_comparison(
            self.group_details.overview.get_summary_details().get('State'), 'Enabled')

    @test_step
    def verify_configuration(self):
        """Verifies the details of the replication group in the configuration tab"""
        self.group_details.access_configuration_tab()

        rpo = self.group_details.configuration.get_rpo_details()
        self.utils.assert_includes(
            f"{self._TARGET_FREQ_NUMBER} {self._TARGET_FREQ_UNIT.lower()}",
            rpo['Replication frequency']
        )
        storages = self.group_details.configuration.get_storage_details()
        self.utils.assert_includes(self.tcinputs['storage'], storages)

        adv_options = self.group_details.configuration.get_advanced_options_details()
        self.utils.assert_comparison(
            adv_options['Unconditionally overwrite if VM already exists'], "ON")

        # checking vm content
        vm_details_1 = self.group_details.configuration.get_vm_details(self.tcinputs['source_vm1'])
        self.utils.assert_comparison(self.tcinputs['source_vm1'], vm_details_1['Source VM'])

        vm_details_2 = self.group_details.configuration.get_vm_details(self.tcinputs['source_vm2'])
        self.utils.assert_comparison(self.tcinputs['source_vm2'], vm_details_2['Source VM'])

    @test_step
    def add_delete_vm_to_group(self):
        """Add VM to group after creation"""
        add_vm = (self.group_details.configuration.
                  add_virtual_machines(vm_type=SOURCE_HYPERVISOR_HYPERV))
        add_vm.add_vm([self.tcinputs['source_vm3']])

        new_vm_details = (self.group_details.configuration.
                          get_vm_details(self.tcinputs['source_vm3']))
        if new_vm_details['Source VM'] != self.tcinputs['source_vm3']:
            raise CVTestStepFailure(
                f"Expected value of Source VM {self.tcinputs['source_vm3']} does not match "
                f"the collected value {new_vm_details['Source VM']}")
        self.group_details.configuration.remove_virtual_machines(self.tcinputs['source_vm3'])
        try:
            self.admin_console.refresh_page()
            self.group_details.configuration.get_vm_details(self.tcinputs['source_vm3'])
        except:
            self.log.info("VM successfully deleted from group")
        else:
            raise CVTestStepFailure("VM could not be deleted successfully")

    @test_step
    def verify_disabled_fields(self, vm_id):
        """Verifies that the disabled fields are disabled or not"""
        self.admin_console.refresh_page()
        self.edit_vm = self.group_details.configuration.edit_virtual_machines(
            self.tcinputs[f'source_vm{vm_id}'], vm_type=SOURCE_HYPERVISOR_HYPERV)

        field_id = 'displayNamePrefixSuffix'
        field_disabled = self.edit_vm.is_field_disabled(field_id)
        if field_disabled is None:
            raise CVTestStepFailure(f"The field {field_id} is not interactable/existent")
        if not field_disabled:
            raise CVTestStepFailure(
                f'In Edit VM The field {field_id} is enabled, but must be disabled')

    @test_step
    def verify_edit_vm(self, after_edit=False, vm_id=1):
        """Verifies the data on the edit VM page"""
        sleep(10)
        if vm_id == 2:
            self.utils.assert_comparison(self.vm_noedit_details, self.edit_vm.network)
        elif after_edit:
            self.utils.assert_includes(
                self.tcinputs['network_2'], self.edit_vm.network)
        else:
            self.utils.assert_includes(
                self.tcinputs['network'], self.edit_vm.network)
        self.edit_vm.cancel()

    @test_step
    def edit_vm_details(self):
        """Modify the group details to check if the detail change is registered on Command Center"""
        self.admin_console.refresh_page()
        edit_vm = (self.group_details.configuration.
                   edit_virtual_machines(self.tcinputs['source_vm1'],
                                         vm_type=SOURCE_HYPERVISOR_HYPERV))
        edit_vm.select_network(self.tcinputs['network_2'])
        edit_vm.save()

    def run(self):
        """Runs the testcase in order"""
        try:
            self.get_recovery_target_details()
            self.delete_replication_group()
            self.create_replication_group()
            self.verify_group_creation()
            self.verify_vm_group()

            self.verify_overview()
            self.disable_replication_group()
            self.verify_configuration()
            self.add_delete_vm_to_group()

            self.verify_disabled_fields(vm_id=1)
            self.verify_edit_vm(after_edit=False, vm_id=1)
            self.verify_disabled_fields(vm_id=2)
            self.verify_edit_vm(after_edit=False, vm_id=2)

            self.edit_vm_details()

            self.verify_disabled_fields(vm_id=1)
            self.verify_edit_vm(after_edit=True, vm_id=1)
            self.verify_disabled_fields(vm_id=2)
            self.verify_edit_vm(after_edit=True, vm_id=2)

            self.delete_replication_group()
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Performs garbage collection for the TC"""
        self.logout()
