# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: Verification of configuring HyperV replication group and its job completion

This test case verifies whether HyperV replication group creation succeeds with default options.
Then it waits for the replication and backup jobs to complete successfully,
 and finally verifies the details on the replication monitor

TestCase: Class for executing this test case
Sample JSON: {
        "ClientName": "HyperV_name",
        "source_vms": ["Auto_HypervVM1", "Auto_HypervVM2"],
         # (Minimum 2) Last VM is for verifying deletion from overview tab
        "recovery_target": "Auto_HyperV ",
        "storage_name" : "storage1",
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.monitor import ReplicationMonitor
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.virtualization_replication import _Target
from Web.AdminConsole.VSAPages.vm_groups import VMGroups
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Server.Scheduler.schedulerhelper import SchedulerHelper


class TestCase(CVTestCase):
    """This class is used to automate the replication """
    test_step = TestStep()
    _VM_TYPE = 'Virtual Server'
    _REPLICATION_TYPE = "Periodic"
    _TARGET_FREQ_NUMBER = 4
    _TARGET_FREQ_UNIT = _Target.FREQUENCY_HOURS
    _MICROSOFT_VENDOR_NAME = "Microsoft Hyper-V"

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Verification of configuring Hyper-V replication group and its Job completion"

        self.tcinputs = {
            "ClientName": None,
            "source_vms": [],
            "recovery_target": None,
            "storage_name": None,
        }
        self.utils = None

        self.browser = None
        self.admin_console = None

        self.replication_group = None
        self.group_details = None
        self.vm_group = None
        self.replication_monitor = None

        self.schedule = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser,
                                          machine=(self.inputJSONnode['commcell']
                                                                     ['webconsoleHostname']))
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])

        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)
        self.vm_group = VMGroups(self.admin_console)
        self.replication_monitor = ReplicationMonitor(self.admin_console)

    def logout(self):
        """Logs out from the admin console"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)
            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The replication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        configure = self.replication_group.configure_hyperv()
        configure.content.set_name(self.group_name)
        for vm_i in self.tcinputs['source_vms']:
            configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                         {"VMs": [vm_i]})
        configure.next()

        configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        configure.target.unconditionally_overwrite_vm(True)
        configure.next()

        configure.storage_cache.select_storage(self.tcinputs['storage_name'])
        configure.next()
        configure.next()

        sleep(5)
        configure.finish()

    @test_step
    def verify_group_creation(self):
        """Verify whether the group creation was successful with a
        visible entry in Replication groups table"""
        if not self.replication_group.has_group(self.group_name):
            self.log.error("The replication group %s was not successfully created", self.group_name)
        table_content = (self.replication_group.
                         get_replication_group_details_by_name(self.group_name))
        expected_content = {
            'Group name': [self.group_name],
            'Source': [self.tcinputs['ClientName']],
            'Destination': [self.tcinputs['recovery_target']],
            'Type': [self._VM_TYPE],
            'Replication type': [self._REPLICATION_TYPE],
            'State': ['Enabled'],
            'Actions': ['']
        }
        self.utils.assert_comparison(table_content, expected_content)

    @test_step
    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        vmgroup_subclient = (self.commcell.clients.get(self.tcinputs['ClientName'])
                             .agents.get('virtual server')
                             .backupsets.get('defaultBackupSet')
                             .subclients.get(self.group_name))
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        backup_job_id = vmgroup_subclient.find_latest_job().job_id
        self.log.info('Waiting for backup job %s to finish', backup_job_id)
        job_obj = self.commcell.job_controller.get(backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_job_completion(self):
        """Waits for replication joib to complete and then verify its details"""
        self.log.info('Waiting to 16 minutes to let the replication job trigger')
        sleep(960)
        self.schedule = (self.replication_group.
                         get_schedule_name_by_replication_group(self.group_name))
        self.client.schedules.refresh()
        schedule_obj = self.client.schedules.get(self.schedule)
        job_obj = SchedulerHelper(schedule_obj, self.commcell).get_jobid_from_taskid()
        self.log.info("Waiting for replication job id %s to finish", job_obj.job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_monitor(self):
        """Verifies the details shown on replication monitor"""
        self.admin_console.navigator.navigate_to_replication_monitor()

        self.log.info("Verifying First Source VM From Monitor")
        table_content = self.replication_monitor.get_replication_group_details(
            [self.tcinputs['source_vms'][0]], self.group_name)
        self.utils.assert_comparison(table_content['Source'], [self.tcinputs['source_vms'][0]])
        self.utils.assert_comparison(table_content['SLA status'], ['Met'])
        rpo_unit = self._TARGET_FREQ_UNIT.lower().replace("(", "").replace(")", "")
        self.utils.assert_comparison(table_content['Frequency'],
                                     [f"{self._TARGET_FREQ_NUMBER} {rpo_unit}"])
        self.utils.assert_comparison(table_content['Replication group'], [self.group_name])
        self.utils.assert_comparison(table_content['Status'], ['In Sync'])

        self.log.info("Verifying Second Source VM From Monitor")
        table_content = self.replication_monitor.get_replication_group_details(
            [self.tcinputs['source_vms'][1]], self.group_name)
        self.utils.assert_comparison(table_content['Source'], [self.tcinputs['source_vms'][1]])
        self.utils.assert_comparison(table_content['SLA status'], ['Met'])
        rpo_unit = self._TARGET_FREQ_UNIT.lower().replace("(", "").replace(")", "")
        self.utils.assert_comparison(table_content['Frequency'],
                                     [f"{self._TARGET_FREQ_NUMBER} {rpo_unit}"])
        self.utils.assert_comparison(table_content['Replication group'], [self.group_name])
        self.utils.assert_comparison(table_content['Status'], ['In Sync'])

    @test_step
    def verify_vm_deletion(self):
        """Deletes the VM from the overview tab and checks to see if it works"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)
        self.group_details.overview.remove_virtual_machines(self.tcinputs['source_vms'][-1],
                                                            configuration_tab=False)

        # Wait for DR orchestration to complete
        sleep(60)
        self.admin_console.refresh_page()
        sleep(10)
        try:
            self.group_details.overview.get_vm_details(self.tcinputs['source_vms'][-1])
        except:
            self.log.info("VM with name %s deleted successfully", self.tcinputs['source_vms'][-1])
        else:
            raise CVTestStepFailure("VM with name %s still exists,"
                                    " even after delete operation from overview tab",
                                    self.tcinputs['source_vms'][-1])

    @test_step
    def verify_deletion(self):
        """Verifies that the pair has been deleted and its associated schedules are disabled too"""
        self.delete_virtualization_group(wait_for_sync=True)

        self.admin_console.navigator.navigate_to_vm_groups()

        self.admin_console.refresh_page()

        if self.vm_group.has_vm_group(self.group_name):
            raise CVTestStepFailure(f"VM group {self.group_name} exists even though"
                                    f"the replication group is deleted")

        self.commcell.schedules.refresh()
        if self.commcell.schedules.has_schedule(schedule_name=self.schedule):
            raise CVTestStepFailure(f"Schedule with name {self.schedule} exists even though"
                                    f"the replication group is deleted")

    def run(self):
        """Runs the testcase in order"""
        try:
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()
            self.verify_group_creation()

            self.logout()
            self.verify_backup_job_completion()
            self.verify_replication_job_completion()

            self.login()
            self.verify_replication_monitor()
            self.verify_vm_deletion()
            self.verify_deletion()

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Performs garbage collection for the TC"""
        self.logout()
