# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""Main file for executing this test case(verify the recovery target for Hyper-V)

TestCase is the only class defined in this file.

TestCase: Class for executing this test case
Sample JSON: {
            "hypervisor": "HyperV_name",
            "access_node": "node1" ,
            "storage_account" : "path1",
            "network" : "network1",
            "access_node_2": "node2",
            "storage_account_2": "path2",
            "network_2": "network2"

}
"""

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.page_object import TestStep
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets, TargetConstants


class TestCase(CVTestCase):
    """This testcase is for the verification and update of HyperV recovery target"""
    test_step = TestStep()

    def __init__(self):
        """Initialises the class and variables are set to None"""
        super(TestCase, self).__init__()
        self.name = "Hyper-V Recovery Targets: Verification of creation and update of target"
        self.tcinputs = {
            "hypervisor": None,
            "access_node": None,
            "storage_account": None,
            "network": None,
            "access_node_2": None,
            "storage_account_2": None,
            "network_2": None,
        }
        self.browser = None
        self.admin_console = None
        self.utils = None
        self.recovery_targets = None
        self.target = None
        self.edit_target = None

    @property
    def target_name(self):
        """Returns the initial target name for the recovery target"""
        return f'Auto_{self.id}'

    @property
    def edited_target_name(self):
        """Returns the modified target name"""
        return f'Auto_edited_{self.id}'

    def login(self):
        """Logs in to the admin console and initialises it"""
        self.admin_console = AdminConsole(self.browser,
                                          machine=(self.inputJSONnode['commcell']
                                                                     ['webconsoleHostname']))
        self.admin_console.goto_adminconsole()
        self.admin_console.login(
            self.inputJSONnode['commcell']['commcellUsername'],
            self.inputJSONnode['commcell']['commcellPassword']
        )

    def setup(self):
        """Sets up the various variables and initiates the admin console"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.login()
            self.recovery_targets = RecoveryTargets(self.admin_console)
        except Exception as exp:
            raise CVTestCaseInitFailure(f"Could not initialise the test case {self.id} "
                                        f"due to following error:\n{str(exp)}")

    @test_step
    def create_recovery_target(self):
        """Creates a recovery target in the recovery targets page"""
        self.admin_console.navigator.navigate_to_replication_targets()
        hyperv_target = (self.recovery_targets.
                         configure_hyperv_recovery_target(recovery_target_name=self.target_name,
                                                          application_type='Replication'))
        hyperv_target.select_destination_hypervisor(self.tcinputs['hypervisor'])
        hyperv_target.select_access_node(self.tcinputs['access_node'])
        hyperv_target.set_vm_display_name("DR-Auto", suffix=True)
        hyperv_target.set_destination_folder(self.tcinputs['storage_account'])
        hyperv_target.select_network(self.tcinputs['network'])
        hyperv_target.save()

    @test_step
    def verify_target_creation(self):
        """Verifies the information of the recovery target on the recovery target page table"""
        self.admin_console.navigator.navigate_to_replication_targets()
        details = self.recovery_targets.get_target_details(self.target_name)
        # comparison
        self.utils.assert_comparison(details['Name'][0], self.target_name)
        self.utils.assert_comparison(details['Vendor'][0], 'Hyper-V')
        self.utils.assert_comparison(details['Application type'][0], 'Replication')
        self.utils.assert_comparison(details['Destination hypervisor'][0],
                                     self.tcinputs['hypervisor'])

    @test_step
    def delete_recovery_target(self):
        """Tries to delete the recovery target"""
        self.admin_console.navigator.navigate_to_replication_targets()
        if self.recovery_targets.has_target(self.edited_target_name):
            self.recovery_targets.delete_recovery_target(self.edited_target_name)
            self.admin_console.refresh_page()
            if self.recovery_targets.has_target(self.edited_target_name):
                raise CVTestStepFailure("Could not delete recovery target")

    @test_step
    def verify_target_details(self, after_edit=False):
        """Verifies the target details in the target's detail page"""
        if not after_edit:
            self.target = self.recovery_targets.access_target(self.target_name)
        target_details = self.target.get_target_summary()
        if not after_edit:
            self.utils.assert_comparison(target_details['Access node'],
                                         self.tcinputs['access_node'])
            self.utils.assert_comparison(target_details['Restore location'],
                                         self.tcinputs['storage_account'])
            self.utils.assert_comparison(target_details['Network adapter'],
                                         self.tcinputs['network'])
        else:
            self.utils.assert_comparison(target_details['Access node'],
                                         self.tcinputs['access_node_2'])
            self.utils.assert_comparison(target_details['Network adapter'],
                                         self.tcinputs['network_2'])
            self.utils.assert_comparison(target_details['Restore location'],
                                         self.tcinputs['storage_account_2'])
        # Common for both
        self.utils.assert_comparison(target_details['Destination hypervisor'],
                                     self.tcinputs['hypervisor'])
        self.utils.assert_comparison(target_details['Application type'], 'Replication')

    @test_step
    def verify_target_fields_disabled(self):
        """Verifies that the edit target page's fields are disabled"""
        self.edit_target = self.target.edit_target(self.target_name,
                                                   TargetConstants.MICROSOFT_HYPERV)
        fields_disabled = 'client'
        field_disabled = self.edit_target.is_field_disabled(fields_disabled)
        if field_disabled is None:
            raise CVTestStepFailure(f"The field {fields_disabled} is not interactable/existent")
        if not field_disabled:
            raise CVTestStepFailure(f'In Edit VM The field {fields_disabled} '
                                    f'is enabled, but must be disabled')

    @test_step
    def edit_recovery_target(self):
        """Edits the target configuration"""
        self.edit_target.recovery_target_name = self.edited_target_name
        self.edit_target.set_recovery_target_name()
        self.edit_target.select_access_node(self.tcinputs['access_node_2'])
        self.edit_target.set_destination_folder(self.tcinputs['storage_account_2'])
        self.edit_target.select_network(self.tcinputs['network_2'])
        self.admin_console.click_button('Save')

    def run(self):
        """Executes the testcase"""
        try:
            self.delete_recovery_target()
            self.create_recovery_target()  # for creating target
            self.verify_target_creation()  # for verifying its creation
            self.verify_target_details(after_edit=False)  # verifying its details before editing
            self.verify_target_fields_disabled()  # run to validate non editable fields
            self.edit_recovery_target()  # for editing recovery target
            self.verify_target_details(after_edit=True)  # verifying edited fields
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Performs garbage collection for the TC"""
        try:
            self.delete_recovery_target()
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)
