# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test Case for OneDrive v2 Verification of Auto-discovery and Refresh Cache Operation

TestCase:   Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup the requirements for the test case

    run()           --  run function of this test case

"""


import re
import time

from Application.CloudApps.cloud_connector import CloudConnector
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.windows_machine import WindowsMachine
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Office365Pages import constants as o365_constants
from Web.AdminConsole.Office365Pages.onedrive import OneDrive
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        """Initializes testcase class object"""
        super(TestCase, self).__init__()
        self.name = "Onedrive v2: Verification of Auto-discovery and Refresh Cache Operation"
        self.regex = ('^CommvaultAutoGenerated.*|^CommvaultSP.*|^CommvaultEX.*|^CVEXBackupAccount.*'
                      '|^CVAzureBackupAccount.*|^CV.*BackupAccount.*|^\\w*#EXT#*')
        self.group_name = 'onedrive_discover_automation_group'
        self.browser = None
        self.cvcloud_object = None
        self.navigator = None
        self.admin_console = None
        self.onedrive = None
        self.users = list()
        self.machine = None

    def setup(self):
        """Initial configuration for the testcase."""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(
                self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(
                self.inputJSONnode['commcell']['commcellUsername'],
                self.inputJSONnode['commcell']['commcellPassword'])

            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_office365()
            self.tcinputs['Name'] += str(int(time.time()))

            self.log.info("Creating an object for office365 helper")
            self.tcinputs['office_app_type'] = OneDrive.AppType.one_drive_for_business
            self.onedrive = OneDrive(self.tcinputs, self.admin_console)

            self.onedrive.create_office365_app()
            self._initialize_sdk_objects()

            # User/Group Creation
            self.cvcloud_object.one_drive.create_group(self.group_name)
            for i in range(1, o365_constants.OneDrive.REFRESH_CACHE_USER_COUNT.value + 1):
                user_name = f'onedrive_discover_automation_user{i}'
                self.users.append(user_name)
                self.cvcloud_object.one_drive.create_user(user_name=user_name)
                if i < o365_constants.OneDrive.REFRESH_CACHE_USER_COUNT.value:
                    self.cvcloud_object.one_drive.add_group_member(user_name, self.group_name)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def _initialize_sdk_objects(self):
        """Initializes the sdk objects after app creation"""
        self.commcell.refresh()
        self.log.info("Create client object for: %s", self.tcinputs['Name'])
        self._client = self.commcell.clients.get(self.tcinputs['Name'])
        self.log.info("Create agent object for: %s", self.tcinputs['AgentName'])
        self._agent = self._client.agents.get(self.tcinputs['AgentName'])
        if self._agent is not None:
            # Create object of Instance, if instance name is provided in the JSON
            if 'InstanceName' in self.tcinputs:
                self.log.info("Create instance object for: %s", self.tcinputs['InstanceName'])
                self._instance = self._agent.instances.get(self.tcinputs['InstanceName'])
            # Create object of the Backupset class
            if 'BackupsetName' in self.tcinputs:
                self.log.info("Creating backupset object for: %s",
                              self.tcinputs['BackupsetName'])
                # If instance object is not initialized, then instantiate backupset object
                # from agent
                # Otherwise, instantiate the backupset object from instance
                if self._instance is None:
                    self._backupset = self._agent.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
                else:
                    self._backupset = self._instance.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
            # Create object of the Subclient class
            if 'SubclientName' in self.tcinputs:
                self.log.info("Creating subclient object for: %s",
                              self.tcinputs['SubclientName'])
                # If backupset object is not initialized, then try to instantiate subclient
                # object from instance
                # Otherwise, instantiate the subclient object from backupset
                if self._backupset is None:
                    if self._instance is None:
                        pass
                    else:
                        self._subclient = self._instance.subclients.get(
                            self.tcinputs['SubclientName']
                        )
                else:
                    self._subclient = self._backupset.subclients.get(
                        self.tcinputs['SubclientName']
                     )
        # Creating CloudConnector object
        self.cvcloud_object = CloudConnector(self)
        self.cvcloud_object.cvoperations.cleanup()

    @test_step
    def wait_until_discovery_task_completes(self):
        """Waits until the discovery process completes on the proxy"""
        self.machine = WindowsMachine(machine_name=self.instance.proxy_client,
                                      commcell_object=self.commcell)
        result = self.machine.wait_for_process_to_exit(
            process_name=o365_constants.OneDrive.DISCOVER_PROCESS_NAME.value,
            time_out=1800,
            poll_interval=60)
        if not result:
            raise Exception('Discover process did not complete in stipulated time')

    @test_step
    def get_group_members(self, group):
        """Get the members of a given group"""
        try:
            self.cvcloud_object.one_drive.discover_group_members(group)
            details = self.cvcloud_object.dbo.get_content(f'{group.lower()}_members_list')
            group_members = list()
            pattern = re.compile(self.regex)
            for detail in details:
                if pattern.search(detail['userPrincipalName']):
                    continue
                group_members.append(detail['userPrincipalName'])
            self.log.info(f'Members of group {group}: {group_members}')
            return group_members
        except Exception:
            raise CVTestStepFailure(f'Unable to obtain group member details for group {group}')

    def run(self):
        try:
            self.wait_until_discovery_task_completes()

            self.cvcloud_object.one_drive.launch_autodiscovery_via_cli(self.tcinputs['Name'])
            self.onedrive.verify_cache_is_fetched()
            self.wait_until_discovery_task_completes()

            # Auto-discovery will fetch new users in groups only if the
            # cache is 1 day old. Else, it will keep the existing cache.
            # So, here, to fetch new groups/users, we will forcibly refresh cache.
            self.onedrive.refresh_cache()
            self.wait_until_discovery_task_completes()
            self.onedrive.add_group(groups=[self.group_name])
            self.wait_until_discovery_task_completes()
            self.onedrive.verify_added_groups([self.group_name])
            members = self.get_group_members(self.group_name)
            self.onedrive.verify_group_members(members)

            self.cvcloud_object.one_drive.remove_group_member(self.users[0], self.group_name)
            self.cvcloud_object.one_drive.delete_user(self.users[1])
            self.cvcloud_object.one_drive.update_user_email_id(
                self.users[2], 'onedrive_automation_updated_email_id')
            self.cvcloud_object.one_drive.add_group_member(self.users[-1], self. group_name)
            new_user_name = f'onedrive_discover_automation_user{len(self.users) + 1}'
            self.cvcloud_object.one_drive.create_user(new_user_name)
            self.cvcloud_object.one_drive.add_group_member(new_user_name, self.group_name)
            self.cvcloud_object.dbo.delete_table(f'{self.group_name}_members_list')
            self.users.append('onedrive_automation_updated_email_id')
            self.users.append(new_user_name)
            self.users.remove(self.users[2])
            self.users.remove(self.users[1])

            self.onedrive.refresh_cache()
            self.wait_until_discovery_task_completes()
            self.onedrive.verify_added_groups([self.group_name])
            members = self.get_group_members(self.group_name)
            self.onedrive.verify_group_members(members)

        except Exception as err:
            handle_testcase_exception(self, err)

    def tear_down(self):
        try:
            if self.status == constants.PASSED:
                self.navigator.navigate_to_office365()
                self.onedrive.delete_office365_app(self.tcinputs['Name'])
                for user in self.users:
                    self.cvcloud_object.one_drive.delete_user(user)
                self.cvcloud_object.one_drive.delete_group(self.group_name)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
