# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase

from Web.Common.cvbrowser import BrowserFactory, Browser

from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.table import Rtable


from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception


class TestCase(CVTestCase):
    """Command Center: Table Component integration testcase"""

    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Integration of  React Table grid component in command center"
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.table = None
        self.group_name = 'master'
        self.role_name = "Master"
        self.group_name_header = 'Group name'

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(username=self.inputJSONnode['commcell']['commcellUsername'],
                                     password=self.inputJSONnode['commcell']['commcellPassword'])
            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_user_groups()
            self.table = Rtable(self.admin_console)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def get_number_of_columns(self):
        """gets number of columns present in react table"""
        self.navigator.navigate_to_user_groups()
        columns = self.table.get_number_of_columns()
        if not columns:
            raise CVTestStepFailure('unable to get columns')

    @test_step
    def get_visible_column_names(self):
        """Get visible Column names from react table"""
        self.navigator.navigate_to_user_groups()
        columns = self.table.get_visible_column_names()
        if self.group_name_header not in columns:
            raise CVTestStepFailure('Group Name not found in column names')

    @test_step
    def get_column_data(self):
        """Get column data from react table"""
        self.navigator.navigate_to_user_groups()
        column_data = self.table.get_column_data(self.group_name_header)
        if not column_data:
            raise CVTestStepFailure('unable to get data for Group name column')

    @test_step
    def access_action_item(self):
        """Selects the action item in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_action_item(self.group_name, 'Add users')
        # to do after modal dialog moved to react. For now handle it using admin console base class
        self.admin_console.switch_to_react_frame()
        # if self.modal_dialog.title() != 'Add users':
        #     raise CVTestStepFailure("Access action item failed. Action item [Add Users] is not clicked on local group")
        # self.modal_dialog.click_cancel()
        self.admin_console.click_button_using_text("Cancel")
        for handle in self.browser.driver.window_handles:
            self.browser.driver.switch_to_window(handle)

    @test_step
    def view_by_title(self):
        """View by title in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.view_by_title('Local group')
        if 'localUserGroups' not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                "View by title drop down failed, "
                "click on local group in dropdown didn't access local user groups page"
            )

    @test_step
    def access_toolbar_menu(self):
        """Tool bar menu in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_toolbar_menu('Add user group')
        if 'addUserGroup' not in self.browser.driver.current_url:
            raise CVTestStepFailure("Access tool bar menu didn't open Add user group page")

    @test_step
    def access_link(self):
        """Access the link from the react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_link(self.group_name)
        if 'userGroup/' not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                "access link failed, click on group name didn't access user group details page"
            )

    @test_step
    def access_link_by_column(self):
        """search by entity_name and access by link_text in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_link_by_column(
            self.group_name,
            self.group_name
        )
        if 'userGroup/' not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                "access link by column failed, click on group name didn't access user group details page"
            )

    @test_step
    def select_rows(self):
        """Select rows in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_link(self.group_name)
        table = Rtable(self.admin_console, title="Users")
        table.select_rows([self.inputJSONnode['commcell']['commcellUsername']])
        # if delete button is not present, then it will throw exception
        self.table.access_toolbar_menu("Remove")
        self.admin_console.wait_for_completion()
        self.admin_console.refresh_page()
        self.admin_console.wait_for_completion()

    @test_step
    def apply_filter_over_column(self):
        """apply filter on given column in react table"""
        self.navigator.navigate_to_user_groups()
        self.table.access_link(self.group_name)
        table = Rtable(self.admin_console, "Associated entities")
        table.apply_filter_over_column('Role', self.role_name)
        role_name = self.table.get_column_data('Role')
        if not role_name or role_name[0].lower() != self.group_name.lower():
            raise CVTestStepFailure(
                f"Filter on column failed, expected [{ self.role_name}],"
                f"received [{role_name[0]}]"
            )

    @test_step
    def clear_column_filter(self):
        """clear filter from column in react table"""

        table = Rtable(self.admin_console, "Associated entities")
        table.clear_column_filter('Role')
        role_name = self.table.get_column_data('Role')
        cleared = False
        for role in role_name:
            if role.lower() != self.role_name.lower():
                cleared = True
                break
        if not cleared:
            raise CVTestStepFailure("Clearing filter failed while removing from input on column Role")
        self.navigator.navigate_to_user_groups()

    @test_step
    def get_table_data(self):
        """get data from the react table"""
        self.navigator.navigate_to_user_groups()
        data = self.table.get_table_data()
        if not data:
            raise CVTestStepFailure(
                f"unable to get data from table"
            )

    @test_step
    def is_entity_present_in_column(self):
        """Check entity present in react table"""
        self.navigator.navigate_to_user_groups()
        if not self.table.is_entity_present_in_column(self.group_name_header, self.group_name):
            raise CVTestStepFailure('unable to check master entity present in column')

    @test_step
    def get_total_rows_count(self):
        """gets total rows count from react table"""
        self.navigator.navigate_to_user_groups()
        count = self.table.get_total_rows_count()
        if count == 0:
            raise CVTestStepFailure("Get total rows count failed with zero count")

    @test_step
    def access_menu_from_dropdown(self):
        """access more menu from dropdown in react table"""
        self.navigator.navigate_to_users()
        self.table.access_menu_from_dropdown('Single user')
        if 'addUser' not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                "Access menu frrom dropdown failed while selecting add-> 'Single user' on Users page")

    @test_step
    def display_hidden_column(self):
        """displays the hidden column in react table"""
        self.navigator.navigate_to_users()
        self.table.display_hidden_column("Enabled")
        columns = self.table.get_visible_column_names()
        if 'Enabled' not in columns:
            raise CVTestStepFailure("Display hidden column failed."
                                    "column with name 'Enabled' is not displayed on Users page")

    @test_step
    def apply_sort_over_column(self):
        """applies sorting order on react table column"""
        self.navigator.navigate_to_users()
        before_sort = self.table.get_column_data('User name')
        self.table.apply_sort_over_column('User name', ascending=False)
        after_sort = self.table.get_column_data('User name')
        if before_sort == after_sort:
            raise CVTestStepFailure(
                "Apply sort over column failed, as User name column data matches before & after sort")

    @test_step
    def get_grid_action_list(self):
        """gets grid action list available in react table"""
        self.navigator.navigate_to_user_groups()
        action_list = self.table.get_grid_actions_list(self.group_name)
        if len(action_list) == 0:
            raise CVTestStepFailure("Get grid action list failed, returned zero for row in Users page React Table")

    @test_step
    def get_all_column_names(self):
        """gets all column names available in react table"""
        self.navigator.navigate_to_users()
        visible_items = self.table.get_visible_column_names()
        invisible_items = self.table.get_all_column_names()
        if len(invisible_items) == len(visible_items):
            raise CVTestStepFailure("Get all columnm names failed, Didn't fetch hidden column in Users page")

    def run(self):
        """Test case run function"""
        try:

            self.init_tc()
            self.get_number_of_columns()
            self.get_visible_column_names()
            self.get_column_data()
            self.view_by_title()
            self.get_table_data()
            self.is_entity_present_in_column()
            self.select_rows()
            self.access_action_item()
            self.access_link()
            self.access_link_by_column()
            #
            # Please don't change this order as both are dependent
            self.apply_filter_over_column()
            self.clear_column_filter()
            #
            self.access_toolbar_menu()
            self.get_total_rows_count()
            self.access_menu_from_dropdown()
            self.display_hidden_column()
            self.apply_sort_over_column()
            self.get_grid_action_list()
            self.get_all_column_names()

        except Exception as err:
            handle_testcase_exception(self, err)
        finally:
            print("")
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
