# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""

from Application.Sharepoint.sharepoint_online import SharePointOnline
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """Class for executing the test case of Office365- SharePoint Online - Auto Association - Basic case"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Office365- SharePoint Online - Auto Association - Basic case"
        self.sp_client_object = None
        self.tcinputs = {
            "PseudoClientName": None,
            "ServerPlanName": None,
            "IndexServer": None,
            "AccessNodes": None,
            "GlobalAdministrator": None,
            "GlobalAdministrator Password": None,
            "AzureAppId": None,
            "AzureAppKeyValue": None,
            "AzureDirectoryId": None,
            "MachineName": None,
            "GraphClientId": None,
            "GraphClientSecret": None,
            "GraphTenantId": None,
            "SiteUrlList": None,
            "Office365Plan": None,
            "ClientId": None,
            "ClientSecret": None
        }
        self.subsites_metadata = {}
        self.group_names = None

    def init_tc(self):
        """ Initialization function for the test case. """
        try:
            self.log.info('Creating SharePoint client object.')
            self.sp_client_object = SharePointOnline(self)

            self.sp_client_object.pseudo_client_name = self.tcinputs.get("PseudoClientName")
            self.sp_client_object.server_plan = self.tcinputs.get('ServerPlanName')
            self.sp_client_object.office_365_plan = [(self.tcinputs.get('Office365Plan'),
                                                      int(self.sp_client_object.cvoperations.get_plan_obj
                                                          (self.tcinputs.get('Office365Plan')).plan_id))]

            # Infrastructure details
            self.sp_client_object.index_server = self.tcinputs.get('IndexServer')
            self.sp_client_object.access_nodes_list = self.tcinputs.get('AccessNodes')

            # SharePoint details
            self.sp_client_object.global_administrator = self.tcinputs.get('GlobalAdministrator')
            self.sp_client_object.global_administrator_password = self.tcinputs.get('GlobalAdministrator Password')

            # Azure Details
            self.sp_client_object.azure_app_id = self.tcinputs.get('AzureAppId', "")
            self.sp_client_object.azure_app_key_id = self.tcinputs.get('AzureAppKeyValue', "")
            self.sp_client_object.azure_directory_id = self.tcinputs.get('AzureDirectoryId', "")

            # Graph API Credentials
            self.sp_client_object.graph_client_id = self.tcinputs.get("GraphClientId", "")
            self.sp_client_object.graph_client_secret = self.tcinputs.get("GraphClientSecret", "")
            self.sp_client_object.graph_tenant_id = self.tcinputs.get("GraphTenantId", "")

            # SharePoint Site and Azure App authentication details
            self.sp_client_object.site_url_list = self.tcinputs.get("SiteUrlList", "")
            self.sp_client_object.api_client_id = self.tcinputs.get("ClientId", "")
            self.sp_client_object.api_client_secret = self.tcinputs.get("ClientSecret", "")
            self.log.info('SharePoint client object created.')

        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def clean_up_sites(self):
        """Cleans up the test sites used in test case"""
        try:
            for site in self.sp_client_object.site_url_list:
                self.sp_client_object.site_url = site
                if self.sp_client_object.get_site_properties("subsite_1"):
                    self.log.info("site is present in SharePoint site")
                    self.sp_client_object.delete_subsite("subsite_1")
        except Exception as exception:
            self.log.exception(f"Exception while deleting sites: %s", str(exception))
            raise exception

    def create_subsite(self, web_template):
        """Creates a subsite
        """
        try:
            self.log.info("Creating a sub site")
            title = "Test Subsite - 1"
            url_end = "subsite_1"
            response = self.sp_client_object.create_subsite(title, url_end, web_template)
            self.subsites_metadata[response.get('ServerRelativeUrl')] = {
                'Url End': response.get('ServerRelativeUrl').split("/")[-1],
                'Title': response.get('Title', ""),
                'Operation': "ADDED",
                'Associated Flags Value': "4",
                'Office 365 Plan Id': self.sp_client_object.office_365_plan[0][1]
            }
            return response.get('ServerRelativeUrl')
        except Exception as exception:
            self.log.exception("Exception while making site level changes: %s", str(exception))
            raise exception

    def validate_group_association(self, group_name, site_url, web_template, total_num_of_sites):
        """Validates auto association group"""
        try:
            self.sp_client_object.cvoperations.configure_group_for_backup(group_name,
                                                                          self.sp_client_object.office_365_plan[0][1])
            self.sp_client_object.cvoperations.update_auto_association_group_properties(
                association_group_name=group_name,
                account_status=0,
                office_365_plan_id=self.sp_client_object.office_365_plan[0][1])
            site_dict, no_of_records = self.sp_client_object.cvoperations.check_sites_under_add_webs(discovery_type=6)
            self.log.info(f"Total number of sites from api= {total_num_of_sites}")
            self.log.info(f"Total number of sites associated under {group_name} group = {no_of_records}")
            if group_name == "All Web Sites":
                if no_of_records - total_num_of_sites > 10 or no_of_records == 0:
                    raise Exception(f"All sites are not associated with {group_name} group")
                else:
                    total_num_of_sites = no_of_records
                    self.log.info(f"All sites are associated with {group_name} group")
            else:
                if total_num_of_sites == no_of_records:
                    self.log.info(f"All sites are associated with {group_name} group")
                else:
                    raise Exception(f"All sites are not associated with {group_name} group")
            self.sp_client_object.site_url = site_url
            subsite = self.create_subsite(web_template)
            self.sp_client_object.cvoperations.run_manual_discovery()
            self.sp_client_object.cvoperations.wait_for_manual_discovery_to_complete()
            self.sp_client_object.validate_subsite_properties(self.subsites_metadata[subsite])
            total_num_of_sites = total_num_of_sites + 1
            site_dict, no_of_records = self.sp_client_object.cvoperations.check_sites_under_add_webs(discovery_type=6)
            self.log.info(f"Total number of sites from api= {total_num_of_sites}")
            self.log.info(f"Total number of sites associated under group = {no_of_records}")
            if total_num_of_sites == no_of_records:
                self.log.info(f"New subsite is associated with {group_name} group")
            else:
                raise Exception(f"New subsite is not associated with {group_name} group")
            self.sp_client_object.cvoperations.update_auto_association_group_properties(
                association_group_name=group_name, account_status=1)
            # There is an MR for removal of auto associated group.
            # The below lines will be uncommented once the MR is resolved
            # site_dict, no_of_records = self.sp_client_object.cvoperations.check_sites_under_add_webs(discovery_type=6)
            # if no_of_records != 0:
            #     raise Exception(f"All sites are not dis-associated for {group_name} group")
        except Exception as exception:
            self.log.exception("Exception while validating all webs group: %s", str(exception))
            raise exception

    def setup(self):
        """Setup function of this test case"""
        self.init_tc()
        self.clean_up_sites()
        self.sp_client_object.cvoperations.add_share_point_pseudo_client()
        self.log.info("Pseudo Client has been created successfully")

    def run(self):
        """Run function of this test case"""
        try:
            self.sp_client_object.cvoperations.run_manual_discovery()
            self.sp_client_object.cvoperations.wait_for_manual_discovery_to_complete()
            total_num_of_sites = self.sp_client_object.get_group_based_sites_count()
            self.group_names = ["All Web Sites", "All Groups And Teams Sites", "All Project Online Sites"]
            # There is an MR for removal of auto associated group. This line is removed once the MR is resolved
            total_num_of_sites["All Groups And Teams Sites"] = total_num_of_sites["All Groups And Teams Sites"] + 1
            total_num_of_sites["All Project Online Sites"] = total_num_of_sites["All Project Online Sites"] + 2
            # --------------------------------------------------------------------------------------------------
            web_templates = ['GROUP', 'STS', 'STS']
            for index in range(len(self.sp_client_object.site_url_list)):
                self.validate_group_association(self.group_names[index], self.sp_client_object.site_url_list[index],
                                                web_templates[index], total_num_of_sites[self.group_names[index]])
        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.status != constants.FAILED:
            self.clean_up_sites()
            self.log.info("All test sites are deleted successfully")
            self.sp_client_object.cvoperations.delete_share_point_pseudo_client\
                (self.sp_client_object.pseudo_client_name)
            self.log.info("Pseudo Client has been deleted successfully")
