# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: This test case verifies whether Azure replication group creation succeeds with default options.
Then it waits for the replication and backup jobs to complete successfully, and finally verifies the details on the
replication monitor

Main file for executing this test case(verify the creation and updation of Azure recovery target
of replication type)

TestCase: Class for executing this test case
Sample JSON: {
        "ClientName": "Azure",
        "source_vms": ["auto_vm"],
        "recovery_target": "target",
        "storage_name": "storage",
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.VMHelpers.AzureVM import AzureVM
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the Azure phase 2 automation TC"""
    test_step = TestStep()
    _AGENT_NAME = 'virtual server'
    _INSTANCE_NAME = 'azure resource manager'
    _BACKUPSET_NAME = 'defaultBackupSet'

    def __init__(self):
        """Initialises the objects and TC inputs"""
        CVTestCase.__init__(self)
        self.name = "Verify deploy only during failover and verify failover and failback"

        self.tcinputs = {
            "ClientName": None,
            "source_vms": [],
            "recovery_target": None,
            "storage_name": None,
        }

        self.client = None
        self.agent = None
        self.instance = None
        self.backupset = None
        self.subclient = None

        self.utils = None
        self.browser = None
        self.admin_console = None

        self.replication_group = None
        self.group_details = None

        self.target_details = None

        self._source_subclient = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    @property
    def schedule(self):
        """Returns the name of the schedule"""
        return self.replication_group.get_schedule_name_by_replication_group(self.group_name)

    @property
    def source_subclient(self):
        """Returns the VSA Auto subclient after initialising all the testcase objects"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        if not self._source_subclient:
            self._source_subclient = VirtualServerUtils.subclient_initialize(self)
        return self._source_subclient

    @property
    def destination_vm_names(self):
        """Gets the destination VM name from the source subclient"""
        return self.source_subclient.get_live_sync_destination_vms(self.schedule)

    @property
    def backup_job_id(self):
        """Get the latest backup job obj"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        return self.subclient.find_latest_job().job_id

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.admin_console.wait_for_completion()

        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)

    def logout(self):
        """Logs out of the admin console and closes the browser"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)

            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    def get_recovery_target_details(self):
        """Gets the recovery target details for verification"""
        self.admin_console.navigator.navigate_to_replication_targets()
        recovery_target = RecoveryTargets(self.admin_console).access_target(
            self.tcinputs['recovery_target']
        )
        self.target_details = recovery_target.get_target_summary()

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The repication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        azure_configure = self.replication_group.configure_azure()
        azure_configure.content.set_name(self.group_name)
        for vm_name in self.tcinputs['source_vms']:
            azure_configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                               {"VMs": [vm_name]})
        sleep(5)
        azure_configure.next()

        azure_configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        azure_configure.target.deploy_vm_during_failover(True)
        azure_configure.target.unconditionally_overwrite_vm(True)
        azure_configure.next()

        azure_configure.storage_cache.select_storage(self.tcinputs['storage_name'])
        azure_configure.next()
        azure_configure.next()

        sleep(5)
        azure_configure.finish()

    @test_step
    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        self.log.info('Waiting for backup job %s to finish', self.backup_job_id)
        job_obj = self.commcell.job_controller.get(self.backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_job_completion(self):
        """Waits for replication job to complete and then verify its details"""
        self.log.info("Waiting 2 minutes to let live sync update")
        sleep(120)
        live_sync_utils = LiveSyncUtils(self.source_subclient, self.schedule)
        job_obj = live_sync_utils.get_recent_replication_job(self.backup_job_id)
        self.log.info("Waiting for replication job id %s to finish", job_obj.job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_non_failover_vm(self):
        """Verifies that the VM does not exist without failover being passed"""
        self.source_subclient.hvobj.get_access_token()
        self.source_subclient.hvobj.collect_all_vm_data()
        current_vms = [vm_entry['name'] for vm_entry in
                       self.source_subclient.hvobj.all_vmdata[self.target_details['Resource group']]['value']]
        for vm_name in self.destination_vm_names:
            if vm_name in current_vms:
                raise CVTestStepFailure(f"The VM [{vm_name}] already exists in the deployed VMs without failover")

    @test_step
    def perform_failover(self):
        """Performs the failover of the VM group VMs after replication is over"""
        self.login()
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)
        for vm_name in self.tcinputs['source_vms']:
            job_id = self.group_details.overview.unplanned_failover(vm_name)
            job_obj = self.commcell.job_controller.get(job_id)
            job_obj.wait_for_completion()
            self.admin_console.refresh_page()

    @test_step
    def perform_failback(self):
        """Performs the failback of the VM group VMs"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)
        for vm_name in self.tcinputs['source_vms']:
            job_id = self.group_details.overview.failback(vm_name)
            job_obj = self.commcell.job_controller.get(job_id)
            job_obj.wait_for_completion()
            self.admin_console.refresh_page()

    @test_step
    def verify_failover_vm(self):
        """Verifies VM exist after failover"""
        self.source_subclient.hvobj.get_access_token()
        self.source_subclient.hvobj.collect_all_vm_data()
        current_vms = [vm_entry['name'] for vm_entry in
                       self.source_subclient.hvobj.all_vmdata[self.target_details['Resource group']]['value']]
        for vm_name in self.destination_vm_names:
            if vm_name not in current_vms:
                raise CVTestStepFailure(f"The VM [{vm_name}] has not been deployed after failover")

        for vm_name in self.destination_vm_names:
            azure_vm = AzureVM(self.source_subclient.hvobj, vm_name)
            azure_vm.power_on()
            azure_vm.power_off()

    def run(self):
        """Runs the testcase in order"""
        try:
            self.get_recovery_target_details()
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()

            self.logout()
            self.verify_backup_job_completion()
            self.verify_replication_job_completion()

            self.verify_non_failover_vm()
            self.perform_failover()
            self.verify_failover_vm()

            self.perform_failback()
            self.verify_non_failover_vm()

            self.delete_virtualization_group(wait_for_sync=False)

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Tears down the TC"""
        self.logout()
