# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""

from Application.Sharepoint.sharepoint_online import SharePointOnline
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.exceptions import CVTestCaseInitFailure
from Web.Common.page_object import TestStep
from Application.Sharepoint.data_generation import TestData
from Application.Sharepoint.restore_options import Restore


class TestCase(CVTestCase):
    """Class for executing the test case of Office365- SharePoint V2 - Site level Restore
    """

    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Office365- SharePoint Online- Site level Restore"
        self.sp_client_object = None
        self.tcinputs = {
            "PseudoClientName": None,
            "ServerPlanName": None,
            "IndexServer": None,
            "AccessNodes": None,
            "TenantUrl": None,
            "Username": None,
            "Password": None,
            "AzureUserName": None,
            "AzureSecret": None,
            "SiteUrl": None,
            "Office365Plan": None,
            "ClientId": None,
            "ClientSecret": None
        }
        self.testdata = None
        self.share_point_data_flag = False
        self.restore_obj = None
        self.site_url_list = []
        self.subsite_end_url_list = []
        self.subsite_title_list = []

    def init_tc(self):
        """ Initialization function for the test case. """
        try:
            self.log.info('Creating SharePoint client object.')
            self.sp_client_object = SharePointOnline(self)

            self.sp_client_object.pseudo_client_name = self.tcinputs.get("PseudoClientName")
            self.sp_client_object.server_plan = self.tcinputs.get('ServerPlanName')
            self.sp_client_object.office_365_plan = [(self.tcinputs.get('Office365Plan'),
                                                      int(self.sp_client_object.cvoperations.get_plan_obj
                                                          (self.tcinputs.get('Office365Plan')).plan_id))]
            # Infrastructure details
            self.sp_client_object.index_server = self.tcinputs.get('IndexServer')
            self.sp_client_object.access_nodes_list = self.tcinputs.get('AccessNodes')

            self.sp_client_object.azure_username = self.tcinputs.get('AzureUserName', "")
            self.sp_client_object.azure_secret = self.tcinputs.get('AzureSecret', "")

            # SharePoint Details
            self.sp_client_object.tenant_url = self.tcinputs.get("TenantUrl")
            self.sp_client_object.user_username = self.tcinputs.get("Username")
            self.sp_client_object.user_password = self.tcinputs.get("Password")

            # SharePoint Site details
            self.sp_client_object.site_url = self.tcinputs.get("SiteUrl", "")
            self.sp_client_object.api_client_id = self.tcinputs.get("ClientId", "")
            self.sp_client_object.api_client_secret = self.tcinputs.get("ClientSecret", "")
            self.log.info('SharePoint client object created.')
            self.testdata = TestData(self.sp_client_object)
            self.restore_obj = Restore(self.sp_client_object)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def validate_subsites_titles(self):
        """Validates proeprties of subsites"""
        try:
            self.sp_client_object.site_url = self.tcinputs.get("SiteUrl", "")
            for subsite_title, subsite_end_url in zip(self.subsite_title_list, self.subsite_end_url_list):
                site_properties = self.sp_client_object.get_site_properties(subsite_end_url,
                                                                            additional_uri="Title")
                self.log.info(f"Restored title for {subsite_end_url} is {site_properties.get('Title', '')}")
                if site_properties.get("Title", "") == subsite_title:
                    self.log.info(f"{subsite_end_url} has title {subsite_title} as expected")
                else:
                    self.log.exception(f"{subsite_end_url} is not restored properly")
                    raise Exception(f"{subsite_end_url} is not restored properly")
        except Exception as exception:
            self.log.exception("Exception while validating subsite properties: %s", str(exception))
            raise exception

    def setup(self):
        """Setup function of this test case"""
        self.init_tc()
        self.site_url_list, self.subsite_end_url_list, self.subsite_title_list = self.testdata.create_initial_subsites()
        self.testdata.create_site_structure_for_backup(site_url_list=self.site_url_list,
                                                       folder=True, list=True)
        self.sp_client_object.site_url = self.tcinputs.get("SiteUrl", "")

    def run(self):
        """Run function of this test case"""
        try:
            self.sp_client_object.cvoperations.add_share_point_pseudo_client()
            self.sp_client_object.cvoperations.browse_for_sp_sites()
            self.sp_client_object.cvoperations.associate_content_for_backup(
                self.sp_client_object.office_365_plan[0][1])
            self.sp_client_object.cvoperations.run_backup()
            self.sp_client_object.cvoperations.run_backup()
            subsite_browse_list = []
            for site in self.subsite_end_url_list:
                subsite_browse_list.append(self.sp_client_object.site_url + "\\Subsites\\" + site)
            self.restore_obj.browse_restore_content_and_verify_browse_response(
                browse_paths=subsite_browse_list)
            self.testdata.delete_backup_content(folder=True, list=True)
            self.sp_client_object.delete_subsites(self.subsite_end_url_list)
            self.restore_obj.restore_and_validate_sharepoint_content(restore_args={
                "paths": ["MB\\" + self.sp_client_object.site_url],
                "overwrite": True
            }, folder=True, list=True,
                site_url_list=self.site_url_list)
            self.validate_subsites_titles()
        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.status != constants.FAILED:
            self.sp_client_object.site_url = self.tcinputs.get("SiteUrl", "")
            self.sp_client_object.delete_subsites(self.subsite_end_url_list)
            self.log.info("Cleaned up all test sub sites")
            self.testdata.delete_backup_site_structure(folder=True, list=True)
            self.testdata.delete_disk_files()
            self.sp_client_object.cvoperations.delete_share_point_pseudo_client \
                (self.sp_client_object.pseudo_client_name)
            self.log.info("Pseudo Client has been deleted successfully")
