# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test Case for validation of group discovery on creation of OneDrive v2 client

TestCase:   Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup the requirements for the test case

    run()           --  run function of this test case

"""
import re
import time

from Application.CloudApps.cloud_connector import CloudConnector
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.windows_machine import WindowsMachine
from Web.AdminConsole.Office365Pages import constants as o365_constants
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Office365Pages.onedrive import OneDrive
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        """Initializes testcase class object"""
        super(TestCase, self).__init__()
        self.testcaseutils = CVTestCase
        self.name = "OneDrive v2: User Group Discovery on Client creation"
        self.regex = ('^CommvaultAutoGenerated.*|^CommvaultSP.*|^CommvaultEX.*|^CVEXBackupAccount.*'
                      '|^CVAzureBackupAccount.*|^CV.*BackupAccount.*|^\\w*#EXT#*')
        self.browser = None
        self.groups = None
        self.plan = None
        self.navigator = None
        self.admin_console = None
        self.onedrive = None
        self.machine = None

    def setup(self):
        """Initial configuration for the testcase."""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(
                self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(
                self.inputJSONnode['commcell']['commcellUsername'],
                self.inputJSONnode['commcell']['commcellPassword'])

            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_office365()
            self.tcinputs['Name'] += str(int(time.time()))
            self.groups = self.tcinputs['Groups']
            self.plan = self.tcinputs['Office365Plan']

            self.log.info("Creating an object for office365 helper")
            self.tcinputs['office_app_type'] = OneDrive.AppType.one_drive_for_business
            self.onedrive = OneDrive(self.tcinputs, self.admin_console)

            self.onedrive.create_office365_app()
            self._initialize_sdk_objects()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def _initialize_sdk_objects(self):
        """Initializes the sdk objects after app creation"""
        self.commcell.refresh()
        self.log.info("Create client object for: %s", self.tcinputs['Name'])
        self._client = self.commcell.clients.get(self.tcinputs['Name'])
        self.log.info("Create agent object for: %s", self.tcinputs['AgentName'])
        self._agent = self._client.agents.get(self.tcinputs['AgentName'])
        if self._agent is not None:
            # Create object of Instance, if instance name is provided in the JSON
            if 'InstanceName' in self.tcinputs:
                self.log.info("Create instance object for: %s", self.tcinputs['InstanceName'])
                self._instance = self._agent.instances.get(self.tcinputs['InstanceName'])
            # Create object of the Backupset class
            if 'BackupsetName' in self.tcinputs:
                self.log.info("Creating backupset object for: %s",
                              self.tcinputs['BackupsetName'])
                # If instance object is not initialized, then instantiate backupset object
                # from agent
                # Otherwise, instantiate the backupset object from instance
                if self._instance is None:
                    self._backupset = self._agent.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
                else:
                    self._backupset = self._instance.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
            # Create object of the Subclient class
            if 'SubclientName' in self.tcinputs:
                self.log.info("Creating subclient object for: %s",
                              self.tcinputs['SubclientName'])
                # If backupset object is not initialized, then try to instantiate subclient
                # object from instance
                # Otherwise, instantiate the subclient object from backupset
                if self._backupset is None:
                    if self._instance is None:
                        pass
                    else:
                        self._subclient = self._instance.subclients.get(
                            self.tcinputs['SubclientName']
                        )
                else:
                    self._subclient = self._backupset.subclients.get(
                        self.tcinputs['SubclientName']
                    )
        # Creating CloudConnector object
        self.cvcloud_object = CloudConnector(self)
        self.cvcloud_object.cvoperations.cleanup()

    @test_step
    def wait_until_discovery_task_completes(self):
        """Waits until the discovery process completes on the proxy"""
        try:
            self.machine = WindowsMachine(machine_name=self.instance.proxy_client,
                                          commcell_object=self.commcell)
            result = self.machine.wait_for_process_to_exit(
                process_name=o365_constants.OneDrive.DISCOVER_PROCESS_NAME.value,
                time_out=1200,
                poll_interval=60)
            if not result:
                raise Exception('Discover process did not complete in stipulated time')
        except Exception:
            raise CVTestStepFailure('Discover process did not complete in stipulated time')

    @test_step
    def get_group_members(self, group):
        """Get the members of a given group"""
        try:
            self.cvcloud_object.one_drive.discover_group_members(group_name=group)
            details = self.cvcloud_object.dbo.get_content(f'{group.lower()}_members_list')
            group_members = list()
            pattern = re.compile(self.regex)
            for detail in details:
                if pattern.search(detail['userPrincipalName']):
                    continue
                group_members.append(detail['userPrincipalName'])
            return group_members
        except Exception:
            raise CVTestStepFailure(f'Unable to obtain group member details for group {group}')

    def run(self):
        try:
            self.onedrive.add_group()
            self.wait_until_discovery_task_completes()
            self.onedrive.verify_added_groups()
            for group in self.groups:
                members = self.get_group_members(group)
                self.onedrive.verify_group_members(members)
        except Exception as err:
            handle_testcase_exception(self, err)

    def tear_down(self):
        try:
            if self.status == constants.PASSED:
                self.navigator.navigate_to_office365()
                self.onedrive.delete_office365_app(self.tcinputs['Name'])
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
