# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test Case for validation of user discovery on creation of
OneDrive v2 client and verification of Office 365 Plan association

TestCase:   Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup the requirements for the test case

    run()           --  run function of this test case

"""


import datetime
import re
import time

from Application.CloudApps.cloud_connector import CloudConnector
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.windows_machine import WindowsMachine
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Office365Pages import constants as o365_constants
from Web.AdminConsole.Office365Pages.onedrive import OneDrive
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        """Initializes testcase class object"""
        super(TestCase, self).__init__()
        self.testcaseutils = CVTestCase
        self.name = "Onedrive v2: User Discovery on Client Creation and Office 365 Plan Association"
        self.regex = ('^CommvaultAutoGenerated.*|^CommvaultSP.*|^CommvaultEX.*|^CVEXBackupAccount.*'
                      '|^CVAzureBackupAccount.*|^CV.*BackupAccount.*|^\\w*#EXT#*')
        self.browser = None
        self.cvcloud_object = None
        self.navigator = None
        self.admin_console = None
        self.onedrive = None
        self.users = None
        self.groups = None
        self.plan = None
        self.new_plan = None
        self.new_user = None
        self.new_group = None
        self.ui_user_count = None
        self.ui_group_count = None
        self.cache_update_time = None
        self.machine = None

    def setup(self):
        """Initial configuration for the testcase."""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(
                self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(
                self.inputJSONnode['commcell']['commcellUsername'],
                self.inputJSONnode['commcell']['commcellPassword'])

            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_office365()
            self.tcinputs['Name'] += str(int(time.time()))
            self.users = self.tcinputs['Users'].split(",")
            self.groups = self.tcinputs['Groups']
            self.plan = self.tcinputs['Office365Plan']
            self.new_plan = self.tcinputs['NewOffice365Plan']
            self.new_user = self.tcinputs['NewUser']
            self.new_group = self.tcinputs['NewGroup']

            self.log.info("Creating an object for office365 helper")
            self.tcinputs['office_app_type'] = OneDrive.AppType.one_drive_for_business
            self.onedrive = OneDrive(self.tcinputs, self.admin_console)

            self.onedrive.create_office365_app()
            self._initialize_sdk_objects()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def _initialize_sdk_objects(self):
        """Initializes the sdk objects after app creation"""
        self.commcell.refresh()
        self.log.info("Create client object for: %s", self.tcinputs['Name'])
        self._client = self.commcell.clients.get(self.tcinputs['Name'])
        self.log.info("Create agent object for: %s", self.tcinputs['AgentName'])
        self._agent = self._client.agents.get(self.tcinputs['AgentName'])
        if self._agent is not None:
            # Create object of Instance, if instance name is provided in the JSON
            if 'InstanceName' in self.tcinputs:
                self.log.info("Create instance object for: %s", self.tcinputs['InstanceName'])
                self._instance = self._agent.instances.get(self.tcinputs['InstanceName'])
            # Create object of the Backupset class
            if 'BackupsetName' in self.tcinputs:
                self.log.info("Creating backupset object for: %s",
                              self.tcinputs['BackupsetName'])
                # If instance object is not initialized, then instantiate backupset object
                # from agent
                # Otherwise, instantiate the backupset object from instance
                if self._instance is None:
                    self._backupset = self._agent.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
                else:
                    self._backupset = self._instance.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
            # Create object of the Subclient class
            if 'SubclientName' in self.tcinputs:
                self.log.info("Creating subclient object for: %s",
                              self.tcinputs['SubclientName'])
                # If backupset object is not initialized, then try to instantiate subclient
                # object from instance
                # Otherwise, instantiate the subclient object from backupset
                if self._backupset is None:
                    if self._instance is None:
                        pass
                    else:
                        self._subclient = self._instance.subclients.get(
                            self.tcinputs['SubclientName']
                        )
                else:
                    self._subclient = self._backupset.subclients.get(
                        self.tcinputs['SubclientName']
                     )
        # Creating CloudConnector object
        self.cvcloud_object = CloudConnector(self)
        self.cvcloud_object.cvoperations.cleanup()

    def get_regex_users_count(self):
        """Method to filter the users in AD with regex pattern
            Returns:
                results (list) -- Users in AD matching with regex pattern
        """
        results = []
        files_list = self.cvcloud_object.dbo.get_content('users_list')
        self.log.info(f'Users table: {files_list}')
        pattern = re.compile(self.regex)
        self.log.info(f'pattern: {pattern}')
        for user in files_list:
            if pattern.search(user['userPrincipalName']):
                results.append(user)
        self.log.info('Total number of users matching with Regex pattern : [{0}]'.format(len(results)))
        return len(results)

    @test_step
    def verify_user_discovery(self):
        """Verifying that discovery ran and the correct number of users were discovered"""
        try:
            self.ui_user_count, self.ui_group_count, self.cache_update_time = (
                self.onedrive.get_onedrive_discovery_count())

            self.cvcloud_object.one_drive.discover()
            total_users_count = len(self.cvcloud_object.dbo.get_content('users_list'))
            regex_count = self.get_regex_users_count()
            if self.ui_user_count != (total_users_count - regex_count):
                raise Exception(f'{self.ui_user_count} users discovered from AC and '
                                f'{total_users_count - regex_count} discovered using Graph API')
        except Exception:
            raise CVTestStepFailure(
                f"Unable to obtain count of users. Discovery unsuccessful.")

    @test_step
    def verify_group_discovery(self):
        """Verifying that discovery ran and the correct number of groups were discovered"""
        try:
            self.cvcloud_object.one_drive.discover_groups()
            total_groups_count = len(self.cvcloud_object.dbo.get_content('groups_list'))
            if self.ui_group_count != total_groups_count:
                raise Exception(f'{self.ui_group_count} users discovered from AC and '
                                f'{total_groups_count} discovered using Graph API')
        except Exception:
            raise CVTestStepFailure(
                f"Unable to obtain count of groups. Discovery unsuccessful.")

    @test_step
    def wait_until_discovery_task_completes(self):
        """Waits until the discovery process completes on the proxy"""
        self.machine = WindowsMachine(machine_name=self.instance.proxy_client,
                                      commcell_object=self.commcell)
        result = self.machine.wait_for_process_to_exit(
            process_name=o365_constants.OneDrive.DISCOVER_PROCESS_NAME.value,
            time_out=1200,
            poll_interval=60)
        if not result:
            raise Exception('Discover process did not complete in stipulated time')

    @test_step
    def get_group_members(self, group):
        """Get the members of a given group"""
        try:
            self.cvcloud_object.one_drive.discover_group_members(group_name=group)
            details = self.cvcloud_object.dbo.get_content(f'{group.lower()}_members_list')
            group_members = list()
            pattern = re.compile(self.regex)
            for detail in details:
                if pattern.search(detail['userPrincipalName']):
                    continue
                group_members.append(detail['userPrincipalName'])
            self.log.info(f'Members of group {group}: {group_members}')
            return group_members
        except Exception:
            raise CVTestStepFailure(f'Unable to obtain group member details for group {group}')

    def run(self):
        try:
            # On client creation, verify discovery is launched and users are discovered
            self.verify_user_discovery()

            # Add 2 users and verify plan
            self.onedrive.add_user()
            self.onedrive.verify_added_users()
            self.onedrive.verify_plan_association()

            # Add 1 more user and verify plan
            self.onedrive.add_user(users=[self.new_user], plan=self.new_plan)
            self.onedrive.verify_added_users(users=[self.new_user])
            self.onedrive.verify_plan_association(users=[self.new_user], plan=self.new_plan)

            # Verify cache update time for users
            cache_update_time = self.cvcloud_object.sqlite.get_last_cache_update_time()
            self.onedrive.verify_cache_update_time(self.cache_update_time, cache_update_time)

            # Change plan for 1 user and verify that plan is changed
            self.onedrive.change_office365_plan(self.users[0], self.new_plan)
            self.onedrive.verify_plan_association([self.users[0]], self.new_plan)

            # Verify that groups are discovered
            self.verify_group_discovery()

            # Add 2 groups, verify members and plan association
            self.onedrive.add_group()
            self.wait_until_discovery_task_completes()
            self.onedrive.verify_added_groups()
            self.onedrive.verify_plan_association(is_group=True)
            for group in self.groups:
                members = self.get_group_members(group)
                self.onedrive.verify_group_members(members)
                self.onedrive.verify_plan_association(members)

            # Add 1 more group, verify members and plan association
            self.onedrive.add_group([self.new_group], self.new_plan)
            self.wait_until_discovery_task_completes()
            self.onedrive.verify_added_groups([self.new_group])
            self.onedrive.verify_plan_association([self.new_group], self.new_plan, is_group=True)
            members = self.get_group_members(self.new_group)
            self.onedrive.verify_group_members(members)
            self.onedrive.verify_plan_association(members, self.new_plan)

            # Change plan for 1 group and verify that plan is changed for all group members as well
            self.onedrive.change_office365_plan(self.groups[0], self.new_plan, is_group=True)
            self.onedrive.verify_plan_association([self.groups[0]], self.new_plan, is_group=True)
            members = self.get_group_members(self.groups[0])
            self.onedrive.verify_plan_association(members, self.new_plan)

            # Verify cache update time for group
            cache_update_time = self.cvcloud_object.sqlite.get_last_cache_update_time(True)
            self.onedrive.verify_cache_update_time(self.cache_update_time, cache_update_time)

        except Exception as err:
            handle_testcase_exception(self, err)

    def tear_down(self):
        try:
            if self.status == constants.PASSED:
                self.navigator.navigate_to_office365()
                self.onedrive.delete_office365_app(self.tcinputs['Name'])
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
