# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                                   -- Initializes TestCase class

    generate_sensitive_data()                    -- Generates sensitive files with PII entities

    get_sensitive_file_details()                 -- Gets a sensitive file with an entity

    delete_files_in_folder()                     -- Traverses the folder to delete all the files present

    init_tc()                                    -- Initial configuration for the testcase

    create_inventory()                           -- Create an inventory with a nameserver

    create_plan()                                -- Creates a plan

    create_sda_project()                         -- Creates a project and runs analysis

    review_move_action_filesystem()              -- Moves a file from a FS datasource and verifies the operation

    review_move_action_full_crawl_filesystem()   -- Verifies the move operation by running a full crawl

    cleanup()                                    -- Runs cleanup

    run()                                        -- Run function for this testcase
"""
import os
import time

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.constants import AUTOMATION_BIN_PATH
from AutomationUtils.machine import Machine
from Web.Common.exceptions import CVWebAutomationException
from Web.AdminConsole.Helper.GDPRHelper import GDPR
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import handle_testcase_exception, TestStep
from dynamicindex.utils.activateutils import ActivateUtils
from dynamicindex.utils.constants import USA_COUNTRY_NAME, DB_ENTITY_DELIMITER,\
    HOST_NAME


class TestCase(CVTestCase):
    """Class for executing Move action in Review page for a single file with local path"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Move action for a single file with local path in Review page in SDG"
        self.tcinputs = {
            "IndexServerName": None,
            "ContentAnalyzer": None,
            "NameServerAsset": None,
            "HostNameToAnalyze": None,
            "FileServerLocalDirectoryPath": None,
            "FileServerUserName": None,
            "FileServerPassword": None,
            "DestinationPathUserName": None,
            "DestinationPathPassword": None,
            "DestinationPath": None

        }
        # Testcase constants
        self.inventory_name = None
        self.plan_name = None
        self.project_name = None
        self.file_server_display_name = None
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.test_case_error = None
        self.gdpr_base = None
        self.explict_wait = 1 * 60
        self.file_count_before_review = 0
        self.file_count_after_review = 0

    def generate_sensitive_data(self):
        """
            Generate sensitive files with PII entities
        """
        self.activateutils.sensitive_data_generation(
            self.source_machine.get_unc_path(
                self.tcinputs['FileServerLocalDirectoryPath']))

    def get_sensitive_file_details(self):
        """
            Get the sensitive file with entity
        """
        entity_delimiter = DB_ENTITY_DELIMITER
        filename = os.path.join(AUTOMATION_BIN_PATH)
        database_file_path = "{0}\\Entity.db".format(filename)
        entities_dict = self.activateutils.db_get_entities_dict_from_sqllite(
            database_file_path)
        dict_used = entities_dict.pop('result')
        self.log.info(dict_used)
        __entity = ""
        __file = ""
        for item in dict_used:
            __file = item.get('FilePath')
            __entity = item.get('Email')
            if __entity is not None:
                if entity_delimiter in __entity:
                    temp = __entity.split(entity_delimiter)
                    __entity = temp[0]
                    break
                else:
                    __entity = __entity
                    break

        self.sensitive_entity = __entity
        self.sensitive_file = __file

    def delete_files_in_folder(self, destination_path):
        """Traverses the destination path to delete all the files present"""
        for root, dirs, files in os.walk(destination_path):
            for file in files:
                os.remove(os.path.join(root, file))

    def init_tc(self):
        """ Initial configuration for the test case"""
        try:
            username = self.inputJSONnode['commcell']['commcellUsername']
            password = self.inputJSONnode['commcell']['commcellPassword']
            self.activateutils = ActivateUtils()
            self.inventory_name = f'{self.id}_inventory'
            self.plan_name = f'{self.id}_plan'
            self.project_name = f'{self.id}_project'
            self.source_machine = Machine(
                machine_name=self.tcinputs['HostNameToAnalyze'],
                username=self.tcinputs["FileServerUserName"],
                password=self.tcinputs["FileServerPassword"])
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(
                self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(username=username, password=password)
            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_governance_apps()
            self.gdpr_base = GDPR(self.admin_console, self.commcell, self.csdb)
            self.cleanup()

        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def create_inventory(self):
        """
            Create an inventory with a nameserver
        """
        self.navigator.navigate_to_governance_apps()
        self.gdpr_base.inventory_details_obj.select_inventory_manager()
        self.gdpr_base.inventory_details_obj.add_inventory(
            self.inventory_name, self.tcinputs['IndexServerName'])

        self.gdpr_base.inventory_details_obj.add_asset_name_server(
            self.tcinputs["NameServerAsset"])
        self.admin_console.log.info(f"Sleeping for {self.explict_wait}")
        time.sleep(self.explict_wait)
        if not self.gdpr_base.inventory_details_obj.wait_for_asset_status_completion(
                self.tcinputs['NameServerAsset']):
            raise CVTestStepFailure("Could not complete the asset scan.")

    @test_step
    def create_plan(self):
        """
            Creates a plan
        """

        self.navigator.navigate_to_plan()
        self.gdpr_base.plans_obj.create_data_classification_plan(self.plan_name, self.tcinputs[
            'IndexServerName'], self.tcinputs['ContentAnalyzer'], None, select_all=True)

    @test_step
    def create_sda_project(self):
        """
            Creates a project and runs analysis
        """
        self.file_server_display_name = '{}_file_server'.format(self.id)
        self.gdpr_base.testdata_path = self.tcinputs['FileServerLocalDirectoryPath']
        self.gdpr_base.data_source_name = self.file_server_display_name
        self.navigator.navigate_to_governance_apps()
        self.gdpr_base.inventory_details_obj.select_sensitive_data_analysis()
        self.gdpr_base.file_server_lookup_obj.add_project(
            self.project_name, self.plan_name, self.inventory_name)
        self.gdpr_base.file_server_lookup_obj.select_add_data_source()
        self.gdpr_base.file_server_lookup_obj.add_file_server(
            self.tcinputs['HostNameToAnalyze'], HOST_NAME,
            self.file_server_display_name, USA_COUNTRY_NAME,
            self.tcinputs['FileServerLocalDirectoryPath'], agent_installed=True, live_crawl=True
        )
        self.log.info(f"Sleeping for: {self.explict_wait} seconds")
        time.sleep(self.explict_wait)
        if not self.gdpr_base.file_server_lookup_obj.wait_for_data_source_status_completion(
                self.file_server_display_name, timeout=60):
            raise CVTestStepFailure("Could not complete the Datasource scan.")
        self.log.info(f"Sleeping for {self.explict_wait} seconds")
        time.sleep(self.explict_wait)
        self.file_count_before_review = int(self.gdpr_base.data_source_discover_obj.
                                            get_total_number_after_crawl())

    @test_step
    def review_move_action_filesystem(self):
        """
        Verifies that the file is moved to the destination path correctly after move operation
        """
        count_before_move = 0
        count_after_move = 0
        source_file_present = False
        filepath = self.sensitive_file
        filename = filepath[filepath.rindex("\\") + 1:]
        destination_path = self.tcinputs["DestinationPath"]
        username = self.tcinputs["DestinationPathUserName"]
        password = self.tcinputs["DestinationPathPassword"]
        # Get the machine name from the shared folder path
        last_sep_index = destination_path.rindex(os.path.sep)
        machine_name = destination_path[2:last_sep_index]
        try:
            self.log.info(f"Destination machine name is {machine_name}")
            destination_machine = Machine(machine_name,
                                          username=self.tcinputs["DestinationPathUserName"],
                                          password=self.tcinputs["DestinationPathPassword"])
            count_before_move = len(
                destination_machine.get_files_in_path(destination_path))
            if(count_before_move != 0):
                # destination_machine.delete_file(destination_path)
                self.delete_files_in_folder(destination_path)
                count_before_move = len(
                    destination_machine.get_files_in_path(destination_path))
            # Perform move review action
            self.gdpr_base.data_source_review_obj.review_move_action(
                filename, destination_path, username, password)
            # Check if the moved file is present in the source path
            source_files = self.source_machine.get_files_in_path(
                self.tcinputs["FileServerLocalDirectoryPath"])
            self.log.info(f"List of files present in the source path {source_files}")
            for file in source_files:
                if filename in file:
                    source_file_present = True
            # Check if the moved file is present in the destination path
            destination_files = destination_machine.get_files_in_path(
                destination_path)
            count_after_move = len(destination_files)
            self.log.info(f"List of files present in the destination path {destination_files}")
            self.log.info(
                f"Number of files present in the destination path before the move operation: {count_before_move}")
            self.log.info(
                f"Number of files present in the destination path after the move operation: {count_after_move}")
            # Check that the destination path's file count is increased
            if count_after_move == count_before_move + \
                    1 and filename in destination_files[0] and (not source_file_present):
                self.log.info(f"Successfully moved {filename} to {destination_path}")
            else:
                raise CVTestStepFailure
        except (CVWebAutomationException, CVTestStepFailure) as error_status:
            self.test_case_error = f'Move action failed:- {str(error_status)}'
            self.gdpr_base.data_source_review_obj.close_action_modal()

    @test_step
    def review_move_action_full_crawl_filesystem(self):
        """
        Verifies that the moved file is not getting picked during a full crawl
        """
        self.navigator.navigate_to_governance_apps()
        self.gdpr_base.inventory_details_obj.select_sensitive_data_analysis()
        self.gdpr_base.file_server_lookup_obj.search_for_project(
            self.project_name)
        self.gdpr_base.data_source_discover_obj.navigate_to_project_details(
            self.project_name)
        self.gdpr_base.file_server_lookup_obj.select_data_source(
            self.file_server_display_name)
        self.log.info(
            f"Starting a full re-crawl of the datasource {self.file_server_display_name}")
        self.gdpr_base.data_source_discover_obj.select_details()
        self.gdpr_base.data_source_discover_obj.start_data_collection_job(
            'full')
        self.navigator.navigate_to_governance_apps()
        self.gdpr_base.inventory_details_obj.select_sensitive_data_analysis()
        self.gdpr_base.file_server_lookup_obj.navigate_to_project_details(
            self.project_name)
        if not self.gdpr_base.file_server_lookup_obj.wait_for_data_source_status_completion(
                self.file_server_display_name):
            raise Exception("Could not complete the Datasource scan.")
        self.log.info(f"Sleeping for {self.explict_wait} seconds")
        time.sleep(self.explict_wait)
        self.file_count_after_review = int(self.gdpr_base.data_source_discover_obj.
                                           get_total_number_after_crawl())
        self.log.info(
            f"Number of files before the Move operation {self.file_count_before_review}"
            f"Number of files after the Move operation {self.file_count_after_review}")
        if self.file_count_before_review - 1 != self.file_count_after_review:
            self.test_case_error = ("Move operation failed.")

    @test_step
    def cleanup(self):
        """
            Cleans up the environment
        """
        self.navigator.navigate_to_governance_apps()
        self.gdpr_base.cleanup(
            self.project_name,
            self.inventory_name,
            self.plan_name, pseudo_client_name=self.file_server_display_name)

    def run(self):
        try:
            self.init_tc()
            self.generate_sensitive_data()
            self.get_sensitive_file_details()
            self.create_inventory()
            self.create_plan()
            self.create_sda_project()
            self.gdpr_base.file_server_lookup_obj.select_data_source(
                self.file_server_display_name)
            self.gdpr_base.data_source_discover_obj.select_review()
            self.review_move_action_filesystem()
            self.review_move_action_full_crawl_filesystem()
            self.cleanup()
            if self.test_case_error is not None:
                raise Exception(self.test_case_error)
        except Exception as err:
            handle_testcase_exception(self, err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
