# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                          --  initialize TestCase class

    verify_client_existance()				- Check if IBMi client entity present

    create_user_defined_backupset()			- Create a new backupset

    create_user_defined_subclient()			- Create a new Subclient

    delete_user_defined_backupset()			- Delete the specified backpset if exists
    
    generate_client_data_lfs()				- Generate LFS data on IBMi client machine

    generate_client_data()					- Generate data on IBMi client machine

    backup_job()							- Run a filesystem backup job

    validate_sc_defaults()					- Verify the client logs

    navigate_to_user_defined_subclient()	- Navigate to subclient page

    wait_for_job_completion()				- wait for job completion

    delete_user_defined_subclient()			- Delete the specified subclient

    restore_compare()						- Restore IBMi data from subclient and compare

    delete_subclient_if_exists()			- Delete the specified subclient if exists

    cleanup()								- Perform cleanup on client machine and items created in CS

    run()                               --  run function of this test case

Input Example:

    "testCases":
            {
                "59224":
                        {
                            "OSType":"Unix",
                            "AgentName": "File System",
                            "PlanName":"Test-Auto",
                            "AgentName": "File System",
                            "ClientName": "Existing-client",
                            "HostName": "IBMi-host-name",
                            "TestPath": "/QSYS.LIB"
                        }
            }

"""

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.FileServerPages.file_servers import FileServers
from Web.AdminConsole.FileServerPages.fsagent import FsSubclient, FsAgent
from Web.AdminConsole.FileServerPages.fssubclientdetails import FsSubclientDetails
from Web.Common.page_object import TestStep, handle_testcase_exception
from Web.AdminConsole.Components.panel import Backup
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from AutomationUtils.cvtestcase import CVTestCase
from FileSystem.FSUtils.fshelper import FSHelper


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "IBMi-CC-Validate IBMi subclient creation with SWA(*LIB) & backup and restore"
        self.test_path = None
        self.slash_format = None
        self.helper = None
        self.browser = None
        self.exclusions = None
        self.exceptions = None
        self.slash_format = None
        self.client_obj = None
        self.display_name = None
        self.navigator = None
        self.file_servers = None
        self.fs_agent_obj = None
        self.fssubclient_obj = None
        self.table = None
        self.fssubclient_details_obj = None
        self.backup_type = None
        self.backupset = None
        self.subclient = None
        self.srcdir = None
        self.destdir = None
        self.admin_console = None
        self.backupset = None
        self.subclient = None
        self.srcdir = None
        self.destdir = None
        self.client_machine = None
        self.tcinputs = {
            "AgentName": None,
            "ClientName": None,
            "PlanName": None,
            "TestPath": None,
            "HostName": None
        }

    def init_tc(self):
        """ Initial configuration for the test case. """

        try:
            # Initialize test case inputs
            self.log.info("***TESTCASE: %s***", self.name)
            FSHelper.populate_tc_inputs(self)
            if self.test_path.endswith(self.slash_format):
                self.test_path = str(self.test_path).rstrip(self.slash_format)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(username=self.inputJSONnode['commcell']['commcellUsername'],
                                     password=self.inputJSONnode['commcell']['commcellPassword'])
            self.navigator = self.admin_console.navigator
            self.file_servers = FileServers(self.admin_console)
            self.fs_agent_obj = FsAgent(self.admin_console)
            self.fssubclient_obj = FsSubclient(self.admin_console)
            self.table = Table(self.admin_console)
            self.fssubclient_details_obj = FsSubclientDetails(self.admin_console)
            self.backup_type = Backup(self.admin_console)
            self.client_obj = self.commcell.clients.get(self.tcinputs['ClientName'])
            self.display_name = self.client_obj.display_name
            self.backupset = "backupset_{0}".format(self.id)
            self.subclient = "subclient_{0}".format(self.id)
            self.srcdir = "/AUT{0}".format(self.id)
            self.destdir = "/RST{0}".format(self.id)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def verify_client_existance(self):
        """Check if IBMi client entity present"""
        self.navigator.navigate_to_file_servers()
        if not self.file_servers.is_client_exists(server_name=self.display_name):
            raise CVTestStepFailure("IBMi client [{0}] does not exists.".format(self.display_name))

    @test_step
    def create_user_defined_backupset(self):
        """ Create a new backupset """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        self.fs_agent_obj.add_ibmi_backupset(backup_set=self.backupset,
                                             plan=self.tcinputs['PlanName'])

    @test_step
    def create_user_defined_subclient(self):
        """ Create a new Subclient """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        content = [self.client_machine.lib_to_path('TC{0}'.format(self.id)),
                   self.srcdir]
        self.fssubclient_obj.add_ibmi_subclient(subclient_name=self.subclient,
                                                plan=self.tcinputs['PlanName'],
                                                backup_set=self.backupset,
                                                save_while_active="*LIB",
                                                active_wait_time=0,
                                                define_own_content=True,
                                                backup_data=content,
                                                exclusions=["/tmp1", "/tmp2"],
                                                exceptions=["/tmp1/abc", "/tmp2/abcd"]
                                                )

    @test_step
    def delete_user_defined_backupset(self):
        """ Delete the specified backpset if exists """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        status = self.fssubclient_obj.is_backupset_exists(backupset_name=self.backupset)
        self.log.info("backupset existance is {0}".format(status))
        if status:
            self.navigator.navigate_to_file_servers()
            self.file_servers.access_server(self.display_name)
            self.fssubclient_obj.delete_backup_set(self.backupset)

    @test_step
    def generate_client_data_lfs(self):
        """ Generate LFS data on IBMi client machine """
        self.client_machine.populate_lib_with_data(library_name='TC{0}'.format(self.id), tc_id=self.id)

    @test_step
    def generate_client_data(self, backup_level="full"):
        """ Generate data on IBMi client machine
            Args:
                backup_level(string)        -   Backup level to generate data
        """
        if "full" in backup_level:
            self.client_machine.populate_ifs_data(directory_name=self.srcdir,
                                                  tc_id=self.id,
                                                  count=5,
                                                  prefix="F",
                                                  delete=True)
        elif "incr" in backup_level:
            self.client_machine.populate_ifs_data(directory_name=self.srcdir,
                                                  tc_id=self.id,
                                                  count=2,
                                                  prefix="I",
                                                  delete=False)
        elif "diff" in backup_level:
            self.client_machine.populate_ifs_data(directory_name=self.srcdir,
                                                  tc_id=self.id,
                                                  count=2,
                                                  prefix="D",
                                                  delete=False)
        else:
            raise CVTestStepFailure("Backup level %sis not valid", backup_level)

    @test_step
    def backup_job(self, backup_level):
        """ Run a filesystem backup job """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        jobid = self.fs_agent_obj.backup(backup_level,
                                         backupset_name=self.backupset,
                                         subclient_name=self.subclient,
                                         notify=False)
        self.wait_for_job_completion(jobid)
        return jobid

    @test_step
    def validate_sc_defaults(self, jobid):
        """ Verify the client logs """
        self.helper.verify_sc_defaults(jobid)

    @test_step
    def navigate_to_user_defined_subclient(self):
        """ Navigate to subclient page """
        self.admin_console.refresh_page()
        self.fssubclient_obj.access_subclient(self.backupset,
                                              self.subclient)

    @test_step
    def wait_for_job_completion(self, job_id):
        """ wait for job completion"""
        job_obj = self.commcell.job_controller.get(job_id)
        job_status = job_obj.wait_for_completion()
        self.log.info(job_status)
        if job_status == 'False':
            raise CVTestStepFailure("Job %s didn't complete successfully", job_id)
        else:
            self.log.info("Job#{0} completed successfully".format(job_id))

    @test_step
    def delete_user_defined_subclient(self):
        """ Delete the specified subclient """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        self.fssubclient_obj.delete_subclient(self.backupset,
                                              self.subclient)

    @test_step
    def restore_compare(self, level="full"):
        """ Restore IBMi data from subclient and compare """
        self.client_machine.remove_directory(directory_name=self.destdir)
        self.client_machine.create_directory(directory_name=self.destdir)
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        files_to_restore = ["{0}/F{1}0.txt".format(self.srcdir, self.id),
                            "{0}/F{1}1.txt".format(self.srcdir, self.id),
                            "{0}/F{1}2.txt".format(self.srcdir, self.id),
                            "{0}/F{1}3.txt".format(self.srcdir, self.id),
                            "{0}/F{1}4.txt".format(self.srcdir, self.id)]
        if "incr" in level:
            files_to_restore.append("{0}/I{1}0.txt".format(self.srcdir, self.id))
            files_to_restore.append("{0}/I{1}1.txt".format(self.srcdir, self.id))

        job_id = self.fssubclient_obj.ibmi_restore_subclient(backupset_name=self.backupset,
                                                             subclient_name=self.subclient,
                                                             restore_path=self.destdir,
                                                             unconditional_overwrite=False,
                                                             selected_files=files_to_restore
                                                             )
        self.wait_for_job_completion(job_id=job_id)
        self.helper.compare_ibmi_data(source_path="{0}/*".format(self.srcdir),
                                      destination_path="{0}/*".format(self.destdir))

    @test_step
    def delete_subclient_if_exists(self):
        """ Delete the specified subclient if exists """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        if self.fssubclient_obj.is_ibmi_subclient_exists(backup_set=self.backupset,
                                                         subclient_name=self.subclient):
            self.fssubclient_obj.delete_subclient(self.backupset,
                                                  self.subclient)

    @test_step
    def cleanup(self):
        """ Perform cleanup on client machine and items created in CS"""
        self.client_machine.remove_directory(self.destdir)
        self.client_machine.remove_directory(self.srcdir)

    def run(self):
        try:
            self.init_tc()
            self.navigator.navigate_to_file_servers()
            self.verify_client_existance()
            self.delete_user_defined_backupset()
            self.create_user_defined_backupset()
            self.delete_subclient_if_exists()
            self.create_user_defined_subclient()
            self.admin_console.refresh_page()
            self.generate_client_data()
            self.generate_client_data_lfs()
            self.validate_sc_defaults(self.backup_job(Backup.BackupType.FULL))
            self.restore_compare(level="full")
            self.generate_client_data(backup_level="incr")
            self.backup_job(Backup.BackupType.INCR)
            self.restore_compare(level="incr")
            self.cleanup()
            self.delete_user_defined_subclient()
            self.delete_user_defined_backupset()
            self.log.info("**IBMi - BACKUPSET MANAGEMENT, SUBCLIENT MANAGEMENT, BACKUP OPERATION,  "
                          "RESTORE OPERATION FROM COMMAND CENTER HAS COMPLETED SUCCESSFULLY**")
            self.log.info("******TEST CASE COMPLETED SUCCESSFULLY AND PASSED******")

        except Exception as exception:
            handle_testcase_exception(self, exception)

        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
