# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case(verify the creation and updation of Azure recovery target
of replication type)

When we create a Azure VM Replication group, override option ->edit VM and then change
Virtual Network and Security group.
After creation, go the group's configuration page and edit the VM, it should show the
selected options for Virtual Network and security group, as selected during creation,
instead of Auto select(default)

TestCase is the only class defined in this file.

TestCase: Class for executing this test case
Sample JSON: {
        "hypervisor": "Azure",
        "source_vm_1": "AzureProxy", -> override
        "source_vm_2": "azurelinuxproxy", -> default before creation
        "source_vm_3": "azureproxy3", -> default after creation
        "recovery_target": "AzureTarget",
        "storage": "test",
        "resource_group": "DR-Destination",
        "storage_account": "drdestinationdiag",
        "vm_size": "Standard_B1ls",
        "virtual_network": "DR-Destination-vnet",
        "security_group": "azureProxy-nsg",
        "vm_size_2": "Standard_B1s",
        "virtual_network_2": "DR-Testing-vnet",
        "security_group_2": "azurelinuxproxy-nsg"
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails, SOURCE_HYPERVISOR_AZURE
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.virtualization_replication import _Target
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets
from Web.AdminConsole.VSAPages.vm_groups import VMGroups
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the replication of Defect #288874"""
    test_step = TestStep()
    _AZURE_VENDOR_NAME = "Microsoft Azure"
    _REPLICATION_TYPE = "Periodic"
    _TARGET_FREQ_NUMBER = 4
    _TARGET_FREQ_UNIT = _Target.FREQUENCY_HOURS
    _AZURE_VENDOR_VMG = "Microsoft Azure"

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Azure Replication group: Verify Creation and edit operation"
        self.tcinputs = {
            "hypervisor": None,
            "source_vm_1": None,
            "source_vm_2": None,
            "source_vm_3": None,
            "recovery_target": None,
            "storage": None,
            "resource_group": None,
            "storage_account": None,
            "vm_size": None,
            "virtual_network": None,
            "security_group": None,
            "vm_size_2": None,
            "virtual_network_2": None,
            "security_group_2": None,
        }
        self.utils = None

        self.browser = None
        self.admin_console = None

        self.replication_group = None
        self.group_details = None
        self.vm_group = None
        self.target_details = None
        self.vm_noedit_details = None
        self.edit_vm = None

    @property
    def group_name(self):
        """Returns the name for the replication group"""
        return f"Group_TC_{self.id}"

    def login(self):
        """Logs in to admin console"""
        self.admin_console = AdminConsole(
            self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname']
        )
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)
        self.vm_group = VMGroups(self.admin_console)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def get_recovery_target_details(self):
        """Gets the recovery target details for verification"""
        self.admin_console.navigator.navigate_to_replication_targets()
        recovery_target = RecoveryTargets(self.admin_console).access_target(
            self.tcinputs['recovery_target']
        )
        summary = recovery_target.get_target_summary()
        self.target_details = [summary['Resource group'],
                               summary['Region'], summary['Storage account'], summary['VM size'],
                               summary['Virtual network'], summary['Security group']]

    @test_step
    def create_replication_group(self):
        """Creates and configures a new Azure replication group with override options"""
        configure = self.replication_group.configure_azure()
        configure.content.set_name(self.group_name)

        configure.content.select_vm_from_browse_tree(self.tcinputs['hypervisor'],
                                                     {"VMs": [self.tcinputs['source_vm_1']]})
        configure.content.select_vm_from_browse_tree(self.tcinputs['hypervisor'],
                                                     {"VMs": [self.tcinputs['source_vm_2']]})
        configure.next()

        configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        configure.next()

        configure.storage_cache.select_storage(self.tcinputs['storage'])
        configure.next()

        override_options = configure.override_options.override_vms(self.tcinputs['source_vm_1'])
        override_options.select_resource_group(self.tcinputs['resource_group'])
        override_options.select_storage_account(self.tcinputs['storage_account'])
        override_options.select_vm_size(self.tcinputs['vm_size'], use_value=True)
        override_options.virtual_network(self.tcinputs['virtual_network'])
        override_options.select_security_group(self.tcinputs['security_group'])
        self.admin_console.click_button('Save')

        self.vm_noedit_details = configure.override_options.get_vm_details(
            self.tcinputs['source_vm_2']
        )
        self.utils.assert_comparison(self.vm_noedit_details[1:], self.target_details)

        configure.next()

        sleep(5)
        configure.finish()

    @test_step
    def verify_creation(self):
        """Verifies whether the replication group is created with correct values in the table"""
        details = self.replication_group.get_replication_group_details_by_name(self.group_name)

        self.utils.assert_comparison(details['Group name'][0], self.group_name)
        self.utils.assert_comparison(details['Source'][0], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(details['Destination'][0], self.tcinputs['recovery_target'])
        self.utils.assert_comparison(details['Type'][0], 'Virtual Server')
        self.utils.assert_comparison(details['Replication type'][0], 'Periodic')
        self.utils.assert_comparison(details['State'][0], 'Enabled')

    @test_step
    def verify_vm_group(self):
        """Verifies whether the VM group exists or not"""
        self.admin_console.navigator.navigate_to_vm_groups()
        self.admin_console.refresh_page()
        table_content = self.vm_group.get_details_by_vm_group(self.group_name)
        self.utils.assert_includes(self.group_name, table_content['Name'][0])
        self.utils.assert_comparison(table_content['Vendor'][0], self._AZURE_VENDOR_VMG)
        self.utils.assert_comparison(table_content['Hypervisor'][0], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(
            table_content['Plan'][0], f"{self.group_name}_ReplicationPlan")

    @test_step
    def disable_replication_group(self):
        """Disables the replication group and re-enables it to verify the group status"""
        self.group_details.disable_replication_group()

        summary = self.group_details.overview.get_summary_details()
        self.utils.assert_comparison(summary['State'], 'Disabled')
        self.group_details.enable_replication_group()
        self.utils.assert_comparison(
            self.group_details.overview.get_summary_details().get('State'), 'Enabled')

    @test_step
    def verify_overview(self):
        """Verifies the details of the replication group in the overview tab"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)

        summary = self.group_details.overview.get_summary_details()
        self.utils.assert_comparison(summary['Source'], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(summary['Recovery target'], self.tcinputs['recovery_target'])
        self.utils.assert_comparison(summary['Destination vendor'], self._AZURE_VENDOR_NAME)
        self.utils.assert_comparison(summary['Replication type'], self._REPLICATION_TYPE)
        self.utils.assert_comparison(summary['State'], "Enabled")

    @test_step
    def verify_configuration(self):
        """Verifies the details of the replication group in the configuration tab"""
        self.group_details.access_configuration_tab()

        rpo = self.group_details.configuration.get_rpo_details()
        self.utils.assert_includes(
            f"{self._TARGET_FREQ_NUMBER} {self._TARGET_FREQ_UNIT.lower()}",
            rpo['Replication frequency']
        )
        storages = self.group_details.configuration.get_storage_details()
        self.utils.assert_includes(self.tcinputs['storage'], storages)

        adv_options = self.group_details.configuration.get_advanced_options_details()

        self.utils.assert_comparison(
            adv_options['Deploy virtual machine only during failover'], "ON")

        vm_details_1 = self.group_details.configuration.get_vm_details(self.tcinputs['source_vm_1'])
        self.utils.assert_comparison(self.tcinputs['source_vm_1'], vm_details_1['Source VM'])
        self.utils.assert_comparison(
            self.tcinputs['resource_group'], vm_details_1['Resource group'])
        self.utils.assert_comparison(
            self.tcinputs['storage_account'], vm_details_1['Storage account'])
        self.utils.assert_includes(self.tcinputs['vm_size'], vm_details_1['VM size'])

        vm_details_2 = self.group_details.configuration.get_vm_details(self.tcinputs['source_vm_2'])
        self.utils.assert_comparison(self.tcinputs['source_vm_2'], vm_details_2['Source VM'])
        self.utils.assert_comparison(self.vm_noedit_details[1], vm_details_2['Resource group'])
        self.utils.assert_comparison(self.vm_noedit_details[3], vm_details_2['Storage account'])
        self.utils.assert_includes(self.vm_noedit_details[4], vm_details_2['VM size'])

    @test_step
    def add_delete_vm_to_group(self):
        """Add VM to group after creation"""
        add_vm = self.group_details.configuration.add_virtual_machines(vm_type=SOURCE_HYPERVISOR_AZURE)
        add_vm.add_vm([self.tcinputs['source_vm_3']])

        new_vm_details = self.group_details.configuration.get_vm_details(self.tcinputs['source_vm_3'])
        if new_vm_details['Source VM'] != self.tcinputs['source_vm_3']:
            raise CVTestStepFailure(
                f"Expected value of Source VM {self.tcinputs['source_vm_3']} does not match "
                f"the collected value {new_vm_details['Source VM']}")
        self.group_details.configuration.remove_virtual_machines(self.tcinputs['source_vm_3'])
        try:
            self.admin_console.refresh_page()
            self.group_details.configuration.get_vm_details(self.tcinputs['source_vm_3'])
        except:
            self.log.info("VM successfully deleted from group")
        else:
            raise CVTestStepFailure("VM could not be deleted successfully")

    @test_step
    def verify_disabled_fields(self, vm_id):
        """Verifies that the disabled fields are disabled or not"""
        self.admin_console.refresh_page()
        self.edit_vm = self.group_details.configuration.edit_virtual_machines(
            self.tcinputs[f'source_vm_{vm_id}'], vm_type=SOURCE_HYPERVISOR_AZURE)

        disabled = ['displayNamePrefixSuffix', 'azureContainer', 'azureRegion',
                    'azureStorageAccount']
        for field_id in disabled:
            field_disabled = self.edit_vm.is_field_disabled(field_id)
            if field_disabled is None:
                raise CVTestStepFailure(f"The field {field_id} is not interactable/existent")
            if not field_disabled:
                raise CVTestStepFailure(
                    f'In Edit VM The field {field_id} is enabled, but must be disabled')

    @test_step
    def verify_edit_vm(self, after_edit=False, vm_id=1):
        """Verifies the data on the edit VM page"""
        sleep(10)
        if vm_id == 2:
            self.utils.assert_comparison(self.vm_noedit_details[4], self.edit_vm.vm_size)
            self.utils.assert_comparison(self.vm_noedit_details[5], self.edit_vm.virtual_network)
            self.utils.assert_comparison(self.vm_noedit_details[6], self.edit_vm.security_group)
        elif after_edit:
            self.utils.assert_includes(self.tcinputs['vm_size_2'], self.edit_vm.vm_size)
            self.utils.assert_includes(
                self.tcinputs['virtual_network_2'], self.edit_vm.virtual_network)
            self.utils.assert_comparison(
                self.edit_vm.security_group, self.tcinputs['security_group_2'])
        else:
            self.utils.assert_includes(self.tcinputs['vm_size'], self.edit_vm.vm_size)
            self.utils.assert_includes(
                self.tcinputs['virtual_network'], self.edit_vm.virtual_network)
            self.utils.assert_comparison(
                self.edit_vm.security_group, self.tcinputs['security_group'])
        self.edit_vm.cancel()

    @test_step
    def edit_vm_details(self):
        """Modify the group details to check if the detail change is registered on Command Center"""
        self.admin_console.refresh_page()
        edit_vm = self.group_details.configuration.edit_virtual_machines(self.tcinputs['source_vm_1'],
                                                                         vm_type=SOURCE_HYPERVISOR_AZURE)
        edit_vm.select_vm_size(self.tcinputs['vm_size_2'])
        edit_vm.select_virtual_network(self.tcinputs['virtual_network_2'])
        edit_vm.select_security_group(self.tcinputs['security_group_2'])
        edit_vm.save()

    @test_step
    def delete_replication_group(self):
        """Deletes the replication group if it exists
        Returns: True if replication group was present and deleted, false if it didn't exist before
        """
        self.admin_console.navigator.navigate_to_replication_groups()
        if not self.replication_group.has_group(self.group_name):
            return False
        self.replication_group.delete_group(self.group_name)
        return True

    def run(self):
        """Runs the testcase in order"""
        try:
            self.get_recovery_target_details()
            self.delete_replication_group()
            self.create_replication_group()
            self.verify_creation()
            self.verify_vm_group()

            self.verify_overview()
            self.disable_replication_group()
            self.verify_configuration()
            self.add_delete_vm_to_group()

            self.verify_disabled_fields(vm_id=1)
            self.verify_edit_vm(after_edit=False, vm_id=1)
            self.verify_disabled_fields(vm_id=2)
            self.verify_edit_vm(after_edit=False, vm_id=2)

            self.edit_vm_details()

            self.verify_disabled_fields(vm_id=1)
            self.verify_edit_vm(after_edit=True, vm_id=1)
            self.verify_disabled_fields(vm_id=2)
            self.verify_edit_vm(after_edit=True, vm_id=2)

            self.delete_replication_group()
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Tears down the TC"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)
