# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case(verify the creation and updation of Azure recovery target
of replication type)

TestCase is the only class defined in this file.

TestCase: Class for executing this test case
Sample JSON: {
          "hypervisor": "azure_hyp",
          "access_node": "node1",
          "resource_group": "dr-test",
          "storage_account": "drtestaccount",
          "vm_size": "size1",
          "virtual_network": "vnet1",
          "security_group": "secgroup1",
          "access_node_2": "node2",
          "vm_size_2": "Standard_B1s",
          "virtual_network_2": vnet2",
          "security_group_2": "secgroup2"
}
"""

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.page_object import TestStep
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets, TargetConstants


class TestCase(CVTestCase):
    """This testcase is defined for the verification of creation and update of Azure recovery target"""
    test_step = TestStep()
    _REGION = 'East US'
    _SUFFIX = 'DRVM'
    _CREATE_PUBLIC_IP = "No"
    _RESTORE_AS_MANAGED = "Yes"

    def __init__(self):
        """Initialises the class and variables are set to None"""
        super(TestCase, self).__init__()
        self.name = "Azure Recovery Targets: Verification of creation and update of target"
        self.tcinputs = {
            "hypervisor": None,
            "access_node": None,
            "resource_group": None,
            "storage_account": None,
            "vm_size": None,
            "virtual_network": None,
            "security_group": None,
            "access_node_2": None,
            "vm_size_2": None,
            "virtual_network_2": None,
            "security_group_2": None,
        }

        self.browser = None
        self.admin_console = None
        self.utils = None
        self.recovery_targets = None
        self.target = None
        self.edit_target = None

    @property
    def target_name(self):
        """Returns the initial target name for the recovery target"""
        return f'test_auto_tc_{self.id}'

    @property
    def edited_target_name(self):
        """Returns the modified target name"""
        return f'test_auto_edited_tc_{self.id}'

    def login(self):
        """Logs in to the admin console and initialises it"""
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.goto_adminconsole()
        self.admin_console.login(
            self.inputJSONnode['commcell']['commcellUsername'],
            self.inputJSONnode['commcell']['commcellPassword']
        )

    def setup(self):
        """Sets up the various variables and initiates the admin console"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.login()
            self.recovery_targets = RecoveryTargets(self.admin_console)
        except Exception as exp:
            raise CVTestCaseInitFailure(f"Could not initialise the test case {self.id} "
                                        f"due to following error:\n{str(exp)}")

    @test_step
    def create_recovery_target(self):
        """Creates a recovery target in the recovery targets page"""
        self.admin_console.navigator.navigate_to_replication_targets()
        azure_target = self.recovery_targets.configure_azure_recovery_target(self.target_name, 'Replication')
        azure_target.select_destination_hypervisor(self.tcinputs['hypervisor'])
        azure_target.select_access_node(self.tcinputs['access_node'])

        azure_target.set_vm_display_name("DR-Auto", suffix=True)

        azure_target.select_resource_group(self.tcinputs['resource_group'])
        azure_target.select_region(self._REGION)
        azure_target.select_storage_account(self.tcinputs['storage_account'])

        azure_target.expand_options()
        azure_target.set_vm_size(self.tcinputs['vm_size'])
        azure_target.select_virtual_network(self.tcinputs['virtual_network'])
        azure_target.set_security_group(self.tcinputs['security_group'])

        azure_target.save()

    @test_step
    def verify_target_creation(self):
        """Verifies the information of the recovery target on the recovery target page table"""
        self.admin_console.navigator.navigate_to_replication_targets()
        details = self.recovery_targets.get_target_details(self.target_name)

        self.utils.assert_comparison(details['Name'][0], self.target_name)
        self.utils.assert_comparison(details['Vendor'][0], 'Microsoft Azure')
        self.utils.assert_comparison(details['Application type'][0], 'Replication')
        self.utils.assert_comparison(details['Destination hypervisor'][0], self.tcinputs['hypervisor'])

    @test_step
    def verify_target_fields_disabled(self):
        """Verifies that the edit target page's fields are disabled"""
        self.edit_target = self.target.edit_target(self.target_name, TargetConstants.MICROSOFT_AZURE)

        fields_disabled = ["client", "azureContainer", "azureRegion", "azureStorageAccount"]
        for field_id in fields_disabled:
            field_disabled = self.edit_target.is_field_disabled(field_id)
            if field_disabled is None:
                raise CVTestStepFailure(f"The field {field_id} is not interactable/existent")
            if not field_disabled:
                raise CVTestStepFailure(f'In Edit VM The field {field_id} is enabled, but must be disabled')

    @test_step
    def verify_target_details(self, after_edit=False):
        """Verifies the target details in the target's detail page"""
        if not after_edit:
            self.target = self.recovery_targets.access_target(self.target_name)
        target_details = self.target.get_target_summary()

        if not after_edit:
            self.utils.assert_comparison(target_details['Access node'], self.tcinputs['access_node'])
            self.utils.assert_includes(self.tcinputs['vm_size'], target_details['VM size'])
            self.utils.assert_includes(self.tcinputs['virtual_network'], target_details['Virtual network'])
            self.utils.assert_comparison(self.tcinputs['security_group'], target_details['Security group'])
            self.utils.assert_comparison(self._CREATE_PUBLIC_IP, target_details['Create public IP'])
            self.utils.assert_comparison(self._RESTORE_AS_MANAGED, target_details['Restore as a managed VM'])
        else:
            self.utils.assert_comparison(target_details['Access node'], self.tcinputs['access_node_2'])
            self.utils.assert_includes(self.tcinputs['vm_size_2'], target_details['VM size'])
            self.utils.assert_includes(self.tcinputs['virtual_network_2'], target_details['Virtual network'])
            self.utils.assert_comparison(self.tcinputs['security_group_2'], target_details['Security group'])
            self.utils.assert_comparison("No" if self._CREATE_PUBLIC_IP == "Yes" else "Yes",
                                   target_details['Create public IP'])
            self.utils.assert_comparison("No" if self._RESTORE_AS_MANAGED == "Yes" else "Yes",
                                   target_details['Restore as a managed VM'])

        # TODO: Add check to verify Users Group
        self.utils.assert_comparison(target_details['Destination hypervisor'], self.tcinputs['hypervisor'])
        self.utils.assert_comparison(target_details['Resource group'], self.tcinputs['resource_group'])
        self.utils.assert_comparison(target_details['Region'], self._REGION)
        self.utils.assert_comparison(target_details['Storage account'], self.tcinputs['storage_account'])

    @test_step
    def edit_recovery_target(self):
        """Edits the target configuration"""
        self.edit_target.recovery_target_name = self.edited_target_name
        self.edit_target.set_recovery_target_name()
        self.edit_target.select_access_node(self.tcinputs['access_node_2'])

        self.edit_target.expand_options()
        self.edit_target.set_vm_size(self.tcinputs['vm_size_2'])
        self.edit_target.select_virtual_network(self.tcinputs['virtual_network_2'])
        self.edit_target.set_security_group(self.tcinputs['security_group_2'])

        self.edit_target.set_public_ip(create=self._CREATE_PUBLIC_IP != 'Yes')
        self.edit_target.restore_as_managed_vm(enable=self._RESTORE_AS_MANAGED != "Yes")

        self.admin_console.click_button('Save')

    @test_step
    def delete_recovery_target(self):
        """Tries to delete the recovery target"""
        self.admin_console.navigator.navigate_to_replication_targets()
        if self.recovery_targets.has_target(self.edited_target_name):
            self.recovery_targets.delete_recovery_target(self.edited_target_name)

    def run(self):
        """Executes the testcase"""
        try:
            self.delete_recovery_target()
            self.create_recovery_target()

            self.verify_target_creation()
            self.verify_target_details(after_edit=False)
            self.verify_target_fields_disabled()

            self.edit_recovery_target()
            self.verify_target_details(after_edit=True)
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Performs garbage collection for the TC"""
        try:
            self.delete_recovery_target()
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)
