# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Test case for creating and validation of a live BLR pair
Sample json
        "view_name": "AutomationView",
        "source_name": "apurandare",
        "source_volume": "E:",
        "destination_name": "apurandare",
        "destination_volume": "F:",
        "copy_volume": "G:"
"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from Web.AdminConsole.DR.monitor import ContinuousReplicationMonitor
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.page_object import TestStep, wait_for_condition
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """Class for executing Command Center DR replica"""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Admonconsole--> live replication validation"
        self.tcinputs = {
            "view_name": None,
            "source_name": None,
            "source_volume": None,
            "destination_name": None,
            "destination_volume": None,
            "copy_volume": None,
        }
        self.utils = None

        self.browser = None
        self.admin_console = None
        self.replication_monitor = None
        self.source = None
        self.destination = None

    @property
    def test_path(self):
        """Generate the path at which files are present"""
        if not self.source:
            return None
        return self.source.join_path(self.tcinputs['source_volume'], 'testData')

    @wait_for_condition(timeout=300)
    def wait_for_sync_status(self, expected):
        """Waits for the sync status to meet expected value"""
        return (self.replication_monitor.sync_status(self.tcinputs['source_name'],
                                                     self.tcinputs['destination_name']) == expected)

    def check_pair_status(self, expected):
        """Checks the sync status of the BLR pair with the expected value"""
        self.wait_for_sync_status(expected)


    def login(self):
        """Logs in to command center"""
        self.admin_console = AdminConsole(self.browser, self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.goto_adminconsole()
        self.admin_console.login(
            self.inputJSONnode['commcell']['commcellUsername'],
            self.inputJSONnode['commcell']['commcellPassword'],
        )

    def setup(self):
        """Sets up the variables for the test case"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()

            self.login()
            self.admin_console.wait_for_completion()

            self.replication_monitor = ContinuousReplicationMonitor(self.admin_console)

            self.source = Machine(
                self.tcinputs['source_name'],
                commcell_object=self.commcell,
            )
            self.destination = Machine(
                self.tcinputs['destination_name'],
                commcell_object=self.commcell,
            )
        except Exception as exp:
            raise CVTestCaseInitFailure(f"Failed to initialize testcase due to {str(exp)}")

    @test_step
    def delete_pair(self, after_run=False):
        """Deletes the pair and information if it exists"""
        self.admin_console.navigator.navigate_to_continuous_replication()

        if self.replication_monitor.has_replication_group(self.tcinputs['source_name'],
                                                          self.tcinputs['destination_name']):
            if self.replication_monitor.check_view(self.tcinputs['view_name']):
                self.replication_monitor.delete_view(self.tcinputs['view_name'])
            self.replication_monitor.create_view(self.tcinputs['view_name'],
                                                 {'Source': self.tcinputs['source_name'],
                                                  'Destination': self.tcinputs['destination_name']})
            self.replication_monitor.select_view(self.tcinputs['view_name'])
            self.replication_monitor.delete_pair()
            self.replication_monitor.delete_view(self.tcinputs['view_name'])

        if after_run:
            for idx in range(1, 4):
                test_path = "%s%d" % (self.test_path, idx)
                destination_path = test_path.replace(self.tcinputs['source_volume'],
                                                     self.tcinputs['destination_volume'])
                copy_path = test_path.replace(self.tcinputs['source_volume'], self.tcinputs['copy_volume'])
                try:
                    self.source.remove_directory(test_path)
                    self.destination.remove_directory(destination_path)
                    self.destination.remove_directory(copy_path)
                except Exception as exp:
                    self.log.info(exp)

    @test_step
    def create_blr(self):
        """1) Create a Live replication pair (taking source / Destination client and volumes)"""
        self.admin_console.navigator.navigate_to_replication_groups()

        replication_group = ReplicationGroup(self.admin_console)

        blr = replication_group.configure_blr()
        blr.add_block_level_replication(
            self.tcinputs['source_name'],
            self.tcinputs['destination_name'],
            [self.tcinputs['source_volume']],
            [self.tcinputs['destination_volume']],
            0,
        )

    @test_step
    def verify_blr_creation(self):
        """Verifies the creating of a Live BLR pair in the replication monitor's continuous tab"""
        self.admin_console.navigator.navigate_to_continuous_replication()

        if not self.replication_monitor.has_replication_group(
                self.tcinputs['source_name'],
                self.tcinputs['destination_name']):
            raise CVTestStepFailure("The BLR pair was not created or found")

        self.replication_monitor.create_view(self.tcinputs['view_name'],
                                             {'Source': self.tcinputs['source_name'],
                                              'Destination': self.tcinputs['destination_name']})

    @test_step
    def generate_test_data(self, idx):
        """Generate at test_path for particular cycle number"""
        try:
            test_path = "%s%d" % (self.test_path, idx)
            self.source.create_directory(test_path, force_create=True)
            self.source.generate_test_data(test_path, dirs=4, files=4, file_size=512 * 128)
        except Exception as exp:
            raise CVTestStepFailure(f"Cycle {idx}: Could not generate"
                                    f" test data on the source client. Exception - {str(exp)}")

    @test_step
    def verify_pair_status(self):
        """Waits for sometime for pair to add new content and then check its status"""
        # 2) Check Status on pair
        self.admin_console.navigator.navigate_to_continuous_replication()
        self.replication_monitor.select_view(self.tcinputs['view_name'])
        self.check_pair_status('Replicating')

    @test_step
    def verify_resync(self):
        """Performs a re-sync operation and verifies the status"""
        self.replication_monitor.resync()
        self.check_pair_status('Replicating')

    @test_step
    def verify_suspend(self):
        """Performs a suspend and verifies the state"""
        self.replication_monitor.suspend()
        self.check_pair_status('Suspended')

    @test_step
    def verify_resume(self):
        """Performs a resume operation"""
        self.replication_monitor.resume()
        self.check_pair_status('Replicating')

    @test_step
    def verify_stop(self):
        """Performs a stop operation"""
        self.replication_monitor.stop()
        self.check_pair_status('Stopped')

    @test_step
    def verify_start(self):
        """Performs a start operation"""
        self.replication_monitor.start()
        self.check_pair_status('Replicating')

    @test_step
    def create_replica_copy(self):
        """Create a replica copy for the BLR pair
        Returns: Job ID of the copy job"""
        replica_copy = self.replication_monitor.create_replica_copy()
        return replica_copy.submit_replica_job(self.tcinputs['copy_volume'], 0)

    @test_step
    def verify_copy_success(self, job_id):
        """Verifies whether the job was successfully completed or not"""
        job_obj = self.commcell.job_controller.get(job_id)
        if job_obj is None:
            raise CVTestStepFailure(f"The copy job with Job ID [{job_id}] does not exist")
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    def run(self):
        try:
            self.delete_pair()
            self.create_blr()

            self.verify_blr_creation()
            # Run 3 cycles
            for cycle_num in range(1, 4):
                self.generate_test_data(cycle_num)

                self.verify_pair_status()

                # 3) Pair operations (Start/Stop / Suspend/Resume ) and validate against status of pair
                # Re-sync
                self.verify_resync()
                # Suspend
                self.verify_suspend()
                # Resume
                self.verify_resume()
                # Stop
                self.verify_stop()
                # Start
                self.verify_start()

                # 5) Run a Replica copy job on pair
                job_id = self.create_replica_copy()
                self.verify_copy_success(job_id)

            self.delete_pair(after_run=True)
        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)
