# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                --  initialize TestCase class

    init_tc()                 --  Initial configuration for the test case
    run_send_log()            -- To run sendLogs job
    extract_send_logs_files   -- Extract send Log file
    verify_scrub_logs()       -- verify that client name, ip address, paths are scrubbed in cvd.log
    run_send_log_for_client() -- To run sendLogs on linux client
    extract_send_logs_files   -- Extract send Log file
    verify_scrub_logs()       -- verify that client name is scrubbed in cvd.log
    run()                    --  run function of this test case

Input Example:

    "testCases":
            {
                "58941":
                        {
                            "Linux_client": "linuxClientName"
                        }
            }


"""
import time
import os
import re
import tarfile
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from cvpysdk.license import LicenseDetails
from Reports.SendLog.utils import SendLogUtils
from Reports.utils import TestCaseUtils
from Web.Common.page_object import handle_testcase_exception
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.page_object import TestStep
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.Commcell import Commcell
from Web.AdminConsole.AdminConsolePages.Servers import Servers
from Web.AdminConsole.AdminConsolePages.send_logs import SendLogs
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.AdminConsole.FileServerPages.file_servers import FileServers

class TestCase(CVTestCase):
    """Sendlogs: verify sendlogs with scrub logfiles option"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case
        """
        super(TestCase, self).__init__()
        self.browser = None
        self.admin_console = None
        self.send_log = None
        self.jobid = None
        self.utils = None
        self.commcell_id = None
        self.commcell_name = None
        self.download_directory = None
        self.machine = None
        self.path = None
        self.directory = None
        self.utils = TestCaseUtils(self)
        self.utils.reset_temp_dir()
        self.download_directory = self.utils.get_temp_dir()
        self.send_log_utils = None
        self.commserv_client = None
        self.linux_client = None
        self.file_server = None
        self.name = "Sendlogs: verify sendlogs with scrub logfiles option"

    def init_tc(self):
        """
            Initial configuration for the test case
            """
        try:
            self.log.info("Initializing pre-requisites")
            self.log.info('Connecting to local machine wait for 1 min')
            self.commserv_client = self.commcell.commserv_client
            self.machine = Machine(self.commserv_client)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(self.inputJSONnode['commcell']["commcellUsername"],
                                     self.inputJSONnode['commcell']["commcellPassword"])
            self.send_log_utils = SendLogUtils(self, self.machine)
            self.directory = self.send_log_utils.create_directory_for_given_path("TC58941")
            navigator = self.admin_console.navigator
            navigator.navigate_to_commcell()
            licence = LicenseDetails(self.commcell)
            self.commcell_id = licence.commcell_id_hex
            self.commcell_name = self.commcell.commserv_name
            self.linux_client = self.tcinputs['linux_client']
            self.file_server = FileServers(self.admin_console)
            comm_cell = Commcell(self.admin_console)
            comm_cell.access_sendlogs()
            self.send_log = SendLogs(self.admin_console)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def run_send_log(self):
        """Running SendLog job"""
        self.send_log.select_local_output(local_path=self.directory)
        information_list = self.send_log.Informationlist
        self.send_log.select_information(information_list=[information_list.OS_LOGS,
                                                    information_list.LOGS])
        advanced_list = self.send_log.Advancedlist
        self.send_log.select_advanced(advanced_list=[advanced_list.SCRUB_LOGFILES])
        self.jobid = self.send_log.submit()
        job_obj = self.commcell.job_controller.get(self.jobid)
        job_status = job_obj.wait_for_completion()
        if not job_status:
            raise CVTestStepFailure(
                f"Send log job id [{self.jobid}] failed"
            )

    @test_step
    def run_send_log_for_client(self):
        """Running SendLog job"""
        self.admin_console.navigator.navigate_to_file_servers()
        self.admin_console.wait_for_completion()
        self.file_server.action_sendlogs(self.linux_client)
        self.send_log.select_local_output(local_path=self.directory)
        advanced_list = self.send_log.Advancedlist
        self.send_log.select_advanced(advanced_list=[advanced_list.SCRUB_LOGFILES])
        self.jobid = self.send_log.submit()
        job_obj = self.commcell.job_controller.get(self.jobid)
        job_status = job_obj.wait_for_completion()
        if not job_status:
            raise CVTestStepFailure(
                f"Send log job id [{self.jobid}] failed"
            )
    @test_step
    def extract_send_logs_files(self):
        """ To unzip send log content """
        self.path = self.send_log_utils.unzip_send_log_file(self.commserv_client, self.path)

    @test_step
    def verify_scrub_logs(self, client_id, is_windows=True):
        """Verifying that client name, ip address and paths are scrubbed """
        self.log.info(
            "Unzipping file and Verifying scrbbing in logs: cvd.log ")
        if is_windows:
            client_dir = "Masked_Clientname_" + str(client_id) + ".7z"
        else:
            client_dir = "Masked_Clientname_" + str(client_id) + "_logs.tar.gz"
        file_path = os.path.join(self.path, client_dir)
        if is_windows:
            file_path = self.send_log_utils.unzip_send_log_file(self.commserv_client, file_path)
        else:
            self.log.info("Unzipping the tar file " + file_path)
            file_tar = tarfile.open(file_path)
            file_tar.extractall(self.path)
            file_tar.close()
            self.log.info("Output location " + self.path)
            cvd_found = False
            for filename in os.listdir(self.path):
                if filename in 'CVD.log':
                    cvd_found =True
                    break
            if not cvd_found:
               for filename in os.listdir(self.path):
                   if 'logFiles_' in filename:
                       break 
               file_path = os.path.join(self.path, filename)
               self.log.info("Unzipping the tar file " + file_path)
               file_tar = tarfile.open(file_path)
               file_tar.extractall(self.path)
               file_tar.close()
            file_path = self.path
        self.send_log_utils.log_file_verification_with_full_name(
            file_path,
            ['cvd.log']
        )
        CS_name_exists = False
        linux_client_name_exists = False
        ip_exists = False
        path_exists = False
        with open(os.path.join(file_path, 'CVD.log')) as cvd_log:
            lines = cvd_log.readlines()
            for text in lines:
                """if self.commcell_name in text:
                    CS_name_exists = True"""
                if not is_windows:
                    if self.linux_client in text:
                        linux_client_name_exists = True
                ip_exists = re.findall(r'(?:[\d]{1,3})\.(?:[\d]{1,3})\.(?:[\d]{1,3})\.'
                                       r'(?:[\d]{1,3})', text)
                if is_windows:
                    path_exists = re.findall(r'^[a-zA-Z]:(\\.*)$', text)
                if CS_name_exists or ip_exists or path_exists or linux_client_name_exists:
                    break
        if CS_name_exists:
            raise CVTestStepFailure(
                f"Commserv name  [{self.commcell_name}] is not scrubbed"
            )
        if ip_exists:
            raise CVTestStepFailure(
                f"IP Address is not scrubbed"
            )
        if path_exists:
            raise CVTestStepFailure(
                f"File path is not scrubbed"
            )
        if linux_client_name_exists:
            raise CVTestStepFailure(
                f"Client name  [{self.linux_client}] is not scrubbed"
            )

    def get_clientid(self):
        """to get the clientId of the given client name"""
        query = "select id from app_client where name ='" + self.linux_client +"'"
        result = self.utils.cre_api.execute_sql(query)
        return result[0][0]

    def run(self):
        try:
            self.init_tc()
            self.run_send_log()
            self.log.info('Waiting for 1 minute to check file present at ' + self.directory +
                          'location for send log job id' + self.jobid)
            time.sleep(60)
            self.path = self.send_log_utils.is_send_log_bundle_exist(self.directory, self.jobid)
            self.extract_send_logs_files()
            self.verify_scrub_logs(2)
            self.machine.remove_directory(self.directory)
            self.log.info('Running senlogs for the linux client ' + self.linux_client)
            self.run_send_log_for_client()
            self.log.info('Waiting for 1 minute to check file present at ' + self.directory +
                          'location for send log job id' + self.jobid)
            time.sleep(60)
            client_id = self.get_clientid()
            self.path = self.send_log_utils.is_send_log_bundle_exist(self.directory, self.jobid)
            self.extract_send_logs_files()
            self.verify_scrub_logs(client_id, False)
            self.machine.remove_directory(self.directory)
        except Exception as exp:
            handle_testcase_exception(self, exp)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
