# coding=utf-8
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

    setup_environment() -- configures entities based on inputs

    get_active_files_store() -- gets active files DDB store id

    cleanup()   --  cleanups all created entities

    run_backup()    -- runs backup need for the case

    run_data_aging()    -- runs data aging job for storage policy copy created by case

    run_dv2_job()   -- runs DV2 job with options provided

    validate_dv2() -- validates the DDB verification job

Note: if no predefined entities provided in input, we will create them.
if need to use predefined entities like Library or SP. add them to input json

Sample JSON: values under [] are optional
"58933": {
            "ClientName": "nayavm",
            "AgentName": "File System",
            "MediaAgentName": "nayavm",
            ["DDBPath": "E:\\DDBs\\dv2tests\\0",
            "MountPath": "E:\\Libraries\\dv2tests_defragmp",
            "LibraryName": "dv2test",
            "StoragePolicyName": "33_sp",
            "BackupsetName": "33_bs",
            "SubclientName": "33_sc",
            "ScaleFactor": "ScaleFactor"]
        }

design:
    run backup J1
    run quick full dv2
    Validate verification time
    run backup J2
    run quick incr dv2
    Validate verification time
    run backup J3
    run complete full dv2
    Validate verification time
    run backup J4
    run complete incr dv2
    Validate verification time
"""

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import MMHelper
from MediaAgents.MAUtils.mahelper import DedupeHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Simplify DV2 project - DV2 new GUI options case"
        self.tcinputs = {
            "MediaAgentName": None
        }
        self.library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.content_path = None
        self.mount_path = None
        self.agent_name = None
        self.ddb_path = None
        self.scale_factor = None
        self.mmhelper = None
        self.dedupehelper = None
        self.client_machine = None
        self.library = None
        self.storage_policy = None
        self.backupset = None
        self.subclient = None
        self.clients = None
        self.client = None
        self.agent = None
        self.primary_copy = None
        self.existing_entities = True

    def setup(self):
        """
        Setup function of this test case. initialises entity names if provided in tcinputs otherwise assigns new names
        """
        self.library_name = self.tcinputs.get('LibraryName')
        self.storage_policy_name = self.tcinputs.get('StoragePolicyName')
        self.backupset_name = self.tcinputs.get('BackupsetName')
        self.subclient_name = self.tcinputs.get('SubclientName')
        self.content_path = self.tcinputs.get('ContentPath')
        self.mount_path = self.tcinputs.get('MountPath')
        self.agent_name = self.tcinputs.get('AgentName')
        self.ddb_path = self.tcinputs.get('DDBPath')
        self.scale_factor = self.tcinputs.get('ScaleFactor')
        self.client_machine = Machine(self.tcinputs['ClientName'], self.commcell)
        if not self.subclient_name:
            self.subclient_name = str(self.id) + "_SC"
            self.existing_entities = False
        if self._subclient:
            self.subclient = self._subclient

        if not self.library_name:
            self.library_name = str(self.id) + "_lib"
            self.existing_entities = False

        if not self.storage_policy_name:
            self.storage_policy_name = str(self.id) + "_SP"
            self.existing_entities = False

        if not self.backupset_name:
            self.backupset_name = str(self.id) + "_BS"
            self.existing_entities = False
        # if self._backupset:
        #     self.backupset = self._backupset

        self.mmhelper = MMHelper(self)
        self.dedupehelper = DedupeHelper(self)

    def setup_environment(self):
        """ configures all entities based tcinputs. if subclient or storage policy or library is provided,
        TC will use these entities and run the case """
        self.log.info("setting up environment...")


        do_subclient_association = False
        if self._subclient:
            self.content_path = self.subclient.content[0]
            if not self.subclient.storage_policy:
                do_subclient_association = True
            else:
                self.storage_policy_name = self.subclient.storage_policy

        if not do_subclient_association and self.commcell.storage_policies.has_policy(self.storage_policy_name):
            self.storage_policy = self.commcell.storage_policies.get(self.storage_policy_name)
            self.library_name = self.storage_policy.library_name

        # select drive on MA for MP and DDB
        op_selector = OptionsSelector(self.commcell)
        media_agent_machine = Machine(self.tcinputs['MediaAgentName'], self.commcell)
        media_agent_drive = media_agent_machine.join_path(
            op_selector.get_drive(media_agent_machine), 'automation', self.id)
        if not self.mount_path:
            self.mount_path = media_agent_machine.join_path(media_agent_drive, 'mountpath')
        if not self.ddb_path:
            self.ddb_path = media_agent_machine.join_path(media_agent_drive, 'DDB')

        # select drive on client for content and restore
        client_drive = self.client_machine.join_path(
            op_selector.get_drive(self.client_machine), 'automation', self.id)
        if not self.content_path:
            self.content_path = self.client_machine.join_path(client_drive, 'content_path')

        if not self.library:
            self.library = self.mmhelper.configure_disk_library(self.library_name,
                                                                self.tcinputs["MediaAgentName"],
                                                                self.mount_path)

        if not self.storage_policy:
            self.storage_policy = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_name,
                                                                                    self.library.name,
                                                                                    self.tcinputs["MediaAgentName"],
                                                                                    self.ddb_path)
        self.primary_copy = self.storage_policy.get_copy('Primary')

        if not self._backupset:
            self.mmhelper.configure_backupset(self.backupset_name, self._agent)

        if not self._subclient:
            self.subclient = self.mmhelper.configure_subclient(self.backupset_name,
                                                               self.subclient_name,
                                                               self.storage_policy_name,
                                                               self.content_path,
                                                               self._agent)
        elif do_subclient_association:
            self.subclient.storage_policy = self.storage_policy_name

    def cleanup(self):
        """
        performs cleanup of all entities only if created by case.
        if case is using existing entities, cleanup is skipped.
        """
        try:
            if not self.existing_entities:
                flag = 0
                self.log.info("cleanup started")
                if self._agent.backupsets.has_backupset(self.backupset_name):
                    self.log.info("deleting backupset...")
                    self._agent.backupsets.delete(self.backupset_name)
                    flag = 1
                if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                    self.log.info("deleting storage policy...")
                    self.commcell.storage_policies.delete(self.storage_policy_name)
                    flag = 1
                if self.commcell.disk_libraries.has_library(self.library_name):
                    self.log.info("deleting library...")
                    self.commcell.disk_libraries.delete(self.library_name)
                    flag = 1
                if not flag:
                    self.log.info("no entities found to clean up!")
                else:
                    self.log.info("cleanup done.")
            else:
                self.log.info("not running cleanup as case is using existing entities.")
                additional_content = self.client_machine.join_path(self.content_path, 'generated_content')
                if self.client_machine.check_directory_exists(additional_content):
                    self.log.info("deleting additional content...")
                    self.client_machine.remove_directory(additional_content)
        except Exception:
            self.log.warning("Something went wrong while cleanup!")

    def run_backup(self, backup_type="FULL", size=1.0):
        """
        this function runs backup by generating new content to get unique blocks for dedupe backups.
        if scalefactor is set in tcinput, creates factor times of backup data

        Args:
            backup_type (str): type of backup to run
                Default - FULL

            size (int): size of backup content to generate
                Default - 1 GB

        Returns:
        (object) -- returns job object to backup job
        """
        # add content
        additional_content = self.client_machine.join_path(self.content_path, 'generated_content')
        if self.client_machine.check_directory_exists(additional_content):
            self.client_machine.remove_directory(additional_content)
        # if scale test param is passed in input json, multiple size 10 times and generate content
        if self.scale_factor:
            size = size * int(self.scale_factor)
        self.mmhelper.create_uncompressable_data(self.client, additional_content, size)

        self.log.info("Running %s backup...", backup_type)
        job = self.subclient.backup(backup_type)
        self.log.info("Backup job: %s", job.job_id)
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup with error: {1}".format(backup_type, job.delay_reason)
            )
        self.log.info("Backup job completed.")
        return job

    def get_active_files_store(self):
        """returns active store object for files iDA"""
        self.commcell.deduplication_engines.refresh()
        engine = self.commcell.deduplication_engines.get(self.storage_policy_name, 'primary')
        for store in engine.all_stores:
            if "_files_" in store[1].lower() and store[2] == 'active':
                return engine.get(store[0])
        return 0

    def run_dv2_job(self, store, dv2_type, option):
        """
        Runs DV2 job with type and option selected and waits for job to complete

        Args:
            store (object) - object of the store to run DV2 job on

            dv2_type (str) - specify type either full or incremental

            option (str) - specify option, either quick or complete

        Returns:
             (object) - completed DV2 job object
        """

        self.log.info("running [%s] [%s] DV2 job on store [%s]...", dv2_type, option, store.store_id)
        if dv2_type == 'incremental' and option == 'quick':
            job = store.run_ddb_verification()
        elif dv2_type == 'incremental' and option == 'complete':
            job = store.run_ddb_verification(quick_verification=False)
        elif dv2_type == 'full' and option == 'quick':
            job = store.run_ddb_verification(incremental_verification=False)
        else:
            job = store.run_ddb_verification(incremental_verification=False, quick_verification=False)
        self.log.info("DV2 job: %s", job.job_id)
        if not job.wait_for_completion():
            raise Exception(f"Failed to run dv2 job with error: {job.delay_reason}")
        self.log.info("DV2 job completed.")
        return job

    def validate_dv2(self, store, dv2_job, is_quick_dv2=True, incr_job_list=None, is_incremental=False):
        """
        validates the dv2 job for following:
        1. idxsidbStore table updation with right column based on option
        2. 2 phases for each dv2 job
        3. last verification time on expected jobs for incr type

        Args:
            store (object)          - store object to get dedupe store related details

            dv2_job (object)        - dv2 job object

            incr_job_list (list)    - list of jobs that are expected to be verified by incr DV2
                                    Default: None

            is_quick_dv2 (bool)     - set false if completed DV2 job validation
                                    Default: True

            is_incremental (bool)   - set true if incremental DV2 job. Default: False
                                    Note: need to pass expected jobs in set to true in incr_job_list
        """
        if incr_job_list is None:
            incr_job_list = []
        if is_quick_dv2:
            # quick dv2 time update
            self.log.info("VALIDATION: is lastQuickDDBVerificationTime updated after Quick dv2 job?")
            query = f"""select 1 from IdxSidbStore S, JMAdminJobStatsTable JM
                    where S.lastQuickDDBVerificationTime = JM.servstart
                    and S.sidbstoreid = {store.store_id} and JM.jobid = {dv2_job.job_id}
                    and S.LastDDBVerificationTime <> JM.servstart"""
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            result = self.csdb.fetch_one_row()
            self.log.info("RESULT: %s", result[0])
            if not result[0]:
                raise Exception("lastQuickDDBVerificationTime is not updated after quick DV2 job")
            self.log.info("lastQuickDDBVerificationTime is updated after Quick dv2 job!")
        else:
            # complete dv2 time update
            self.log.info("VALIDATION: is LastDDBVerificationTime updated after complete dv2 job?")
            query = f"""select 1 from IdxSidbStore S, JMAdminJobStatsTable JM
                                where S.lastQuickDDBVerificationTime <> JM.servstart
                                and S.sidbstoreid = {store.store_id} and JM.jobid = {dv2_job.job_id}
                                and S.LastDDBVerificationTime = JM.servstart"""
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            result = self.csdb.fetch_one_row()
            self.log.info("RESULT: %s", result[0])
            if not result[0]:
                raise Exception("LastDDBVerificationTime is not updated after complete DV2 job")
            self.log.info("LastDDBVerificationTime is updated after complete dv2 job!")

        # validate phase 2 for quick dv2
        self.log.info("VALIDATION: phase 2 for DV2 job")
        query = f"""select count(distinct phasenum) from JMAdminJobAttemptStatsTable where jobid = {dv2_job.job_id}"""
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info("RESULT: %s", result[0])
        if result[0] == '' or int(result[0]) != 2:
            raise Exception("invalid phase count found for DV2 job")
        self.log.info("DV2 job was run with 2 phases!")

        # validate verification time update on job
        # Note: for incremental dv2, by design, we would have run full before incr. this means, all incr expected jobs
        #       are noted down in case thus only these jobs should be considered for incr verification.
        if is_incremental:
            self.log.info("VALIDATE: only expected jobs should be updated with verified time")
            query = f"""select distinct JMD.jobid from JMJobDataStats JMD, JMAdminJobStatsTable JMA
                    where JMD.archCheckEndTime between JMA.servStart and JMA.servEnd
                        and JMA.jobid = {dv2_job.job_id}"""
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            result = self.csdb.fetch_all_rows()
            self.log.info("RESULT: %s", result[0])
            if not len(incr_job_list) == len(result[0]):
                unexpected_job_list = list(set(result[0]) - set(incr_job_list))
                raise Exception(f"unexpected jobs verified by incr DV2 {unexpected_job_list}")
            self.log.info("only expected jobs are updated with incremental DV2")

    def run(self):
        """Run function of this test case"""
        try:
            self.cleanup()
            self.setup_environment()
            self.run_backup()
            store = self.get_active_files_store()
            quick_dv2 = self.run_dv2_job(store, 'full', 'quick')
            self.validate_dv2(store, quick_dv2)

            job = self.run_backup()
            quick_incr_dv2 = self.run_dv2_job(store, 'incremental', 'quick')
            self.validate_dv2(store, quick_incr_dv2, incr_job_list=[job], is_incremental=True)

            self.run_backup()
            complete_dv2 = self.run_dv2_job(store, 'full', 'complete')
            self.validate_dv2(store, complete_dv2, is_quick_dv2=False)

            job = self.run_backup()
            complete_incr_dv2 = self.run_dv2_job(store, 'incremental', 'complete')
            self.validate_dv2(store, complete_incr_dv2, is_quick_dv2=False, incr_job_list=[job], is_incremental=True)
            self.cleanup()

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED
