# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test case to verify creating and updating vmware replication type recovery target

Sample input:
 "58903": {
                          "ClientName": "client_name",
                          "access_node": "node1",
                          "destination_host":"host_name_1",
                          "vm_folder" : "DRFolder",
                          "destination_network":"Lab_10045",
                          "access_node_2": "pingvsaproxy",
                          "destination_host_2":"host_name_2",
                          "vm_folder_2" : "Discovered virtual machine",
                          "destination_network_2":"Lab_7204"
                     }

"""
from Web.AdminConsole.DR.recovery_targets import RecoveryTargets, TargetConstants
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """Test case to verify creating and updating recovery target"""
    test_step = TestStep()

    def setup(self):
        """Initialize required veriables/inputs"""
        self.destination_hypervisor = self.tcinputs["ClientName"]
        self.access_node = self.tcinputs["access_node"]
        self.edited_access_node = self.tcinputs["access_node_2"]
        self.destination_host = self.tcinputs["destination_host"]
        self.vm_folder = self.tcinputs["vm_folder"]
        self.destination_network = self.tcinputs["destination_network"]
        self.edited_destination_network = self.tcinputs["destination_network_2"]
        self.target_name = 'Auto_target_' + self.id
        self.target_vendor = TargetConstants.VMWARE_VCENTER
        self.target_application_type = TargetConstants.REPLICATION
        self.vm_display_name = "Auto_vm_" + self.id
        self.edited_target_name = 'Auto_edit_' + self.id
        self.edited_vm_display_name = "Auto_edited_vm_" + self.id
        self.expected_list = [self.destination_hypervisor, self.access_node, self.vm_display_name,
                              self.destination_host, self.vm_folder, self.destination_network]
        self.expected_updated_list = [self.destination_hypervisor, self.edited_access_node,
                                      self.edited_vm_display_name,
                                      self.destination_host, self.vm_folder,
                                      self.edited_destination_network]
        self.expected_non_editable_fields = ['displayNamePrefixSuffix', 'destinationHost',
                                             'dataStore', 'resourcePool', 'vmFolder']

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "VMWare recovery target: Verify creating and updating recovery target"
        self.browser = None
        self.utils = None
        self.admin_console = None
        self.navigator = None
        self.recovery_targets = None
        self.destination_hypervisor = None
        self.access_node = None
        self.destination_host = None
        self.target_name = None
        self.target_vendor = None
        self.target_application_type = None
        self.vm_display_name = None
        self.vm_folder = None
        self.destination_network = None
        self.target = None
        self.target_details = None
        self.edited_target_name = None
        self.edited_vm_display_name = None
        self.edited_access_node = None
        self.edited_destination_host = None
        self.edited_vm_folder = None
        self.edited_destination_network = None
        self.expected_list = None
        self.expected_updated_list = None
        self.recovery_target = None
        self.expected_non_editable_fields = None
        self.utils = TestCaseUtils(self)

    def init_commandcenter(self):
        """Initialize browser and redirect to page"""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.recovery_targets = RecoveryTargets(self.admin_console)
            self.navigator.navigate_to_replication_targets()
        except Exception as _exception:
            raise CVTestCaseInitFailure(_exception) from _exception

    @test_step
    def delete_recovery_target(self):
        """Delete recovery target"""
        self.navigator.navigate_to_replication_targets()
        self.recovery_targets.delete_recovery_target(self.edited_target_name)
        if self.recovery_targets.has_target(self.edited_target_name):
            raise CVTestStepFailure("Recovery target [%s] could not be deleted" %
                                    self.edited_target_name)
        if self.recovery_targets.has_target(self.target_name):
            raise CVTestStepFailure("Recovery target [%s] could not be deleted" % self.target_name)
        self.log.info("Recovery target [%s] deleted!", self.target_name)

    @test_step
    def configure_recovery_target(self):
        """Configure recovery target"""
        self.recovery_target = self.recovery_targets.configure_vmware_recovery_target(
            self.target_name, self.target_application_type)
        self.recovery_target.select_destination_hypervisor(self.destination_hypervisor)
        self.recovery_target.select_access_node(self.access_node)
        self.recovery_target.set_vm_display_name(self.vm_display_name)
        self.recovery_target.set_destination_host(self.destination_host)
        self.recovery_target.set_vm_folder(self.vm_folder)
        self.recovery_target.select_destination_network(self.destination_network)
        self.recovery_target.save()
        self.log.info("Created [%s] recovery target successfully!" % self.target_name)

    @test_step
    def verify_target_details(self, expected_list):
        """Verify recovery target details are expected"""
        dest_hypervisor, access_node, vm_display_name, destination_host, vm_folder, \
        destination_network = expected_list
        summary = self.target_details.get_target_summary()
        error_string = "For [%s] field expected [%s], but found [%s]"

        if dest_hypervisor != summary['Destination hypervisor']:
            raise CVTestStepFailure(error_string %
                                    ('Destination hypervisor', dest_hypervisor,
                                     summary['Destination hypervisor']))

        if vm_display_name != summary['VM display name ( Prefix )']:
            raise CVTestStepFailure(error_string %
                                    ('VM display name ( Prefix )', vm_display_name,
                                     summary['VM display name ( Prefix )']))

        if access_node != summary['Access node']:
            raise CVTestStepFailure(error_string %
                                    ('Access node', access_node, summary['Access node']))

        if destination_host != summary['Destination host']:
            raise CVTestStepFailure(error_string %
                                    ('Destination host', destination_host,
                                     summary['Destination host']))

        if vm_folder != summary['VM folder']:
            raise CVTestStepFailure(error_string %
                                    ('VM folder', vm_folder, summary['VM folder']))

        if destination_network != summary['Destination network']:
            raise CVTestStepFailure(error_string %
                                    ('Destination network', destination_network,
                                     summary['Destination network']))
        self.log.info("Verified recovery target details successfully!")

    @test_step
    def verify_non_editable_fields(self):
        """Verify specific fields are not editable in edit panel"""
        target = self.target_details.edit_target()
        for each_field in self.expected_non_editable_fields:
            if not target.is_field_disabled(each_field):
                raise CVTestStepFailure("[%s] field is editable in Edit panel of [%s] target, "
                                        "Please check this should not be editable!" %
                                        (each_field, self.target_name))
        target.cancel()
        self.log.info("Verified non editable fields in edit panel of recovery target")

    @test_step
    def edit_target(self):
        """Verify editable and non editable fields of recovery target"""
        target = self.target_details.edit_target(name=self.edited_target_name)
        target.select_access_node(self.edited_access_node)
        target.set_vm_display_name(self.edited_vm_display_name)
        target.select_destination_network(self.edited_destination_network)
        target.save()
        self.log.info("Updated recovery target successfully!")

    def delete_existing_recovery_targets(self):
        """Delete recovery targets if they exists"""
        for target in [self.target_name, self.edited_target_name]:
            if self.recovery_targets.has_target(target):
                self.recovery_targets.delete_recovery_target(target)

    def verify_recovery_targets_page_details(self):
        """Verify recovery targets page has correct details associated"""
        details = self.recovery_targets.get_target_details(self.target_name)

        error_string = f"For [%s] column expected [%s], but found [%s] for recovery target " \
                       f"{self.target_name} on Recovery targets page"

        if details['Name'] != [self.target_name]:
            raise CVTestStepFailure(error_string % ('Name', self.target_name, details['Name']))
        if details['Vendor'] != ['VMware']:
            raise CVTestStepFailure(error_string % ('Vendor', 'VMware', details['Vendor']))
        if details['Application type'] != [self.target_application_type]:
            raise CVTestStepFailure(error_string % ('Application type',
                                                    self.target_application_type,
                                                    details['Application type']))
        if details['Destination hypervisor'] != [self.destination_hypervisor]:
            raise CVTestStepFailure(error_string % ('Destination hypervisor',
                                                    self.destination_hypervisor,
                                                    details['Destination hypervisor']))
        self.log.info("Verified recovery target table details!")

    def run(self):
        """Starting test case steps"""
        try:
            self.init_commandcenter()
            self.delete_existing_recovery_targets()
            self.configure_recovery_target()
            self.verify_recovery_targets_page_details()
            self.target_details = self.recovery_targets.access_target(self.target_name)
            self.verify_target_details(self.expected_list)
            self.verify_non_editable_fields()
            self.edit_target()
            self.admin_console.refresh_page()
            self.verify_target_details(self.expected_updated_list)
            self.delete_recovery_target()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
