# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""
import calendar
import time
import datetime

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase

from dynamicindex.Datacube.dcube_sync_helper import SyncHelper
from dynamicindex.Datacube.dcube_solr_helper import SolrHelper
from dynamicindex.Datacube.data_source_helper import DataSourceHelper
from dynamicindex.Datacube.crawl_job_helper import CrawlJobHelper
from dynamicindex.utils import constants as dynamic_constants


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs    (dict)          --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.datasource_type = dynamic_constants.FILE_SYSTEM_DSTYPE
        self.datasource_id = None
        self.datasource_obj = None
        self.timestamp = None
        self.solr_core_name = None
        self.total_crawlcount = 0
        self.fs_data_source_name = "DcubeSyncAutomation"
        self.dcube_helper = None
        self.ds_helper = None
        self.crawl_helper = None
        self.dssync_regname = "nLastDCubeSyncTime"
        self.sync_helper = None
        self.file_ds_prefix = "DC_file_"
        self.entities_to_extract = {
            "RER": 'Email,CreditCard'
        }
        self.name = "DataCube FileSystem Sync Status to CS DataBase with EntityExtraction Enabled"
        self.tcinputs = {
            "IndexServerName": None,
            "IncludedirectoriesPath": None,
            "UserName": None,
            "Password": None,
            "AccessNode": None,
            "CANode": None
        }

    def setup(self):
        """Setup function of this test case"""

        try:
            self.sync_helper = SyncHelper(self)
            self.dcube_helper = SolrHelper(self)
            self.crawl_helper = CrawlJobHelper(self)
            self.ds_helper = DataSourceHelper(self.commcell)
            self.timestamp = calendar.timegm(time.gmtime())
            self.fs_data_source_name = f"{self.fs_data_source_name}{self.timestamp}"

            self.log.info("Going to create file system data source : %s", self.fs_data_source_name)

            access_node_client_obj = self.commcell.clients.get(
                self.tcinputs['AccessNode'])
            self.log.info("Client object Initialised")
            access_node_clientid = access_node_client_obj.client_id
            self.log.info("Accessnode Client id : %s", str(access_node_clientid))
            caconfig = self.ds_helper.form_entity_extraction_config(entities=self.entities_to_extract,
                                                                    entity_fields=["content"])
            self.log.info("Going to get CA cloud details for : %s", self.tcinputs['CANode'])
            ca_cloud_obj = self.commcell.content_analyzers.get(self.tcinputs['CANode'])
            self.log.info("CA cloud URL : %s", ca_cloud_obj.cloud_url)
            fs_dynamic_property = {
                "includedirectoriespath": self.tcinputs['IncludedirectoriesPath'],
                "username": self.tcinputs['UserName'],
                "password": self.tcinputs['Password'],
                "accessnodeclientid": access_node_clientid,
                "iscaenabled": "true",
                "caconfig": caconfig,
                "cacloudid": str(ca_cloud_obj.cloud_id)
            }

            file_properties = self.ds_helper.form_file_data_source_properties(fs_dynamic_property)

            self.datasource_obj = self.ds_helper.create_file_data_source(data_source_name=self.fs_data_source_name,
                                                                         index_server_name=self.tcinputs[
                                                                             'IndexServerName'],
                                                                         fs_properties=file_properties)
            self.datasource_id = self.datasource_obj.datasource_id

            self.crawl_helper.monitor_crawl_job(self.fs_data_source_name)

            self.log.info("Going to Get Status for this data source")
            datasource_status = self.datasource_obj.get_status()
            self.total_crawlcount = datasource_status['status']['totalcount']
            if self.total_crawlcount is not None and self.total_crawlcount == 0:
                raise Exception("Total crawled data count is zero. Please check")
            self.log.info("Crawled Data Count : %s", str(self.total_crawlcount))

        except Exception as ee:
            self.log.error('Setup for the test case failed.')
            self.log.exception(ee)
            self.result_string = str(ee)
            self.status = constants.FAILED
            raise Exception("Test case setup(Creation of Datasource failed). Please check")

    def run(self):
        """Run function of this test case"""
        try:

            current_time = datetime.datetime.now()
            self.sync_helper.set_dcube_sync_registry(current_time, 2, 24, True)
            statusxml = self.sync_helper.get_syncxml_by_csrestart(self.datasource_id)
            self.solr_core_name = self.datasource_obj.computed_core_name
            solr_url = self.dcube_helper.get_solr_baseurl(self.tcinputs['IndexServerName'], 1)
            self.log.info("Base solr URl : %s", solr_url)

            stats_response = self.dcube_helper.get_corestats(solr_url, self.solr_core_name)
            self.log.info("Core stats response : %s", str(stats_response))

            fsstats_response = self.dcube_helper.get_fs_sync_facets(solr_url, self.solr_core_name)
            self.log.info("FS Core specific stats response : %s", str(fsstats_response))

            # validation starts
            testresult = self.sync_helper.verify_sync_xml(
                statusxml, stats_response, str(int(current_time.timestamp())), fsstats_response)
            if testresult:
                self.log.info("Sync xml validation passed")
            else:
                raise Exception("Sync xml validation failed")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: ')
            self.result_string = str(exp)
            self.log.exception(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.datasource_obj is not None and self.status == constants.PASSED:
            self.log.info("Delete the data source with id = %s", str(self.datasource_id))
            self.datasource_obj.delete_datasource()
