# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: This test case verifies whether Azure replication group creation succeeds with default options.
Then it waits for the replication and backup jobs to complete successfully, and finally verifies the details on the
replication monitor

Main file for executing this test case(verify the creation and updation of Azure recovery target
of replication type)

TestCase: Class for executing this test case
Sample JSON: {
        "ClientName": "Azure",
        "source_vm": "vm"
        "recovery_target": "AzureTarget",
        "storage_name" : "storage",
        "resource_pool": None,
        "vm_folder": None,
        "network_1": None,
        "network_2": None,
        ""
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from VirtualServer.VSAUtils import VirtualServerUtils, OptionsHelper
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from Web.AdminConsole.DR.monitor import ReplicationMonitor
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the replication of Defect #288874"""
    test_step = TestStep()
    _AGENT_NAME = 'virtual server'
    _INSTANCE_NAME = 'vmware'
    _BACKUPSET_NAME = 'defaultBackupSet'

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Azure Replication group: Verify Creation and edit operation"

        self.tcinputs = {
            "ClientName": None,
            "source_vm": None,
            "recovery_target": None,
            "storage_name": None,
            "secondary_storage_name": None,
        }
        self.utils = None

        self.browser = None
        self.admin_console = None

        self.replication_group = None
        self.replication_monitor = None

        self._source_subclient = None
        self.live_sync_utils = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    @property
    def schedule(self):
        """Returns the name of the schedule"""
        return self.replication_group.get_schedule_name_by_replication_group(self.group_name)

    @property
    def source_subclient(self):
        """Returns the VSA Auto subclient after initialising all the testcase objects"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        if not self._source_subclient:
            self._source_subclient = VirtualServerUtils.subclient_initialize(self)
        return self._source_subclient

    @property
    def destination_vm_name(self):
        """Gets the destination VM name from the source subclient"""
        return self.source_subclient.get_live_sync_destination_vms(self.schedule)[0]

    @property
    def backup_job_id(self):
        """Get the latest backup job obj"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        return self.subclient.find_latest_job().job_id

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.admin_console.wait_for_completion()

        self.replication_group = ReplicationGroup(self.admin_console)
        self.replication_monitor = ReplicationMonitor(self.admin_console)

    def logout(self):
        """Logs out of the admin console and closes the browser"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)

            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The repication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        vmware_configure = self.replication_group.configure_vmware()
        vmware_configure.content.set_name(self.group_name)
        vmware_configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                            {"VMs and templates": [self.tcinputs['source_vm']]})
        sleep(5)
        vmware_configure.next()

        vmware_configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        vmware_configure.target.unconditionally_overwrite_vm(True)
        vmware_configure.next()

        vmware_configure.storage_cache.select_storage(self.tcinputs['storage_name'])
        vmware_configure.storage_cache.select_secondary_storage(self.tcinputs['secondary_storage_name'])

        vmware_configure.next()
        vmware_configure.next()

        sleep(5)
        vmware_configure.finish()

    @test_step
    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        self.log.info('Waiting for backup job %s to finish', self.backup_job_id)
        job_obj = self.commcell.job_controller.get(self.backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_job_completion(self):
        """Waits for replication job to complete and then verify its details"""
        self.log.info("Waiting 2 minutes to let live sync update")
        sleep(120)
        self.live_sync_utils = LiveSyncUtils(self.source_subclient, self.schedule)
        job_obj = self.live_sync_utils.get_recent_replication_job(self.backup_job_id)
        self.log.info("Waiting for replication job id %s to finish", job_obj.job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def disable_group(self):
        """Wait for backup and replication to finish and then disable the replication group sync"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.disable(self.group_name)

    @test_step
    def perform_backups(self):
        """Adds data to the VM and then performs different backups
            1. Incremental
            2. Full
            3. Incremental
        """
        backup_options = OptionsHelper.BackupOptions(self.source_subclient)
        backup_options.backup_type = "INCREMENTAL"
        self.source_subclient.backup(backup_options, skip_discovery=True)
        self.verify_backup_job_completion()

        backup_options.backup_type = "FULL"
        self.source_subclient.backup(backup_options, skip_discovery=True)
        self.verify_backup_job_completion()

        backup_options.backup_type = "INCREMENTAL"
        self.source_subclient.backup(backup_options, skip_discovery=True)
        self.verify_backup_job_completion()

    @test_step
    def enable_group(self):
        """Enable group and then wait for replication to complete"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.enable(self.group_name)

    @test_step
    def validate_live_sync(self):
        """Validates the live sync is performed"""
        self.live_sync_utils.validate_live_sync(replication_run=False,
                                                check_replication_size=False,
                                                schedule=self.schedule)

    @test_step
    def perform_test_boot(self):
        """Perform test boot on the destination VM"""
        test_boot_job_id = self.replication_monitor.test_boot_vm(self.tcinputs['source_vm'], self.group_name)
        job_obj = self.commcell.job_controller.get(test_boot_job_id)

        self.logout()
        self.log.info('Waiting for Job [%s] to complete', test_boot_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    def run(self):
        """Runs the testcase in order"""
        try:
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()

            self.logout()
            self.verify_backup_job_completion()
            self.verify_replication_job_completion()

            self.login()
            self.disable_group()
            self.logout()

            self.perform_backups()

            self.login()
            self.enable_group()
            self.logout()
            self.verify_replication_job_completion()

            self.validate_live_sync()

            self.login()
            self.admin_console.navigator.navigate_to_replication_monitor()
            self.perform_test_boot()

            self.login()
            self.delete_virtualization_group(wait_for_sync=False)

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        self.logout()
