# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase to perform Restore from Archive tier of Azure or Amazon using Cloud Archive Recall Workflow.

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                  --  initialize TestCase class

    _get_wfjobid(self, restore_job_id)
                                -- Get the workflow Job id for Cloud archive Recall running at the time on commcell.

    _get_restorejobid_bywfjobid(self, wf_jobid)
                                -- Get the restore Job id which was input for the Cloud archive Recall Job running
                                in the commcell.

    _cleanup()                  --  Cleanup the entities created

    set_multiple_readers()        -- Allow multiple data readers to subclient

    setup()                     --  setup function of this test case

    run()                       --  run function of this test case

    tear_down()                 --  teardown function of this test case
"""
import time
from cvpysdk.job import Job
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.idautils import CommonUtils
from MediaAgents.MAUtils.mahelper import (DedupeHelper, MMHelper)


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializing the Test case file"""
        super(TestCase, self).__init__()
        self.name = "Restore from Archive tier of Azure or Amazon using Cloud Archive Recall Workflow."
        self.tcinputs = {
            "ClientName": None,
            "MediaAgentName": None,
            "CloudMountPath": None,
            "CloudUserName": None,
            "CloudPassword": None,
            "CloudServerType": None
        }
        self.cloud_library_name = None
        self.storage_policy_dedupe_name = None
        self.backupset_name = None
        self.client_machine = None
        self.ma_machine = None
        self.partition_path = None
        self.content_path1 = None
        self.restore_dest_path = None
        self.common_util = None
        self.dedupehelper = None
        self.mmhelper = None

    def _get_wfjobid(self, restore_job_id):
        """
        Get the workflow Job id for Cloud archive Recall running at the time on commcell.
        Args:
            (Int)   --  Restore Job ID, to filter off query results.
        Return:
            (Int)   --  workflow Job Id list.
        """
        for _ in range(0, 10):
            query = f""" SELECT Jobid FROM JMJobInfo WHERE opType = 90
                    AND cast(xmlJobInfo as xml).value('(/Workflow_StartWorkflow/workflow/@workflowId)[1]', 'int') =
                     {self.commcell.workflows.get('Cloud Storage Archive Recall').workflow_id}"""
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            cur_wf = self.csdb.fetch_all_rows()
            if cur_wf[0] != ['']:
                # workflow job ID should be greater than Restore Job ID.
                cur_wf = [int(x[0]) for x in cur_wf if int(x[0]) > restore_job_id]
                if len(cur_wf) > 0:
                    self.log.info("RESULT: %s", cur_wf)
                    return cur_wf
            self.log.info("Sleep for 20 sec")
            time.sleep(20)

        self.log.error("Failed to get the workflow Job id for Cloud archive Recall")
        raise Exception("Failed to get the workflow Job id for Cloud archive Recall")

    def _get_restorejobid_bywfjobid(self, wf_jobid):
        """
        Get the restore Job id which was input for the Cloud archive Recall Job running in the commcell.
        Args:
            (Int)   wf_jobid -- Cloud Archive recall work flow job id.
        Return:
            (List)  job_id_r -- Input Job id which started cloud archive recall workflow job
        """

        if self.commcell.workflows.has_workflow('Cloud Storage Archive Recall'):
            query = f"""select cast(xmlJobInfo as xml).value('(/Workflow_StartWorkflow/@options)[1]', 'nvarchar(max)')
                    from JMJobStats
                    where (jobId={wf_jobid})
                    and xmlJobInfo like '%workflowId%'
                    and cast(xmlJobInfo as xml).value('(/Workflow_StartWorkflow/workflow/@workflowId)[1]', 'int')= 
                    {self.commcell.workflows.get('Cloud Storage Archive Recall').workflow_id}
                    """
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            option = self.csdb.fetch_one_row()[0]
            self.log.info("option RESULT: %s", option)
            query = f"""select cast(REPLACE('{option}',
                        '<?xml version="1.0" encoding="UTF-8" standalone="no"?>','') as xml)
                        .value('(/inputs/job_id/text())[1]', 'int')
                        """
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            job_id_r = int(self.csdb.fetch_one_row()[0])
            self.log.info("RESULT: %s", job_id_r)
            return job_id_r
        self.log.error("No workflow with name Cloud Storage Archive Recall")
        raise Exception("No workflow with name Cloud Storage Archive Recall")

    @staticmethod
    def set_multiple_readers(subclient, num_readers):
        """
            Allow multiple data readers to subclient
            Args:
                subclient          (object) --  instance of sub-client to set data readers
                num_readers        (int)    --  Number of data readers
        """

        subclient.allow_multiple_readers = True
        subclient.data_readers = num_readers

    def _cleanup(self):
        """Cleanup the entities created"""
        self.log.info("********************** CLEANUP STARTING *************************")
        try:
            # Delete backup-set
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting Backup-set: %s ", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)
                self.log.info("Deleted Backup-set: %s", self.backupset_name)
            # Delete Storage Policies
            if self.commcell.storage_policies.has_policy(self.storage_policy_dedupe_name):
                self.log.info("Deleting storage policy: %s", self.storage_policy_dedupe_name)
                self.commcell.storage_policies.delete(self.storage_policy_dedupe_name)
                self.log.info("Deleted storage policy: %s", self.storage_policy_dedupe_name)
            # Delete Cloud Library
            if self.commcell.disk_libraries.has_library(self.cloud_library_name):
                self.log.info("Deleting library: %s", self.cloud_library_name)
                self.commcell.disk_libraries.delete(self.cloud_library_name)
                self.log.info("Deleted library: %s", self.cloud_library_name)
            # Clean up Partition path if the path exists.
            if self.ma_machine.check_directory_exists(self.partition_path):
                self.log.info("Cleaning up the partition path %s", self.partition_path)
                self.ma_machine.remove_directory(self.partition_path)
        except Exception as exp:
            self.log.warning("Error encountered during cleanup : %s", str(exp))

        self.log.info("********************** CLEANUP COMPLETED *************************")

    def setup(self):
        """Setup function of this test case"""
        options_selector = OptionsSelector(self.commcell)

        self.cloud_library_name = '%s_cloud' % str(self.id)
        self.storage_policy_dedupe_name = '%s_dedupe' % str(self.id)
        self.backupset_name = '%s_bs' % str(self.id)
        self.client_machine = Machine(self.client)
        self.ma_machine = Machine(self.tcinputs['MediaAgentName'], self.commcell)
        # To select drive with space available in client machine
        self.log.info('Selecting drive in the client machine based on space available')
        client_drive = options_selector.get_drive(self.client_machine, size=5 * 1024)
        if client_drive is None:
            raise Exception("No free space for content on client machine.")
        self.log.info('selected drive: %s', client_drive)
        # To select drive with space available in Media agent machine
        self.log.info('Selecting drive in the Media agent machine based on space available')
        ma_drive = options_selector.get_drive(self.ma_machine, size=5 * 1024)
        if ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', ma_drive)

        self.partition_path = self.ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'DDB')

        self.content_path1 = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'Testdata1')
        if self.client_machine.check_directory_exists(self.content_path1):
            self.client_machine.remove_directory(self.content_path1)
        self.client_machine.create_directory(self.content_path1)

        self.restore_dest_path = self.client_machine.join_path(client_drive, 'Automation', str(self.id), 'Restoredata')
        if self.client_machine.check_directory_exists(self.restore_dest_path):
            self.client_machine.remove_directory(self.restore_dest_path)
        self.client_machine.create_directory(self.restore_dest_path)
        self.dedupehelper = DedupeHelper(self)
        self.mmhelper = MMHelper(self)
        self.common_util = CommonUtils(self)
        # self.workflow = Workflow(self)

    def run(self):
        """Run function of this test case"""
        try:
            # Creating Cloud Library
            cloud_lib_obj = self.mmhelper.configure_cloud_library(self.cloud_library_name,
                                                                  self.tcinputs['MediaAgentName'],
                                                                  self.tcinputs["CloudMountPath"],
                                                                  self.tcinputs["CloudUserName"],
                                                                  self.tcinputs["CloudPassword"],
                                                                  self.tcinputs["CloudServerType"])
            # Adding second MP on cloud lib
            self.mmhelper.configure_cloud_mount_path(cloud_lib_obj, self.tcinputs["CloudMountPath"],
                                                     self.tcinputs['MediaAgentName'], self.tcinputs["CloudUserName"],
                                                     self.tcinputs["CloudPassword"], self.tcinputs["CloudServerType"])
            # Configuring de-dupe storage policy
            sp_dedup_obj = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_dedupe_name,
                                                                             self.cloud_library_name,
                                                                             self.tcinputs['MediaAgentName'],
                                                                             self.partition_path)
            # Set retention of 0-day & 1-cycle on configured storage policy copy
            self.log.info("Setting Retention: 0-days and 1-cycle on Primary Copy")
            sp_dedup_primary_obj = sp_dedup_obj.get_copy("Primary")
            retention = (0, 1, -1)
            sp_dedup_primary_obj.copy_retention = retention

            # create backup-set
            self.mmhelper.configure_backupset(self.backupset_name, self.agent)
            # create sub-client
            subclient1_name = "%s_SC1" % str(self.id)
            sc1_obj = self.mmhelper.configure_subclient(self.backupset_name, subclient1_name,
                                                        self.storage_policy_dedupe_name, self.content_path1, self.agent)
            # Allow multiple data readers to subclient
            self.log.info("Setting Data Readers=4 on Subclient")
            self.set_multiple_readers(sc1_obj, 4)

            self.log.info("Generating Data at %s", self.content_path1)
            if not self.client_machine.generate_test_data(self.content_path1, dirs=1, file_size=(2 * 1024),
                                                          files=10):
                self.log.error("unable to Generate Data at %s", self.content_path1)
                raise Exception("unable to Generate Data at {0}".format(self.content_path1))
            self.log.info("Generated Data at %s", self.content_path1)

            self.common_util.subclient_backup(sc1_obj, 'full')

            # Restore from cloud
            restore_job = sc1_obj.restore_out_of_place(self.client, self.restore_dest_path,
                                                       [self.content_path1])
            self.log.info("restore job [%s] from cloud has started.", restore_job.job_id)
            # sleep for 20 seconds
            self.log.info("Sleeping for 20 seconds.")
            time.sleep(20)
            # Find running Workflow JOb id
            wf_job_id_list = self._get_wfjobid(int(restore_job.job_id))
            # Finding WF job id whose input jobid is same as restore job started above.
            for job_id in wf_job_id_list:
                if self._get_restorejobid_bywfjobid(job_id) == int(restore_job.job_id):
                    wf_jobid = job_id
                    break
            else:
                self.log.error("Failed to get workflow id")
                raise Exception("Failed to get workflow id")

            self.log.info("Workflow Job ID : %s", wf_jobid)
            # Find input job id of workflow Job.
            restore_input_jobid = self._get_restorejobid_bywfjobid(wf_jobid)
            self.log.info("Input Job ID of the workflow is : %s", restore_input_jobid)
            # Adding sleep so that WF job can process for sometime, before we check it's phase/status.
            self.log.info("Sleeping for 600 seconds, to allow WF job to run for sometime before checking on it.")
            time.sleep(600)

            # Getting phase of the running Cloud archive recall workflow job.
            wf_job_obj = Job(self.commcell, wf_jobid)
            self.log.info("Status of workflow job object is: %s", wf_job_obj)
            wf_phase = wf_job_obj.phase
            if wf_phase is not None:
                wf_jobphase = wf_phase.lower()
            else:
                wf_jobphase = wf_phase
            self.log.info("Phase of workflow job is: %s", wf_jobphase)
            # Get status of WF job, if it is not running any more.
            wf_status = wf_job_obj.status
            if wf_status is not None:
                wf_jobstatus = wf_status.lower()
            else:
                wf_jobstatus = wf_status
            self.log.info("Status of workflow job is: %s", wf_jobstatus)

            # Creating case where we check status and take next action accordingly.
            for index in range(0, 24):
                if (wf_jobphase in ('waiting for restore job completion', 'recalling files')
                        and wf_jobstatus == 'running'):
                    # if status is waiting for restore completion or still recalling files, re-check after 1 hour.
                    self.log.info("Sleeping for 3600 seconds before re-checking WF job phase.")
                    time.sleep(3600)
                    # Getting phase of the running Cloud archive recall workflow job.
                    wf_job_obj = Job(self.commcell, wf_jobid)
                    wf_phase = wf_job_obj.phase
                    if wf_phase is not None:
                        wf_jobphase = wf_phase.lower()
                        self.log.info("Phase of workflow job is: %s", wf_jobphase)
                    else:
                        wf_jobphase = wf_phase
                        self.log.info("Phase of workflow job is: %s", wf_jobphase)
                    wf_status = wf_job_obj.status
                    if wf_status is not None:
                        wf_jobstatus = wf_status.lower()
                        self.log.info("Status of workflow job is: %s", wf_jobstatus)
                    else:
                        wf_jobstatus = wf_status
                        self.log.info("Status of workflow job is: %s", wf_jobstatus)
                else:
                    break
            else:
                if not (wf_jobphase in ('waiting for restore job completion', 'recalling files')
                        and wf_jobstatus == 'running'):
                    self.log.error("Workflow is still in recalling files phase")
                    raise Exception("Workflow is still in recalling files phase")

            if wf_jobstatus in ('completed with error', 'completed', 'killed', 'failed'):
                # if WF job has completed with any of the above status
                self.log.info("Workflow job [%s] [%s] with delay reason %s.", wf_jobid, wf_jobstatus,
                              wf_job_obj.delay_reason)
                if not restore_job.wait_for_completion():
                    self.log.error("restore job [%s] has failed with %s.", restore_job.job_id,
                                   restore_job.delay_reason)
                    raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                    restore_job.delay_reason))

            else:
                self.log.info("Workflow job [%s] [%s] with delay reason %s.", wf_jobid, wf_jobstatus,
                              wf_job_obj.delay_reason)
                if not restore_job.wait_for_completion(60):
                    self.log.error("restore job [%s] has failed with %s.", restore_job.job_id,
                                   restore_job.delay_reason)
                    raise Exception("restore job [{0}] has failed with {1}.".format(restore_job.job_id,
                                                                                    restore_job.delay_reason))
            self.log.info("restore job from cloud [%s] has completed.", restore_job.job_id)

        except Exception as exp:
            self.log.error('Failed to execute test case with error:%s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear Down function of this test case"""
        self._cleanup()
