# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                                    --  initialize TestCase class
    init_tc()                                     --  Initialize pre-requisites
    run_cloud_troubleshooting_send_log()          --  method for run cloud troubleshoot request
    is_index_file_exists()                        --  method that verifies the index file in titan
    verify_email_filter()
    run()                                         --  run function of this test case
Input Example:

    "testCases":
            {
                "indexupload":
                        {
                           "EmailId": "your_email_id@commvault.com",
                           "SendLogsClientName1": "client_test1",
                           "SendLogsClientName2": "client_test2",
                           "LogsPath": None,
                        }
            }


"""
import time
from Web.Common.page_object import handle_testcase_exception
from Web.Common.page_object import TestStep
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Troubleshooting.troubleshoot import CloudSendLog
from Web.API.customreports import CustomReportsAPI
from Reports.utils import TestCaseUtils
from Reports.metricsutils import MetricsServer
from Reports.SendLog.utils import SendLogUtils
from AutomationUtils.config import get_config
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from cvpysdk.commcell import Commcell
from cvpysdk.job import JobController


class TestCase(CVTestCase):
    """ Sendlogs with Index upload """
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

        """
        super(TestCase, self).__init__()
        self.name = "Sendlogs with Index upload"
        self.browser = None
        self.web_console = None
        self.metrics_server = None
        self.request_id = None
        self.metrics_server = None
        self.cloud_send_log = None
        self.tcinputs = {
            "EmailId": None,
            "SendLogsClientName1": None,
            "LogsPath": None
        }
        self.sendlog_client_name1 = None
        self.sendlog_client_name2 = None
        self.backupset_guid1 = None
        self.backupset_guid2 = None
        self.send_log_utils = None
        self.machine = None
        self.job_id = None
        self.logs_path = None
        self.commcell_id_hex = None
        self.email_id = None
        self.download_directory = None
        self.commcell_name = None
        self.commcell_user_name = None
        self.commcell_password = None
        self.commcell_object = None
        self.utils = TestCaseUtils(self)
        self.cre_api = None
        self.job_controller = None

    def init_tc(self):
        """
            Initial configuration for the test case
            """
        try:
            self.log.info("Initializing pre-requisites")
            _STORE_CONFIG = get_config()
            username = _STORE_CONFIG.Cloud.username
            password = _STORE_CONFIG.Cloud.password
            cloud_url = _STORE_CONFIG.Cloud.host_name
            self.machine = Machine(self.commcell.commserv_client)
            self.sendlog_client_name1 = self.tcinputs["SendLogsClientName1"]
            self.sendlog_client_name2 = self.tcinputs["SendLogsClientName2"]
            self.commcell_name = self.tcinputs["CommCellName"]
            self.logs_path = self.tcinputs["LogsPath"]
            self.email_id = self.tcinputs["EmailId"]
            self.commcell_user_name = self.tcinputs["CommCellUserName"]
            self.commcell_password = self.tcinputs["CommCellPassword"]
            self.commcell_id_hex = self.tcinputs["CommCellId"]
            self.commcell_object = Commcell(self.commcell_name, self.commcell_user_name,
                                            self.commcell_password)
            self.job_controller = JobController(self.commcell_object)
            if not username or not password:
                self.log.info("Cloud username and password are not configured in config.json")
                raise Exception("Cloud username and password are not configured. Please update "
                                "the username and password details under "
                                "<Automation_Path>/CoreUtils/Templates/template-config.json")
            self.metrics_server = MetricsServer(cloud_url, username, password)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.web_console = WebConsole(self.browser, cloud_url)
            self.web_console.login(username, password)
            self.web_console.goto_troubleshooting()
            self.send_log_utils = SendLogUtils(self, self.machine)
            self.download_directory = self.send_log_utils.create_directory_for_given_path("TC58674")

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def init_backupset_guids(self):
        """
        read backup set guid from the database
        """
        self.webconsole = WebConsole(
            self.browser, self.commcell_name, self.commcell_user_name, self.commcell_password)
        self.webconsole.login()
        self.cre_api = CustomReportsAPI(self.commcell_name)
        query = r"select distinct BS.GUID from APP_Client C  " \
                "inner join app_application A " \
                "on C.id = A.clientid " \
                "and A.subclientStatus & 8 =0 " \
                "inner join APP_BackupSetName BS " \
                "on A.backupSet = BS.id " \
                " where C.name = '" + self.sendlog_client_name1 + "'"

        response = self.cre_api.execute_sql(query)
        if not response:
            raise CVTestStepFailure(
                "Retreiving Backupset set guid from Database failed"
            )
        self.backupset_guid1 = response[0][0]

        query = r"select distinct BS.GUID from APP_Client C  " \
                "inner join app_application A " \
                "on C.id = A.clientid " \
                "and A.subclientStatus & 8 =0 " \
                "inner join APP_BackupSetName BS " \
                "on A.backupSet = BS.id " \
                " where C.name = '" + self.sendlog_client_name2 + "'"

        response = self.cre_api.execute_sql(query)
        if not response:
            raise CVTestStepFailure(
                "Retreiving Backupset set guid from Database failed"
            )
        self.backupset_guid2 = response[0][0]

    @test_step
    def run_cloud_troubleshooting_send_log(self):
        """
                Method to run cloud troubleshooting request for send logs send_log
        """
        self.cloud_send_log = CloudSendLog(self.web_console)
        self.cloud_send_log.access_commcell(self.commcell_name)
        self.cloud_send_log.close_popup()
        self.cloud_send_log.select_computer_information([self.sendlog_client_name1,
                                                         self.sendlog_client_name2])
        self.cloud_send_log.select_email_notification(True, [self.email_id])
        self.cloud_send_log.select_index()
        self.cloud_send_log.submit()
        if self.cloud_send_log.is_request_submit_success() is False:
            raise CVTestStepFailure("Cloud troubleshooting request for send log failed ")
        self.log.info('Send Log request submitted from cloud')

    @test_step
    def get_request_id(self):
        """get request id"""
        self.log.info('Wait for 2 minutes for xml creation in script folder')
        time.sleep(60)
        self.request_id = self.cloud_send_log.get_request_id()

    def run(self):
        try:
            self.init_tc()
            self.run_cloud_troubleshooting_send_log()
            self.get_request_id()
            WebConsole.logout_silently(self.web_console)
            self.init_backupset_guids()
            all_jobs_dict = self.job_controller.active_jobs()
            jobs = []
            for job_id, props in all_jobs_dict.items():
                if props['job_type'] == 'Send Log Files':
                    jobs.append(job_id)
            last_job = max(jobs)
            job_obj = self.job_controller.get(last_job)
            self.log.info('Waiting for send logs job ' + str(last_job) + ' completion')
            job_obj.wait_for_completion()
            retries = 1
            while retries <= 4:
                self.log.info(f'Waiting for 15 minutes to check completion email'
                              f' for send logs request'
                              f' {self.request_id}')
                time.sleep(900)
                subject = "Gather Log Request for [ RegAutomation ] - CommCell ID: [" +\
                           self.commcell_id_hex + "] [Request# " \
                           + str(self.request_id) + "] completed"
                retries = retries + 1
                try:
                    self.send_log_utils.verify_email(self.download_directory, subject)
                    break
                except Exception as exp:
                    if retries == 4:
                        raise
                    else:
                        pass
            self.send_log_utils.is_index_file_exists(self.logs_path, self.request_id,
                                                     self.sendlog_client_name1,
                                                     self.backupset_guid1)
            self.send_log_utils.is_index_file_exists(self.logs_path, self.request_id,
                                                     self.sendlog_client_name2,
                                                     self.backupset_guid2)
        except Exception as exp:
            handle_testcase_exception(self, exp)
        finally:
            WebConsole.logout_silently(self.web_console)
            Browser.close_silently(self.browser)
