# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test case to verify editing of replication group

Sample input:
"58673": {
            "ClientName": "idcvc61",
            "vm_names_to_backup": ["backup1", "backup2"],
            "recovery_target":"test-3",
            "storage_name" : "hyd_storage1",
            "source_vm_network": "VM Network",
            "destination_network":"Lab_10045"
       }
"""
from time import sleep

from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.virtualization_replication import _Target, SOURCE_HYPERVISOR_VMWARE
from Web.AdminConsole.VSAPages.vm_groups import VMGroups
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """Test case to verify editing of replication group"""
    test_step = TestStep()

    _TRANSPORT_MODE = 'NAS'  # Transport more is edited to this string
    _SNAPSHOTS = 1  # snapshots are edited to this number
    # replication name will be renamed as below
    _REPLICATION_GROUP_NAME_EDITED = 'Auto_edited_group_58673'
    _TARGET_FREQ_NUMBER = 5
    _TARGET_FREQUENCY_PERIOD = _Target.FREQUENCY_HOURS  # "Hours"
    _FREQUENCY = str(6) + " " + _TARGET_FREQUENCY_PERIOD.lower()  # frequency is edited to this
    _VENDOR_NAME = "VMware"

    def setup(self):
        """Initialize required veriables/inputs"""
        self.source_hypervisor = self.tcinputs["ClientName"]
        self.vm_names_to_backup = self.tcinputs["vm_names_to_backup"]
        self.recovery_target = self.tcinputs["recovery_target"]
        self.storage_name = self.tcinputs["storage_name"]
        self.vitualization_group_name = "Auto_vitualization_group_58673"

        self.source_vm_network = self.tcinputs["source_vm_network"]
        self.destination_network = self.tcinputs["destination_network"]

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Vmware Replication Group: Verify Create and edit configuration"
        self.browser = None
        self.utils = None
        self.admin_console = None
        self.navigator = None
        self.replication_details = None
        self.vm_names_to_backup = None
        self.source_vm_network = None
        self.destination_network = None
        self.source_hypervisor = None
        self.recovery_target = None
        self.storage_name = None
        self.vitualization_group_name = None
        self.replication_groups = None
        self.vm_groups = None
        self.utils = TestCaseUtils(self)

    def init_commandcenter(self):
        """Initialize browser and redirect to page"""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.replication_groups = ReplicationGroup(self.admin_console)
            self.replication_details = ReplicationDetails(self.admin_console)
            self.vm_groups = VMGroups(self.admin_console)
            self.navigator.navigate_to_replication_groups()
        except Exception as _exception:
            raise CVTestCaseInitFailure(_exception) from _exception

    @test_step
    def verify_replication_group_status(self):
        """Verify replication group status(Enable/Disable)"""
        # Disable replication group, verify its disabled status, and verify commcell schedule
        # is disabled, enable the replication group, verify summary status is enabled,
        # and schedule is enabled back
        self.navigator.navigate_to_replication_groups()
        self.replication_groups.access_group(self.vitualization_group_name)

        self.log.info("disable replication group [%s]" % self.vitualization_group_name)

        # disable replication group
        self.replication_details.disable_replication_group()

        # verify disabled
        summary = self.replication_details.overview.get_summary_details()
        if summary['State'] != 'Disabled':
            raise CVTestStepFailure("[%s] replication should have been disabled, but current "
                                    "state is [%s]" % (self.vitualization_group_name,
                                                       summary['State']))
        # verify schedule disabled in commcell
        schedule_name = self.replication_groups.get_schedule_name_by_replication_group(
            self.vitualization_group_name)
        #
        self.commcell.schedules.refresh()
        schedule_obj = self.commcell.schedules.get(schedule_name)
        if not schedule_obj.is_disabled:
            raise CVTestStepFailure("Schedule [%s] is expected to be disabled by disabling "
                                    "replication group [%s]" % (schedule_name,
                                                                self.vitualization_group_name))

        self.log.info("Enable replication group [%s]" % self.vitualization_group_name)
        self.replication_details.enable_replication_group()
        summary = self.replication_details.overview.get_summary_details()
        if summary['State'] != 'Enabled':
            raise CVTestStepFailure("ReplicationGroup [%s] should be enabled, but current state "
                                    "is [%s]" % (self.vitualization_group_name, summary['State']))
        self.log.info("Verified replication group state(Enabled/Disabled)")

        # verify schedule enabled in commcell
        schedule_obj.refresh()
        if schedule_obj.is_disabled:
            raise CVTestStepFailure("Schedule [%s] is expected to be enabled by enabling "
                                    "replication group [%s]" % (schedule_name,
                                                                self.vitualization_group_name))

    @test_step
    def add_delete_vm_to_group(self):
        """Add VM to group from configuration tab after creation, verify details and then delete it"""
        self.replication_details.access_configuration_tab()
        add_vm = self.replication_details.configuration.add_virtual_machines(vm_type=SOURCE_HYPERVISOR_VMWARE)
        add_vm.add_vm([self.vm_names_to_backup[1]])

        sleep(10)
        self.admin_console.refresh_page()

        new_vm_details = self.replication_details.configuration.get_vm_details(self.vm_names_to_backup[1])
        if new_vm_details['Source VM'] != self.vm_names_to_backup[1]:
            raise CVTestStepFailure(f"Expected value of Source VM {self.vm_names_to_backup[1]} does not match"
                                    f"the collected value {new_vm_details['Source VM']}")
        self.replication_details.configuration.remove_virtual_machines(self.vm_names_to_backup[1])
        try:
            self.admin_console.refresh_page()
            self.replication_details.configuration.get_vm_details(self.vm_names_to_backup[1])
        except:
            self.log.info("VM successfully added and deleted from group")
        else:
            raise CVTestStepFailure(f"Vm {self.vm_names_to_backup[1]} is not deleted from "
                                    f"group {self.vitualization_group_name}")

    @test_step
    def verify_edit_replication_group(self):
        """Verify editing of replication group"""
        self.log.info("Update Transport mode to [%s]" % self._TRANSPORT_MODE)
        self.replication_details.configuration.edit_transport_mode(self._TRANSPORT_MODE)

        self.log.info("Update snapshots to retain on destination vm to [%s]" %
                      self._SNAPSHOTS)
        self.replication_details.configuration.edit_snapshots(self._SNAPSHOTS)

        self.log.info("Update replication frequency to [%s]" % self._FREQUENCY)
        self.replication_details.configuration.edit_replication_frequency(self._FREQUENCY)

        self.log.info("Update replication group [%s] to [%s]" %
                      (self.vitualization_group_name, self._REPLICATION_GROUP_NAME_EDITED))
        self.replication_details.edit_replication_group_name(self._REPLICATION_GROUP_NAME_EDITED)

        # refresh the page
        self.admin_console.refresh_page()

        # verify fields are updated correctly
        self.log.info("Verifying Transport Mode and snapshots updated")
        advanced_options = self.replication_details.configuration.get_advanced_options_details()
        if advanced_options['Transport Mode'] != self._TRANSPORT_MODE:
            raise CVTestStepFailure("Expected [%s] Transport Mode, but found [%s]" %
                                    (self._TRANSPORT_MODE, advanced_options['Transport Mode']))
        if int(advanced_options['Snapshots to retain on destination VM']) != self._SNAPSHOTS:
            raise CVTestStepFailure("Expected Snapshots to retain on destination VM [%s], "
                                    "but found [%s]" %
                                    (
                                        self._SNAPSHOTS, advanced_options['Snapshots to '
                                                                          'retain on destination '
                                                                          'VM']
                                    )
                                    )

        # verify replication frequency
        self.log.info("Verifying replication frequency is updated")
        rpo_details = self.replication_details.configuration.get_rpo_details()
        if self._FREQUENCY not in rpo_details['Replication frequency']:
            raise CVTestStepFailure("Expected [%s] replication frequency, but found [%s]" %
                                    (self._FREQUENCY, rpo_details['Replication frequency']))

        self.log.info("verify replication group name is updated")
        replication_name = self.replication_details.get_replication_group_name()
        if replication_name != self._REPLICATION_GROUP_NAME_EDITED:
            raise CVTestStepFailure("[%s] is expected updated replication group name, but found ["
                                    "%s]" % (self._REPLICATION_GROUP_NAME_EDITED,
                                             replication_name))

    def verify_vm_settings(self):
        """Verify vm settings: General settings should not be editable, network settings should be
         editable"""
        self.log.info("verify the vm general settings are not editable")
        vm_settings = self.replication_details.configuration.edit_virtual_machines(self.vm_names_to_backup[0],
                                                                                   vm_type=SOURCE_HYPERVISOR_VMWARE)
        if vm_settings.is_general_settings_editable():
            raise CVTestStepFailure("General settings in replication group [%s] is editable")

        # edit network settings and verify settings are saved correctly
        vm_settings.edit_network_settings(self.source_vm_network, self.destination_network)
        vm_settings.save()

        self.admin_console.refresh_page()
        # verify network settings are updated
        vm_settings = self.replication_details.configuration.edit_virtual_machines(self.vm_names_to_backup[0],
                                                                                   vm_type=SOURCE_HYPERVISOR_VMWARE)
        _source, _destination = vm_settings.get_network_settings()
        if _source != self.source_vm_network and _destination != self.destination_network:
            raise CVTestStepFailure("Source/Destination network settings are not updated. "
                                    "Expected source network [%s] and destination network [%s], "
                                    "but found source network [%s] and destination network [%s]"
                                    % (self.source_vm_network, self.destination_network,
                                       _source, _destination)
                                    )
        vm_settings.save()

    @test_step
    def configure_vitualization_group(self):
        """Create a vitualization replication group"""
        try:
            self.navigator.navigate_to_replication_groups()
            vitualization_grp = self.replication_groups.configure_vmware()
            vitualization_grp.add_default_group(self.vitualization_group_name,
                                                self.source_hypervisor,
                                                [self.vm_names_to_backup[0]],
                                                self.recovery_target,
                                                self.storage_name)

            self.log.info('Replication group created successfully')
        except Exception as err:
            raise CVTestStepFailure(err)

    @test_step
    def verify_vm_group(self):
        """Verify that the VM group exists for the associated replication group"""
        sleep(60)
        self.admin_console.navigator.navigate_to_vm_groups()
        self.admin_console.refresh_page()
        table_content = self.vm_groups.get_details_by_vm_group(self.vitualization_group_name)
        if len(table_content['Name']) > 1:
            raise CVTestStepFailure(
                "The get details returned multiple rows. This is most likely due to search fail")
        self.utils.assert_includes(self.vitualization_group_name, table_content['Name'][0])
        self.utils.assert_comparison(table_content['Vendor'][0], self._VENDOR_NAME)
        self.utils.assert_comparison(table_content['Hypervisor'][0], self.source_hypervisor)
        self.utils.assert_comparison(
            table_content['Plan'][0], f"{self.vitualization_group_name}_ReplicationPlan"
        )

    @test_step
    def delete_vitualization_group(self):
        """Delete replication group"""
        self.navigator.navigate_to_replication_groups()
        if self.replication_groups.has_group(self._REPLICATION_GROUP_NAME_EDITED):
            self.replication_groups.delete_group(self._REPLICATION_GROUP_NAME_EDITED)
            self.log.info("Replication[%s] group deleted!", self._REPLICATION_GROUP_NAME_EDITED)
            return True
        self.log.info("Replication group [%s] does not exist!",
                      self._REPLICATION_GROUP_NAME_EDITED)
        #  in case replication group is created and terminated before editing the name then
        #  replication group should be deleted before next run.
        if self.replication_groups.has_group(self.vitualization_group_name):
            self.replication_groups.delete_group(self.vitualization_group_name)
            self.log.info("Replication[%s] group deleted!", self.vitualization_group_name)
            return True
        self.log.info("Replication group [%s] does not exist!",
                      self.vitualization_group_name)
        return False

    def wait_for_sync(self):
        """Wait for sync up, during waiting period page should not timeout, so refresh the page
        every 2 minutes"""
        self.log.info("Wait for sync up for 5 mins")
        sleep(120)
        self.admin_console.refresh_page()
        sleep(120)
        self.admin_console.refresh_page()
        sleep(60)

    def run(self):
        """Starting test case steps"""
        try:
            self.init_commandcenter()
            if self.delete_vitualization_group():
                # if group exists and deleted, wait for schedules/plans to sync once deleted
                self.wait_for_sync()
            self.configure_vitualization_group()
            self.verify_vm_group()
            self.verify_replication_group_status()
            self.add_delete_vm_to_group()
            self.verify_edit_replication_group()
            self.verify_vm_settings()
            self.delete_vitualization_group()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
