# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""Main file for executing this test case

Test cases to validate schedule.

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:

    __init__()                      --  Initialize TestCase class

    init_tc()                       --  To perform initial configuration for the test case

    get_webconsole_table_data()     --  To get data from the table present in web page

    get_mail_table_data()           --  To get the data from the table embedded in the mail

    cleanup_schedule()              --  Deletes the schedules which contain 'Automation_tc_54786_'
                                        in schedule name

    verify_schedule_exists()        --  To verify is schedule is created successfully

    run_schedule()                  --  To run the created schedule

    validate_schedule_mail()        --  To validate the schedule mail received with
                                        the web page report

    modify_schedule()               --  Modifies schedule and update details

    validate_recipient_column()       -- To validate the Recipient User and User Group column values

    delete_schedule()               --  To delete the created schedule

    run()                           --  Main function for test case execution

"""

import time

from cvpysdk import schedules

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from Web.WebConsole.Reports.manage_schedules import ScheduleSettings
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.Reports.Custom._components.table import MailTable
from Reports.Custom.report_templates import DefaultReport
from Reports.utils import TestCaseUtils
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils import mail_box
from AutomationUtils import config
from AutomationUtils.cvtestcase import CVTestCase

CONSTANTS = config.get_config()


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = self.report_name = "Custom report: Users to Notify feature in schedule"
        self.browser = None
        self.webconsole = None
        self.navigator = None
        self.report = None
        self.schedule_name = OptionsSelector.get_custom_str('Automation_58660')
        self.format = "INLINE"
        self.user_name = CONSTANTS.email.username
        self.tcinputs = {
            "user_group": None
        }
        self.schedule_window = None
        self.mail = None
        self.mails_download_directory = None
        self.schedules = None
        self.schedule_settings = None
        self.table = None
        self.viewer = None
        self.utils = TestCaseUtils(self)


    def init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.mail = mail_box.MailBox()
            self.mail.connect()
            self.schedules = schedules.Schedules(self.commcell)
            if not self.user_name:
                raise CVTestCaseInitFailure("User name is not specified in config file")

            # open browser
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.open()

            # login to web console
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.webconsole.goto_commcell_dashboard()
            self.utils.webconsole = self.webconsole

            self.navigator = Navigator(self.webconsole)
            self.report = MetricsReport(self.webconsole)
            self.viewer = viewer.CustomReportViewer(self.webconsole)
            self.table = viewer.DataTable("Automation Table")
            self.schedule_settings = ScheduleSettings(self.webconsole)
            self.cleanup_schedules()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def get_webconsole_table_data(self):
        """Read the data present in table"""
        self.viewer.associate_component(self.table)
        data = self.table.get_table_data()
        self.log.info("Table data present in webpage:")
        self.log.info(data)
        return data

    def get_mail_table_data(self, table_id):
        """
        To get the mail table data
        table_id    (str)   -- Comp ID of the table
        """
        mail_table = MailTable(self.browser.driver, table_id)
        data = mail_table.get_table_data()
        self.log.info("Table data present in Mail:")
        self.log.info(data)
        return data

    def cleanup_schedules(self):
        """ Deletes the schedules which contain 'Automation_tc_58660_' in schedule name """
        self.navigator.goto_schedules_configuration()
        self.schedule_settings.cleanup_schedules("Automation_58660")

    @test_step
    def create_schedule(self):
        """To create a new schedule"""
        schedule_window = self.report.open_schedule()
        self.log.info("Creating schedule for the [%s] report with [%s] file format",
                      self.report_name, self.format)
        schedule_window.set_schedule_name(self.schedule_name)
        schedule_window.set_users_to_notify(self.user_name)
        schedule_window.select_format(self.format)
        schedule_window.clear_recipient()
        schedule_window.save()
        time.sleep(5)
        self.log.info("Schedule created successfully for the report [%s]", self.report_name)

    @test_step
    def modify_schedule(self):
        """To edit the schedule"""
        self.navigator.goto_schedules_configuration()
        edit_schedule_window = self.schedule_settings.edit_schedule(self.schedule_name)
        edit_schedule_window.set_users_to_notify(self.tcinputs["user_group"])
        edit_schedule_window.save()
        time.sleep(5)
        self.log.info("Schedule modified successfully for the report [%s]", self.report_name)

    @test_step
    def verify_schedule_exists(self):
        """To verify whether schedule is created"""
        self.log.info("Checking [%s] schedule is created", self.schedule_name)
        self.schedules.refresh()
        if not self.schedules.has_schedule(self.schedule_name):
            err_str = "[%s] schedule does not exists in db, created on [%s] report with [%s]" \
                      " file extension" % (self.schedule_name, self.report_name, self.format)
            raise CVTestStepFailure(err_str)
        self.log.info("[%s] schedule is created successfully", self.schedule_name)

    @test_step
    def run_schedule(self):
        """To run the schedule"""
        schedule = schedules.Schedule(self.commcell, schedule_name=self.schedule_name)
        self.log.info("Running [%s] schedule", self.schedule_name)
        job_id = schedule.run_now()
        time.sleep(5)

        job = self.commcell.job_controller.get(job_id)
        self.log.info("Wait for [%s] job to complete", str(job))
        if job.wait_for_completion():
            self.log.info("Schedule job completed with job id:[%s], for the report:[%s], "
                          "with file format:[%s]",
                          job_id,
                          self.report_name,
                          self.format)
        else:
            err_str = "Schedule job failed with job id [%s], for the report name [%s],file " \
                      "format [%s]" % (job_id, self.report_name, self.format)
            raise CVTestStepFailure(err_str)

    @test_step
    def validate_schedule_mail(self):
        """Validate schedule mails"""
        self.utils.reset_temp_dir()
        self.log.info("verifying [%s] schedule email for [%s] report with [%s] file extension",
                      self.schedule_name, self.report_name, self.format)
        self.utils.download_mail(self.mail, self.schedule_name)

        web_report_table_data = self.get_webconsole_table_data()

        # To navigate to the downloaded mail
        file_path = self.utils.poll_for_tmp_files(ends_with="html")[0]
        self.browser.open_new_tab()
        self.browser.switch_to_latest_tab()
        self.browser.driver.get(file_path)
        time.sleep(3)

        mail_report_table_data = self.get_mail_table_data(self.table.id)

        self.browser.close_current_tab()
        self.browser.switch_to_latest_tab()

        if web_report_table_data != mail_report_table_data:
            self.log.error("Mail table contents are not matching with report table content")
            self.log.error("Mail content:%s", mail_report_table_data)
            self.log.error("Web report content:%s", web_report_table_data)
            raise CVTestStepFailure("Mail table contents are not matching with report table")
        self.log.info("Mail contents are verified successfully")

    @test_step
    def validate_recipient_column(self):
        """Validate that User and User group is displayed in Recipient Users and Groups column"""
        recipient_users = self.schedule_settings.get_recipients_user(self.schedule_name)
        recipient_list = recipient_users.replace('Users : ', '').replace('User Groups : ', '').split("\n")
        if [self.user_name, self.tcinputs["user_group"]] == recipient_list:
            self.log.info("User and User Group are displayed correctly in "
                          "Recipient Users and Groups column as [%s]", [self.user_name, self.tcinputs["user_group"]])
        else:
            self.log.info("Expected User in Recipient Users and Groups column - %s",
                          [self.user_name, self.tcinputs["user_group"]])
            self.log.info("Actual User in Recipient Users and Groups column - %s",
                          recipient_list)
            raise CVTestStepFailure("Expected User and User group are NOT displayed in "
                                    "Recipient Users and Groups column")

    @test_step
    def delete_schedule(self):
        """Delete schedules"""
        self.navigator.goto_schedules_configuration()
        self.schedule_settings.delete_schedules([self.schedule_name])
        self.log.info("Schedule [%s] deleted successfully", self.schedule_name)

    def run(self):
        """Main function for test case execution"""
        try:
            self.init_tc()

            # To create a new report
            DefaultReport(self.utils).build_default_report(
                sql="""
                         SELECT TOP 5 id, name, csHostName
                         FROM App_Client
                         """,
                overwrite=False
            )

            self.create_schedule()
            self.verify_schedule_exists()
            self.run_schedule()
            self.validate_schedule_mail()
            self.modify_schedule()
            self.validate_recipient_column()
            self.delete_schedule()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.mail.disconnect()
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
