# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: This test case verifies whether whether VMware replications honors the VM folder, resource pool and the
source and destination network at the target

TestCase: Class for executing this test case
Sample JSON: {
        "ClientName": "Azure",
        "source_vm": "vm",
        "recovery_target": "AzureTarget",
        "storage_name" : "storage",
        "resource_pool": None,
        "vm_folder": None,
        "source_network": None,
        "destination_network": None,
        ""
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from VirtualServer.VSAUtils.VMHelpers.VmwareVM import VmwareVM
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the VMware replication"""
    test_step = TestStep()
    _AGENT_NAME = 'virtual server'
    _INSTANCE_NAME = 'vmware'
    _BACKUPSET_NAME = 'defaultBackupSet'

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Command Center: Live Sync: Honor VM folder, resource pool and N/W selected at the target."

        self.tcinputs = {
            "ClientName": None,
            "source_vm": None,
            "recovery_target": None,
            "storage_name": None,
            "resource_pool": None,
            "vm_folder": None,
            "source_network": None,
            "destination_network": None,
        }
        self.utils = None

        self.client = None
        self.agent = None
        self.instance = None
        self.backupset = None
        self.subclient = None

        self.browser = None
        self.admin_console = None

        self.replication_group = None

        self._source_subclient = None
        self._vm = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    @property
    def schedule(self):
        """Returns the name of the schedule"""
        return self.replication_group.get_schedule_name_by_replication_group(self.group_name)

    @property
    def source_subclient(self):
        """Returns the VSA Auto subclient after initialising all the testcase objects"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        if not self._source_subclient:
            self._source_subclient = VirtualServerUtils.subclient_initialize(self)
        return self._source_subclient

    @property
    def destination_vm_name(self):
        """Gets the destination VM name from the source subclient"""
        return self.source_subclient.get_live_sync_destination_vms(self.schedule)[0]

    @property
    def vm(self):
        """Returns the vmware vm after fetching it from the hypervisor object"""
        if not self._vm:
            self._vm = VmwareVM(self.source_subclient.hvobj, self.destination_vm_name)
        return self._vm

    @property
    def backup_job_id(self):
        """Get the latest backup job obj"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        return self.subclient.find_latest_job().job_id

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.goto_adminconsole()
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.admin_console.wait_for_completion()

        self.replication_group = ReplicationGroup(self.admin_console)

    def logout(self):
        """Logs out of the admin console and closes the browser"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)

            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The repication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        vmware_configure = self.replication_group.configure_vmware()
        vmware_configure.content.set_name(self.group_name)
        vmware_configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                            {"VMs and templates": [self.tcinputs['source_vm']]})
        sleep(5)
        vmware_configure.next()

        vmware_configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        vmware_configure.target.unconditionally_overwrite_vm(True)
        vmware_configure.next()

        vmware_configure.storage_cache.select_storage(self.tcinputs['storage_name'])

        vmware_configure.next()

        override_vm = vmware_configure.override_options.override_vms(self.tcinputs['source_vm'])
        override_vm.select_resource_pool(self.tcinputs['resource_pool'])
        override_vm.set_vm_folder(self.tcinputs['vm_folder'])

        network_settings = override_vm.edit_network(0)
        network_settings.select_source_network(self.tcinputs['source_network'])
        network_settings.select_destination_network(self.tcinputs['destination_network'])
        network_settings.save()

        override_vm.save()
        vmware_configure.next()

        sleep(5)
        vmware_configure.finish()

    @test_step
    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        self.log.info('Waiting for backup job %s to finish', self.backup_job_id)
        job_obj = self.commcell.job_controller.get(self.backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_job_completion(self):
        """Waits for replication job to complete and then verify its details"""
        self.log.info("Waiting 2 minutes to let live sync update")
        sleep(120)
        live_sync_utils = LiveSyncUtils(self.source_subclient, self.schedule)
        job_obj = live_sync_utils.get_recent_replication_job(self.backup_job_id)
        self.log.info("Waiting for replication job id %s to finish", job_obj.job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_override_settings(self):
        """Verify the overridden settings for the destination VM"""
        self.utils.assert_comparison(self.vm.vm_obj.parent.name, self.tcinputs['vm_folder'])
        # The resource pool input to the admin console is prefixed by a '/'. However, the folder name received from
        # the hypervisor does not contain this
        self.utils.assert_comparison(self.vm.vm_obj.resourcePool.name, self.tcinputs['resource_pool'].split('/')[-1])

        networks = [network.name for network in self.vm.vm_obj.network]
        self.utils.assert_includes(self.tcinputs['destination_network'], networks)

    def run(self):
        """Runs the testcase in order"""
        try:
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()

            self.logout()
            self.verify_backup_job_completion()
            self.verify_replication_job_completion()

            self.verify_override_settings()

            self.login()
            self.delete_virtualization_group(wait_for_sync=False)

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Tears down the TC"""
        self.logout()
