# coding=utf-8
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

    setup_environment() -- configures entities based on inputs

    remove_drillhole_flag() -- removes mount path with drill hole flag if set

    update_mmpruneprocess() -- updates MMPruneProcessInterval to 10 mins or reverts to noted time

    get_active_files_store() -- gets active files DDB store id

    cleanup()   --  cleanups all created entities

    run_backup()    -- runs backup need for the case

    run_data_aging()    -- runs data aging job for storage policy copy created by case

    run_dv2_job()   -- runs DV2 job with options provided

    run_space_reclaim_job()   -- runs space reclaim job with validation of validate and prune

    create_orphan_chunk()   -- created a dummy orphan chunk for store id

    validate_space_reclaim() -- validates the DDB space reclaim job

    wait_for_pruning()  -- waits for phase 2 and phase 3 pruning completion

    get_size()     -- this is a static method to written size for unix accordingly

Note:
    1. will be considering MP and DDB path if provided for configurations
    2. Make sure "Ransomware protection" is disabled on MA properties and service restart before running the case

Sample JSON: values under [] are optional
"58327": {
            "ClientName": "bdcmmlinux1",
            "AgentName": "File System",
            "MediaAgentName": "bdcmmlinux1",
            "SqlSaPassword": "builder!12",
            ["DDBPath": "/ddb/58327/ddb",
            "MountPath": "/data/58327/lib",
            "ContentPath": "/data/58327/content",
            "ScaleFactor": "5"]
        }

design:
    add disable drillhole key on MA
    add regkey to 0% fragment consideration
    create library with provided mp path or self search path
    disable flag from MP for drillhole

    add dedupe sp with provided DDB path or self search path
    disable garbage collection on dedupe store

    generate content considering scale factor ture or fals
    Run job with 100 files - J1
    Delete alternate files in content
    Run job with alternate 50 files - J2

    Delete J1
    Run aging and wait for physical pruning
    wait for phase 2 & 3 pruning to happen
    add a dummy orphan chunk

    run space reclaim with OCL enabled (level 4)
        verify orphan chunk is pruned
        verify 3 phases overall
        verify store is set with validate and prune flags

    generate content considering scale factor ture or fals
    Run job with 100 files - J1
    Delete alternate files in content
    Run job with alternate 50 files - J2

    Delete J1
    Run aging and wait for physical pruning
    wait for phase 2 & 3 pruning to happen
    add a dummy orphan chunk

    run space reclaim with ocl disabled (level 4)
        verify compacted chunks
        verify orphan dummy chunk is not pruned
        verify 2 phases overall
        verify store is set with validate and prune flags
"""
import time
from AutomationUtils import constants
from AutomationUtils import cvhelper
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import MMHelper
from MediaAgents.MAUtils.mahelper import DedupeHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Simplify DV2 project - Space reclaim job case"
        self.tcinputs = {
            "MediaAgentName": None,
        }
        self.library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.content_path = None
        self.mount_path = None
        self.ddb_path = None
        self.scale_factor = None
        self.mmhelper = None
        self.dedupehelper = None
        self.client_machine = None
        self.library = None
        self.storage_policy = None
        self.backupset = None
        self.subclient = None
        self.drillhole_key_added = True
        self.mmpruneprocess_value = None
        self.primary_copy = None
        self.media_agent_machine = None
        self.orphan_chunks_folder = None
        self.orphan_chunks_file = None
        self.sql_password = None

    def setup(self):
        """ Setup function of this test case. """
        self.content_path = self.tcinputs.get('ContentPath')
        self.mount_path = self.tcinputs.get('MountPath')
        self.ddb_path = self.tcinputs.get('DDBPath')
        self.scale_factor = self.tcinputs.get('ScaleFactor')
        self.subclient_name = str(self.id) + "_SC"
        self.library_name = str(self.id) + "_lib"
        self.storage_policy_name = str(self.id) + "_SP"
        self.backupset_name = str(self.id) + "_BS"
        self.client_machine = Machine(self.tcinputs['ClientName'], self.commcell)
        self.media_agent_machine = Machine(self.tcinputs['MediaAgentName'], self.commcell)
        self.mmhelper = MMHelper(self)
        self.dedupehelper = DedupeHelper(self)
        encrypted_pass = Machine(self.commcell.commserv_client).get_registry_value("Database", "pAccess")
        self.sql_password = cvhelper.format_string(self._commcell, encrypted_pass).split("_cv")[1]

    def setup_environment(self):
        """
        configures all entities based tcinputs. if path is provided TC will use this path instead of self selecting
        """
        self.log.warning("""*** Make sure "RANSOMWARE PROTECTION" is disabled on MA properties and service restart
        before running the case ***""")
        self.log.info("setting up environment...")

        # select drive on MA for MP and DDB
        op_selector = OptionsSelector(self.commcell)
        media_agent_drive = self.media_agent_machine.join_path(
            op_selector.get_drive(self.media_agent_machine), 'automation', self.id)
        if not self.mount_path:
            self.mount_path = self.media_agent_machine.join_path(media_agent_drive, 'mountpath')
        else:
            self.log.info("will be using user specified path [%s] for mount path configuration", self.mount_path)
        if not self.ddb_path:
            self.ddb_path = self.media_agent_machine.join_path(media_agent_drive, 'DDB')
        else:
            self.log.info("will be using user specified path [%s] for DDB path configuration", self.ddb_path)

        # select drive on client for content and restore
        client_drive = self.client_machine.join_path(
            op_selector.get_drive(self.client_machine), 'automation', self.id)
        if not self.content_path:
            self.content_path = self.client_machine.join_path(client_drive, 'content_path')
        else:
            self.log.info("will be using user specified path [%s] for backup content", self.content_path)

        if not self.media_agent_machine.check_registry_exists('MediaAgent', 'DedupDrillHoles'):
            self.media_agent_machine.create_registry('MediaAgent', value='DedupDrillHoles', data='0', reg_type='DWord')
            self.log.info("added regkey to disable drillholes!")
            self.drillhole_key_added = True
        else:
            self.log.info("drillhole regkey already exists")

        # uncomment below part of the code to make it a fix if fragmentation % is lower than 20
        # if not self.media_agent_machine.check_registry_exists('MediaAgent', 'AuxcopySfileFragPercent'):
        #     self.media_agent_machine.create_registry('MediaAgent', value='AuxcopySfileFragPercent',
        #                                              data='0', reg_type='DWord')
        #     self.log.info("adding sfile fragment percentage to 0!")

        self.library = self.mmhelper.configure_disk_library(self.library_name,
                                                            self.tcinputs["MediaAgentName"],
                                                            self.mount_path)
        self.remove_drillhole_flag(self.library.library_id)

        self.storage_policy = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_name,
                                                                                self.library.name,
                                                                                self.tcinputs["MediaAgentName"],
                                                                                self.ddb_path)

        self.log.info("setting primary copy retention to 1 day, 0 cycle")
        self.primary_copy = self.storage_policy.get_copy('Primary')
        self.primary_copy.copy_retention = (1, 0, 1)

        self.log.info("disabling garbage collection to avoid complications in waiting for physical prune")
        store = self.get_active_files_store()
        store.enable_garbage_collection = False

        self.mmhelper.configure_backupset(self.backupset_name, self._agent)

        self.subclient = self.mmhelper.configure_subclient(self.backupset_name,
                                                           self.subclient_name,
                                                           self.storage_policy_name,
                                                           self.content_path,
                                                           self._agent)
        self.update_mmpruneprocess()

    def remove_drillhole_flag(self, library_id, revert=False):
        """
        this method will remove drill hole flag for all mount paths of a library

        Args:
             library_id - library id which has all mount paths to disable drill hole
        """
        if not revert:
            self.log.info("removing drill hole flag at mount path level...")
            query = f"""
                    update MMMountPath
                    set Attribute = Attribute & ~128
                    where LibraryId = {library_id}"""
        else:
            self.log.info("reverting drill hole flag at mount path level...")
            query = f"""
                    update MMMountPath
                    set Attribute = Attribute | 128
                    where LibraryId = {library_id}"""
        self.log.info("QUERY: %s", query)
        self.mmhelper.execute_update_query(query, db_password=self.sql_password, db_user='sqladmin_cv')

    def update_mmpruneprocess(self):
        """reduced MMPruneProcess interval to 2mins and reverts back if already set"""
        if not self.mmpruneprocess_value:
            self.log.info("setting MMPruneProcessInterval value to 10")
            query = f"""update MMConfigs set value = 10 where name = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"""
        else:
            self.log.info("reverting MMPruneProcessInterval value to %s", self.mmpruneprocess_value)
            query = f"""update MMConfigs set value = {self.mmpruneprocess_value}
            where name = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"""
        self.log.info("QUERY: %s", query)
        self.mmhelper.execute_update_query(query, db_password=self.sql_password, db_user='sqladmin_cv')

    def get_active_files_store(self):
        """returns active store object for files iDA"""
        self.commcell.deduplication_engines.refresh()
        engine = self.commcell.deduplication_engines.get(self.storage_policy_name, 'primary')
        for store in engine.all_stores:
            if "_files_" in store[1].lower() and store[2] == 'active':
                return engine.get(store[0])
        return 0

    def cleanup(self):
        """
        performs cleanup of all entities
        """
        try:
            flag = 0
            self.log.info("cleanup started")
            self.remove_drillhole_flag(self.library.library_id, revert=True)
            if self._agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("deleting backupset...")
                self._agent.backupsets.delete(self.backupset_name)
                flag = 1
            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("deleting storage policy...")
                self.commcell.storage_policies.delete(self.storage_policy_name)
                flag = 1
            if self.commcell.disk_libraries.has_library(self.library_name):
                self.log.info("deleting library...")
                self.commcell.disk_libraries.delete(self.library_name)
                flag = 1
            additional_content = self.client_machine.join_path(self.content_path, 'generated_content')
            if self.client_machine.check_directory_exists(additional_content):
                self.log.info("deleting additional content...")
                self.client_machine.remove_directory(additional_content)
                flag = 1
            if not flag:
                self.log.info("no entities found to clean up!")
            else:
                self.log.info("cleanup done.")

        except Exception:
            self.log.warning("Something went wrong while cleanup!")

    def run_backup(self, backup_type="FULL", size=1.0, delete_alternative=False):
        """
        this function runs backup by generating new content to get unique blocks for dedupe backups.
        if scalefactor in tcinput, creates factor times of backup data

        Args:
            backup_type (str): type of backup to run
                Default - FULL

            size (int): size of backup content to generate
                Default - 1 GB

            delete_alternative (bool): to run a backup by deleting alternate content, set True
                Default - False

        Returns:
        (object) -- returns job object to backup job
        """
        additional_content = self.client_machine.join_path(self.content_path, 'generated_content')
        if not delete_alternative:
            # add content
            if self.client_machine.check_directory_exists(additional_content):
                self.client_machine.remove_directory(additional_content)
            # if scale test param is passed in input json, multiple size factor times and generate content
            if self.scale_factor:
                size = size * int(self.scale_factor)
            # calculate files
            files = (size * 1024 * 1024) / 10240
            self.client_machine.generate_test_data(additional_content, dirs=1, files=int(files), file_size=10240)
        else:
            files_list = self.client_machine.get_files_in_path(additional_content)
            self.log.info("deleting alternate content files...")
            for i, file in enumerate(files_list):
                if i & 2 == 0:
                    self.client_machine.delete_file(file)
        self.log.info("Running %s backup...", backup_type)
        job = self.subclient.backup(backup_type)
        self.log.info("Backup job: %s", job.job_id)
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup with error: {1}".format(backup_type, job.delay_reason)
            )
        self.log.info("Backup job completed.")
        return job

    def run_data_aging(self):
        """
        runs data aging job for a given storage policy, copy.
        """
        da_job = self.commcell.run_data_aging(copy_name='Primary',
                                              storage_policy_name=self.storage_policy_name,
                                              is_granular=True,
                                              include_all_clients=True)

        self.log.info("data aging job: %s", da_job.job_id)
        if not da_job.wait_for_completion():
            raise Exception(f"Failed to run data aging with error: {da_job.delay_reason}")
        self.log.info("Data aging job completed.")

    def run_dv2_job(self, store, dv2_type, option):
        """
        Runs DV2 job with type and option selected and waits for job to complete

        Args:
            store (object) - object of the store to run DV2 job on

            dv2_type (str) - specify type either full or incremental

            option (str) - specify option, either quick or complete

        Returns:
             (object) - completed DV2 job object
        """

        self.log.info("running [%s] [%s] DV2 job on store [%s]...", dv2_type, option, store.store_id)
        if dv2_type == 'incremental' and option == 'quick':
            job = store.run_ddb_verification()
        elif dv2_type == 'incremental' and option == 'complete':
            job = store.run_ddb_verification(quick_verification=False)
        elif dv2_type == 'full' and option == 'quick':
            job = store.run_ddb_verification(incremental_verification=False)
        else:
            job = store.run_ddb_verification(incremental_verification=False, quick_verification=False)
        self.log.info("DV2 job: %s", job.job_id)
        if not job.wait_for_completion():
            raise Exception(f"Failed to run dv2 job with error: {job.delay_reason}")
        self.log.info("DV2 job completed.")
        return job

    def run_space_reclaim_job(self, store, with_ocl=False):
        """
        runs space reclaim job on the provided store object

        Args:
            store (object) - store object wher espace reclaim job needs to run

            with_ocl (bool) - set True if the job needs to run with OCL phase

        Returns:
            (object) job object for the space reclaim job
        """
        space_reclaim_job = store.run_space_reclaimation(level=4, clean_orphan_data=with_ocl)
        self.log.info("Space reclaim job with OCL[%s]: %s", with_ocl, space_reclaim_job.job_id)
        # validate resync scheduled immediately on job start
        self.log.info("VALIDATION: is store was marked for Validate and Prune?")
        query = f"select flags from idxSidbStore where SIDBStoreid = {store.store_id}"
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info("RESULT (flags): %s", result[0])
        if int(result[0]) & 33554432:
            self.log.info("validate and prune flags were set on store as expected")
        else:
            raise Exception("validate and prune flags were not set on store after space reclaim operation")

        if not space_reclaim_job.wait_for_completion():
            raise Exception(f"Failed to run DDB Space reclaim with error: {space_reclaim_job.delay_reason}")
        self.log.info("DDB Space reclaim job completed.")
        return space_reclaim_job

    def create_orphan_data(self, store_id):
        """this method creates a dummy dedupe chunk with testcase id

        Args:
            store_id (int)  - store id on which dummy chunks needs to be created"""

        self.log.info("creating orphan data...")
        query = f"""
        select top 1 DC.folder, MP.mountpathname,'CV_MAGNETIC', V.volumename from archChunk AC,
        mmvolume V, MMMountPath MP, MMMountpathToStorageDevice MPSD, MMDeviceController DC
        where V.SIDBStoreId = {store_id}
        and MP.mountpathid = V.currmountpathid
        and MPSD.mountpathid = MP.mountpathid
        and DC.deviceid = MPSD.deviceid"""
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info("RESULT (mount path folder): %s", result)
        if not result:
            raise Exception("mount path folder not found")
        orphan_data_path = self.media_agent_machine.os_sep.join(result)
        orphan_chunkid = int(self.id)
        file_content = "*dummy file*" * 1024
        self.orphan_chunks_folder = self.media_agent_machine.join_path(orphan_data_path, f'CHUNK_{orphan_chunkid}')
        self.orphan_chunks_file = self.media_agent_machine.join_path(orphan_data_path,
                                                                     f'CHUNKMAP_TRAILER_{orphan_chunkid}')
        self.media_agent_machine.create_file(self.orphan_chunks_file, file_content)
        self.media_agent_machine.create_directory(self.orphan_chunks_folder, force_create=True)
        self.media_agent_machine.create_file(
            self.media_agent_machine.join_path(self.orphan_chunks_folder, 'SFILE_CONTAINER.idx'), file_content)
        self.media_agent_machine.create_file(
            self.media_agent_machine.join_path(self.orphan_chunks_folder, 'SFILE_CONTAINER_001'), file_content)
        self.media_agent_machine.create_file(
            self.media_agent_machine.join_path(self.orphan_chunks_folder, 'SFILE_CONTAINER_002'), file_content)
        self.media_agent_machine.create_file(
            self.media_agent_machine.join_path(
                self.orphan_chunks_folder, f'CHUNK_META_DATA_{orphan_chunkid}'), file_content)
        self.media_agent_machine.create_file(
            self.media_agent_machine.join_path(self.orphan_chunks_folder, f'CHUNK_META_DATA_{orphan_chunkid}.idx'),
            file_content)

    def validate_space_reclaim(self, space_reclaim_job, with_ocl=False):
        """
        validates the space reclaim job for following:
        1. validate and prune flags set on store (needs to be done immediatly after job completion)
        2. orphan chunk deletion (prune and skip)
        3. orphan chunk phase (with and without)
        4. sub optype for space reclaim job
        5. size difference before and after space reclaim (this is done in main run)

        Args:
            space_reclaim_job (object)        - space reclaim job object

            with_ocl (bool)         - set True to validate job with OCL. [Default: False]

        """
        self.log.info("VALIDATION: orphan chunk existance")

        def is_orphan_chunk_exists():
            if self.media_agent_machine.check_file_exists(self.orphan_chunks_file)\
                    and self.media_agent_machine.check_directory_exists(self.orphan_chunks_folder):
                self.log.info("Orphan chunk exists")
                return True
            self.log.info("Orphan chunk is removed")
            return False
        if (with_ocl and not is_orphan_chunk_exists()) or (not with_ocl and is_orphan_chunk_exists()):
            self.log.info("orphan chunk validation pass")
        else:
            raise Exception("orphan chunk validation failed")

        self.log.info("VALIDATION: phases for space reclaim job with OCL [%s]", with_ocl)
        query = f"select count(1) from JMAdminJobAttemptStatsTable where jobid = {space_reclaim_job.job_id}"
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info("RESULT (job attempts): %s", result[0])
        if result and (with_ocl and result[0] == 3) or (not with_ocl and result[0] == 2):
            raise Exception("Space reclaimation job attempts not expected")
        self.log.info("Space reclaimation job phases ran as expected")

        self.log.info("VALIDATION: sub optype for space reclamation job")
        query = f"select opType, subOpType from jmjobstats where jobid = {space_reclaim_job.job_id}"
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info("RESULT (optype and suboptype): %s", result)
        if not result:
            raise Exception("no result returned from query")
        if int(result[0]) != 31 and int(result[1]) != 141:
            raise Exception("Space reclaim job optype is not correct")
        self.log.info("Space reclamation job optype is set as expected")

    def wait_for_pruning(self, store_id, primary_recs, size_before):
        """waits 10 mins to see if primary recs count is reduced

        Args:
            store_id (int)  - id of the dedupe store where pruning is expected to run

            primary_recs (int)  - primary records count before pruning to confirm phase 2

            size_before (float) - size before pruning to confirm phase 3

        Returns:
            True/False based on count reduction"""
        query = f"""select PrimaryEntries from idxsidbusagehistory
        where sidbstoreid = {store_id} order by modifiedtime desc"""
        self.log.info("QUERY: %s", query)
        for _ in range(5):
            self.csdb.execute(query)
            result = self.csdb.fetch_one_row()
            self.log.info("RESULT: %s", result[0])
            if int(result[0]) < int(primary_recs):
                self.log.info("looks like phase 2 started, waiting for phase 3...")
                count = 0
                while size_before == self.get_size(self.media_agent_machine, self.mount_path) and count < 15:
                    count += 1
                    time.sleep(60)
                self.log.info("""size before [%s], size after [%s]""",
                              size_before, self.get_size(self.media_agent_machine, self.mount_path))
                if size_before > self.get_size(self.media_agent_machine, self.mount_path):
                    self.log.info("looks like phase 3 started...")
                    return True
                self.log.error("timeout reached while waiting for phase 3 pruning")
                return False
            time.sleep(60)
        self.log.error("timeout reached while waiting for phase 2 pruning")
        return False

    @staticmethod
    def get_size(machine_obj, folder):
        """returns the size of folder"""
        if machine_obj.os_info.lower() == 'unix':
            output = machine_obj.execute(f'du {folder}')
            final_len = len(output.formatted_output) - 1
            return output.formatted_output[final_len][0]
        else:
            return machine_obj.get_folder_size(folder)

    def run(self):
        """Run function of this test case"""
        try:
            self.cleanup()
            self.setup_environment()

            for case in range(2):
                if case % 2 == 0:
                    self.log.info("*** space reclaim with OCL case %s ***", case)
                    with_ocl = True
                else:
                    self.log.info("*** space reclaim without OCL case %s ***", case)
                    with_ocl = False

                job1 = self.run_backup()
                self.run_backup(delete_alternative=True)
                store = self.get_active_files_store()
                primary_recs = self.dedupehelper.get_primary_recs_count(store.store_id, db_password=self.sql_password,
                                                                        db_user='sqladmin_cv')
                self.primary_copy.delete_job(job1.job_id)
                for _ in range(2):
                    self.run_data_aging()
                size_before = self.get_size(self.media_agent_machine, self.mount_path)
                if not self.wait_for_pruning(store.store_id, primary_recs, size_before):
                    raise Exception("pruning did not happen")
                store.refresh()
                self.create_orphan_data(store.store_id)

                size_before_space_reclaim = self.get_size(self.media_agent_machine, self.mount_path)
                space_reclaim_job = self.run_space_reclaim_job(store, with_ocl=with_ocl)
                size_after = self.get_size(self.media_agent_machine, self.mount_path)
                self.validate_space_reclaim(space_reclaim_job, with_ocl=with_ocl)
                if size_before_space_reclaim > size_after:
                    self.log.info("space reclaim job reduced mount path size from [%s] to [%s]",
                                  size_before_space_reclaim, size_after)
                else:
                    raise Exception(f"""space reclaim job did not reduce mount path size,
                    before[{size_before_space_reclaim}] after[{size_after}]""")
                self.log.info("run DV2 to make sure all good after space reclaim job")
                self.run_dv2_job(store, 'full', 'complete')
            self.cleanup()

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            if self.mmpruneprocess_value:
                self.update_mmpruneprocess()
            # uncomment below part of the code to make it a fix if fragmentation % is lower than 20
            # self.log.info("removing regkey AuxcopySfileFragPercent...")
            # self.media_agent_machine.remove_registry('MediaAgent', 'AuxcopySfileFragPercent')
            if self.drillhole_key_added:
                self.media_agent_machine.remove_registry('MediaAgent', value='DedupDrillHoles')
                self.log.info("removed regkey to disable drillholes!")
