# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""

from AutomationUtils.constants import FAILED
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from cvpysdk.datacube.constants import IndexServerConstants
from dynamicindex.index_server_helper import IndexServerHelper
from dynamicindex.Datacube.exchange_client_helper import ExchangeClientHelper
from Application.Exchange.ExchangeMailbox.constants import RETENTION_POLICY_DEFAULT
from Application.Exchange.ExchangeMailbox.constants import ARCHIVE_POLICY_DEFAULT
from Application.Exchange.ExchangeMailbox.constants import CLEANUP_POLICY_DEFAULT


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Validation the role update operation on cloud index server"
        self.tcinputs = {
            "IndexServerNodeNames": [],
            "JobResultDirectory": None,
            "ProxyServers": None,
            "MediaAgentName": None,
            "ServiceAccountDetails": None,
            "ExchangeServerName": None,
            "UserMailBox": None
        }
        self.index_server_name = None
        self.index_pool_name = None
        self.hac_cluster_name = None
        self.index_directory = None
        self.node_machine_obj_list = []
        self.exchange_role_version = None
        self.storage_policy_name = None
        self.archive_policy_default = None
        self.retention_policy_default = None
        self.cleanup_policy_default = None
        self.media_agent_name = None
        self.media_agent_machine_obj = None
        self.library_name = None
        self.restore_path = None
        self.library_mount_path = None
        self.exchange_client_helper = None
        self.exchange_client = None

    def setup(self):
        self.hac_cluster_name = "HAC%s" % self.id
        self.index_pool_name = "IndexPool%s" % self.id
        self.index_server_name = "IndexServer%s" % self.id
        self.archive_policy_default = ARCHIVE_POLICY_DEFAULT % self.id
        self.retention_policy_default = RETENTION_POLICY_DEFAULT % self.id
        self.cleanup_policy_default = CLEANUP_POLICY_DEFAULT % self.id
        for role in self.commcell.index_servers.roles_data:
            if role["roleName"] == IndexServerConstants.ROLE_EXCHANGE_INDEX:
                self.exchange_role_version = role["roleVersion"]
        option_selector = OptionsSelector(self.commcell)
        for node_name in self.tcinputs['IndexServerNodeNames']:
            self.node_machine_obj_list.append(Machine(node_name, self.commcell))
        self.index_directory = "%sindex_directory_%s" % (option_selector.get_drive(
            self.node_machine_obj_list[0]), self.id)
        if not self.commcell.hac_clusters.has_cluster(self.hac_cluster_name):
            self.log.info("Creating HAC cluster with nodes %s" % ",".join(self.tcinputs['IndexServerNodeNames']))
            self.commcell.hac_clusters.add(self.hac_cluster_name, self.tcinputs['IndexServerNodeNames'])
            self.log.info("HAC cluster created")
        if not self.commcell.index_pools.has_pool(self.index_pool_name):
            self.log.info("Creating index pool with nodes %s" % ",".join(self.tcinputs['IndexServerNodeNames']))
            self.commcell.index_pools.add(self.index_pool_name,
                                          self.tcinputs['IndexServerNodeNames'],
                                          self.hac_cluster_name)
            self.log.info("Index pool created")
        if not self.commcell.index_servers.has(self.index_server_name):
            for machine_obj in self.node_machine_obj_list:
                machine_obj.remove_directory(self.index_directory, 0)
            self.log.info("Creating index server")
            self.commcell.index_servers.create(
                self.index_server_name,
                self.tcinputs['IndexServerNodeNames'],
                self.index_directory,
                [IndexServerConstants.ROLE_ONEDRIVE_INDEX],
                index_pool_name=self.index_pool_name,
                is_cloud=True
            )
            self.log.info("Index server created")
        option_selector_obj = OptionsSelector(self.commcell)
        self.library_name = "Library_%s" % self.id
        self.media_agent_name = self.tcinputs['MediaAgentName']
        self.storage_policy_name = "storagePolicy%s" % self.id
        self.media_agent_machine_obj = Machine(
            machine_name=self.media_agent_name,
            commcell_object=self.commcell
        )
        restore_directory_drive_letter = option_selector_obj.get_drive(self.media_agent_machine_obj)
        self.restore_path = "%srestore_mail_%s" % (restore_directory_drive_letter, self.id)
        self.library_mount_path = "%sLibrary_%s" % (restore_directory_drive_letter, self.id)
        if not self.commcell.disk_libraries.has_library(self.library_name):
            self.log.info("Creating %s on mount path %s" % (self.library_name,
                                                            self.library_mount_path))
            self.commcell.disk_libraries.add(self.library_name,
                                             self.media_agent_name,
                                             self.library_mount_path)
            self.log.info("Library created")
        if not self.commcell.storage_policies.has_policy(self.storage_policy_name):
            self.log.info("Creating storage policy")
            self.commcell.storage_policies.add(
                self.storage_policy_name,
                self.library_name,
                self.media_agent_name
            )
            self.log.info("Storage policy added successfully")

    def run(self):
        """Run function of this test case"""
        try:
            self.log.info("Modifying index server by adding a new role")
            index_server_obj = self.commcell.index_servers.get(self.index_server_name)
            index_server_obj.update_role([{
                "roleName": IndexServerConstants.ROLE_EXCHANGE_INDEX,
                "operationType": 1
            }])
            if self.exchange_role_version not in index_server_obj.roles:
                raise Exception("Failed to add a role to index server")
            self.log.info("Exchange index role added to the index server")
            self.exchange_client_helper = ExchangeClientHelper(self.commcell)
            self.exchange_client = self.exchange_client_helper.create_exchange_mailbox_client(
                self, self.storage_policy_name, self.index_server_name
            )
            self.exchange_client_helper.create_exchange_configuration_policies(self.id)
            subclient_content = {
                "mailboxNames": self.tcinputs['UserMailBox'],
                "archive_policy": self.archive_policy_default,
                "retention_policy": self.retention_policy_default,
                "cleanup_policy": self.cleanup_policy_default
            }
            self.log.info("Now setting up the user email.")
            self.exchange_client.cvoperations.subclient.set_user_assocaition(subclient_content)
            self.log.info("User mailbox associated successfully")
            self.exchange_client.cvoperations.run_backup()
            self.exchange_client_helper.restore_exchange_mailbox_client(
                self.exchange_client, self.media_agent_name, self.restore_path
            )
            self.log.info("Deleting exchange pseudo client")
            self.commcell.clients.delete(self.exchange_client.cvoperations.client.name)
            self.log.info("Exchange client deleted")
            self.log.info("Modifying index server by removing a role")
            index_server_obj.update_role([{
                "roleName": IndexServerConstants.ROLE_EXCHANGE_INDEX,
                "operationType": 2
            }])
            if self.exchange_role_version in index_server_obj.roles:
                raise Exception("Failed to delete a role from index server")
            self.log.info("Exchange index role removed from the index server")

        except Exception as err:
            self.log.error('Test case failed.')
            self.log.exception(err)
            self.status = FAILED
            raise Exception("Test case failed.")

    def tear_down(self):
        if self.status != FAILED:
            self.log.info("Deleting index server")
            self.commcell.index_servers.delete(self.index_server_name)
            self.log.info("Index server deleted")
            for machine_obj in self.node_machine_obj_list:
                machine_obj.remove_directory(self.index_directory, 0)
            self.log.info("Deleting index pool")
            self.commcell.index_pools.delete(self.index_pool_name)
            self.log.info("Index pool deleted")
            self.log.info("Deleting HAC cluster")
            self.commcell.hac_clusters.delete(self.hac_cluster_name)
            self.log.info("HAC cluster deleted")
            self.log.info("Deleting exchange configuration policies")
            self.commcell.policies.configuration_policies.delete(
                self.archive_policy_default
            )
            self.commcell.policies.configuration_policies.delete(
                self.cleanup_policy_default
            )
            self.commcell.policies.configuration_policies.delete(
                self.retention_policy_default
            )
            self.log.info("Exchange configuration policies deleted")
            self.log.info("Deleting storage policy")
            self.commcell.storage_policies.delete(self.storage_policy_name)
            self.log.info("Storage policy deleted")
            self.media_agent_machine_obj.remove_directory(self.restore_path, 0)
            self.log.info("Deleting the library")
            self.commcell.disk_libraries.delete(self.library_name)
            self.log.info("Library deleted")
