# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

"""
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import constants
from Install.install_validator import InstallValidator
from Install import installer_messages
from AutomationUtils.options_selector import OptionsSelector
from Install.install_helper import InstallHelper
from cvpysdk.deployment.deploymentconstants import DownloadOptions, DownloadPackages


class TestCase(CVTestCase):
    """Negative Testcase : Push updates to a client when its RC is not in sync."""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Negative Scenario - Push updates to a client when its RC is not in sync."
        self.config_json = None
        self.rc_machine_obj = None
        self.unix_machine_obj = None
        self.option_selector = None

    def setup(self):
        """Initializes pre-requisites for this test case"""
        install_helper = InstallHelper(self.commcell)
        machine_objects = install_helper.get_machine_objects()
        self.rc_machine_obj = machine_objects[0]
        self.unix_machine_obj = machine_objects[1]
        self.option_selector = OptionsSelector(self.commcell)

    def run(self):
        """Main function for test case execution"""
        try:
            download_job = self.commcell.download_software(
                options=DownloadOptions.LATEST_HOTFIXES.value,
                os_list=[DownloadPackages.WINDOWS_64.value, DownloadPackages.UNIX_LINUX64.value])

            if not download_job.wait_for_completion():
                raise Exception("Software Cache on CS is empty; Can't Sync Software to Remote Cache ")

            rc_client = self.commcell.clients.get(self.rc_machine_obj.machine_name)
            rc_helper = self.commcell.get_remote_cache(rc_client.client_name)
            unix_client = self.commcell.clients.get(self.unix_machine_obj.machine_name)
            cache_path = self.option_selector.get_drive(self.rc_machine_obj)+"rc_temp"
            rc_helper.configure_remotecache(cache_path=cache_path)
            rc_helper.configure_packages_to_sync(win_os=["WINDOWS_64"],
                                                 win_package_list=["FILE_SYSTEM", "MEDIA_AGENT"],
                                                 unix_os=["UNIX_LINUX64"],
                                                 unix_package_list=["FILE_SYSTEM", "MEDIA_AGENT"])

            rc_helper.assoc_entity_to_remote_cache(client_name=unix_client.client_name)
            sync_job = self.commcell.sync_remote_cache([rc_client.client_name])
            if not sync_job.wait_for_completion():
                self.log.error("Remote Cache Sync Job Failed")

            self.log.info("Sync up Job Successful")
            rc_cache_path = rc_helper.get_remote_cache_path() + "\\CVMedia"

            if self.rc_machine_obj.check_directory_exists(rc_cache_path):
                self.rc_machine_obj.remove_directory(rc_cache_path)

            install_job = unix_client.push_servicepack_and_hotfix()

            if install_job.wait_for_completion():
                raise Exception("Installing updates successful with empty Remote Cache")

            job_status = install_job.delay_reason

            if installer_messages.QINSTALL_SWCACHE_PACKAGES_MISSING not in job_status:
                self.log.error("Job Failed due to some other reason than the expected one.")
                raise Exception(job_status)

            self.log.info("JobFailingReason:%s", job_status)
            sync_job = sync_job.resubmit()
            if not sync_job.wait_for_completion():
                raise Exception("Remote Cache Sync Job Failed")

            self.log.info("Sync up Job Successful")
            install_job = install_job.resubmit()
            if install_job.wait_for_completion():
                self.log.info("Packages successfully installed on the machine")
                install_validation = InstallValidator(unix_client.client_hostname, self)
                install_validation.validate_baseline()
                install_validation.validate_services()
                install_validation.validate_sp_version()
                rc_helper.delete_remote_cache_contents()

            else:
                job_status = install_job.delay_reason
                self.log.error("Failed to Install updates on the client:%s" % unix_client.client_name)
                raise Exception(job_status)

        except Exception as exp:
            self.log.error("Failed with an error: %s", exp)
            self.result_string = str(exp)
            self.status = constants.FAILED
