# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

"""
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import config, constants
from AutomationUtils.machine import Machine
from Install import installer_messages
from Install.install_validator import InstallValidator


class TestCase(CVTestCase):
    """Negative Testcase : Push updates to existing client with not enough disk space"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Negative Scenario - Push updates to existing client with not enough disk space"
        self.config_json = None
        self.client_machine_obj = None

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.config_json = config.get_config()
        self.client_machine_obj = Machine(machine_name=self.config_json.Install.windows_client.machine_host,
                                          commcell_object=self.commcell)

    def run(self):
        """Main function for test case execution"""
        try:
            windows_client = self.commcell.clients.get(self.config_json.Install.windows_client.machine_host)
            path = windows_client.install_directory.split(":")[0]
            available_drive_space = self.client_machine_obj.get_storage_details()[path]['available']
            extra_files_path = path + ":\\Temp"
            if available_drive_space > 500:
                # Filling up the drive with random files to make the drive to have less disk space
                file_size = (available_drive_space*1000/50)-1000
                flag = self.client_machine_obj.generate_test_data(file_path=extra_files_path,
                                                                  file_size=int(file_size),
                                                                  dirs=5,
                                                                  files=10)

                if not flag:
                    raise Exception("Failed to fill up space")

            self.log.info("Sending updates to Windows Client")
            job_install = windows_client.push_servicepack_and_hotfix()
            if job_install.wait_for_completion(10):
                if self.client_machine_obj.check_directory_exists(extra_files_path):
                    self.client_machine_obj.remove_directory(extra_files_path)
                self.log.error("Installation Successful! Please provide a low space directory")
                raise Exception("Client Updates was Successful due to sufficient disk space")

            if self.client_machine_obj.check_directory_exists(extra_files_path):
                self.client_machine_obj.remove_directory(extra_files_path)

            job_status = job_install.delay_reason
            if not (installer_messages.QUPDATE_LOW_CACHE_SPACE in job_status):
                self.log.error("Job Failed due to some other reason than the expected one.")
                raise Exception(job_status)

            self.log.info("JobFailingReason:%s", job_status)
            job_install = job_install.resubmit()
            if job_install.wait_for_completion(5):
                install_validation = InstallValidator(windows_client.client_hostname, self)
                install_validation.validate_baseline()
                install_validation.validate_services()
                install_validation.validate_sp_version()
                self.log.info("Packages successfully installed on client machine")
            else:
                job_status = job_install.delay_reason
                self.log.error("Failed to Push Updates to the client:%s" % windows_client.client_name)
                raise Exception("JobFailingReason:%s", job_status)

        except Exception as exp:
            self.log.error("Failed with an error: %s", exp)
            self.result_string = str(exp)
            self.status = constants.FAILED
