# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""
import time
import os
from AutomationUtils.cvtestcase import CVTestCase
from VirtualServer.VSAUtils import VirtualServerUtils, KubernetesHelper
from VirtualServer.VSAUtils.KubernetesHelper import KubernetesHelper
from AutomationUtils import constants, machine

class TestCase(CVTestCase):
    """Class for executing Basic acceptance Test of VSA backup and Restore test case
        This test case does the following
        1) Create Virtual  Kubernetes client
        2) create Application/ SUbclient.
        3) Connect to VSA Host setup testbed
         (Create namespacefor backup and restore &cleanup if already exists) and deploying test pods
        4) List all pods created
        5) upload data to pod
        6) RUn Backup(FULL) --> check backup is launched and successfully backed up all items
        7) Run Browse and validate the pods backedup or not.
        8) Run Restore and wait until job complates( Fails on failures / CWE)
        9) Connect to host and validate pods and status
        10) Validate data on pod
        11) Verify the pods before backup(Step4) to (Step 8) Report on failures.
        12) add more data on pod
        13) upload it to pod
        14) List all pods created (INCLUDES ALL pods in namespace)
        15) RUn Backup(INCREMENTAL) and successfully backed up all items
        16) Run Browse and validate the pods backedup or not.
        17) Run FULL VM Restore and wait until job complates( Fails on failures / CWE)
        18) Validate data post restore.
        19) Connect to host and validate pods and status
        20) Verify the pods before backup(Step11) to (Step 15) Fail on Failures.
        21) RUn Backup(Synthfull ) and successfully backed up all items
        22) Run Browse and validate the pods backedup or not.
        23) Run FULL VM Restore and wait until job complates( Fails on failures / CWE)
        24) Validate data post restore.
        25) Connect to host and validate pods and status
        26) Verify the pods before backup(Step11) to (Step 15) Fail on Failures

    """

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "VSA KUBERNETES BACKUP AND RESTORE"
        self.product = self.products_list.VIRTUALIZATIONVMWARE
        self.feature = self.features_list.DATAPROTECTION
        self.test_individual_status = True
        self.test_individual_failure_message = ""
        self.utils_path = VirtualServerUtils.UTILS_PATH
        self.pod_create = "poddeployment.yml"
        self.show_to_user = True
        self.namespace = ''
        self.restore_namespace = ''
        self.master_machine = None
        self.planid = ''
        self.controller = None
        self.result_string = ''
        self.status = ''
        self.tcinputs = {}
        self.kubehelper = KubernetesHelper(TestCase)

    def run(self):
        """Main function for test case execution"""
        try:
            self.tcinputs.update({"SubclientName": "automation-58045"})
            self.tcinputs.update({"ClientName": "k8sauto-58045"})
            self.tcinputs.update({"DestinationClient": "k8sauto-58045"})
            self.tcinputs.update({"Namespace": "automation-58045"})
            self.tcinputs.update({"RestoreNamespace": "restoretest-58045"})
            self.namespace = 'automation-58045'
            self.restore_namespace = 'restoretest-58045'
            if 'VSAClient' in self.tcinputs:
                self.controller = self.commcell.clients.get(self.tcinputs['VSAClient'])
            self.log.info(" Checking client with name {0} already exists"
                          .format(self.tcinputs['ClientName']))
            if self.commcell.clients.has_client(self.tcinputs.get('ClientName')):
                self.log.info("Found client:{0} deleting it ".format(self.tcinputs['ClientName']))
                self.commcell.clients.delete(self.tcinputs.get('ClientName'))
            pod_deployment_path = os.path.join(self.utils_path, self.pod_create)
            self.log.info(" Creating cluster with name %s", self.tcinputs['ClientName'])
            self.commcell.clients.add_kubernetes_client(self.tcinputs.get('ClientName'),
                                                        self.tcinputs.get('MasterNode'),
                                                        self.tcinputs.get('secretName'),
                                                        self.tcinputs.get('secretKey'),
                                                        self.tcinputs.get('VSAClient'),
                                                        int(self.controller.client_id))
            self.master_machine = machine.Machine(self.tcinputs.get('MasterNode'),
                                                  username=self.tcinputs.get('Username'),
                                                  password=self.tcinputs.get('Password'))

            VirtualServerUtils.decorative_log("Creating Class Objects")
            if 'ClientName' in self.tcinputs:
                self.log.info("Create client object for: %s", self.tcinputs['ClientName'])
                self._client = self.commcell.clients.get(self.tcinputs['ClientName'])
            if self._client is not None and 'AgentName' in self.tcinputs:
                self.log.info("Create agent object for: %s", self.tcinputs['AgentName'])
                self._agent = self._client.agents.get(self.tcinputs['AgentName'])
            if self._agent is not None:
                # Create object of Instance, if instance name is provided in the JSON
                if 'InstanceName' in self.tcinputs:
                    self.log.info("Create instance object for: %s", self.tcinputs['InstanceName'])
                    self._instance = self._agent.instances.get(self.tcinputs['InstanceName'])
            if 'BackupsetName' in self.tcinputs:
                self.log.info("Creating backupset object for: %s",
                              self.tcinputs['BackupsetName'])
                if self._instance is None:
                    self._backupset = self._agent.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
                else:
                    self._backupset = self._instance.backupsets.get(
                        self.tcinputs['BackupsetName']
                    )
            path = "/tmp/automation_{0}".format(self.id)
            VirtualServerUtils.decorative_log("Done Creating ")
            self.kubehelper.populate_tc_inputs(self)

            VirtualServerUtils.decorative_log("setting Kuberenetes Test Environment")
            self.kubehelper.create_name_space()
            self.kubehelper.populate_data(pod_deployment_path)

            pods = self.kubehelper.get_pods(self.namespace)
            test_pod = pods[0]
            time.sleep(60)

            self.kubehelper.upload_pod_data(test_pod, path, self.id)
            list_before_backup = self.kubehelper.get_pods_services(self.namespace)
            self.log.info("Listing pods services and replication controllers before backup "
                          "in namespace %s", str(list_before_backup))
            if len(list_before_backup) == 0:
                Exception("Failed to get namespace entities or failed to create")

            self.backupset.application_groups.create_application_group(content=self.namespace,
                                                                       plan_name=
                                                                       str(self.tcinputs.get('PlanName')),
                                                                       subclient_name=
                                                                       str(self.tcinputs.get('SubclientName')))

            self.log.info("Application Group Created  %s", str(self.tcinputs.get('SubclientName')))

            self._backupset = self._instance.backupsets.get(
                self.tcinputs['BackupsetName'])
            self.log.info("Creating subclient object for: %s",
                          self.tcinputs['SubclientName'])
            self._subclient = self._backupset.subclients.get(
                self.tcinputs['SubclientName'])
            VirtualServerUtils.decorative_log("Backup")
            self.kubehelper.source_vm_object_creation(self)
            self.kubehelper.backup('FULL')
            self.log.info("Running Restore")
            self.kubehelper.restore_out_of_place(
                                                  self.client.client_name,
                                                  self.tcinputs['StorageType'],
                                                  self.restore_namespace)
            list_pods_services_rst = self.kubehelper.get_pods_services(self.restore_namespace)

            self.log.info("list of pods post restore {0} on namespace {1} "
                          .format(list_pods_services_rst, self.restore_namespace))
            self.log.info("Running Validation")

            # bkp_pod_path = "/tmp/automation_restore_{0}_{1}".format(self.id, self.namespace)
            src_pod_path = self.kubehelper.download_pod_data(test_pod, self.namespace, self.id)
            rst_pod_path = self.kubehelper.download_pod_data(test_pod, self.restore_namespace , self.id)

            result, diff_output = self.master_machine.compare_meta_data(
                src_pod_path,
                rst_pod_path
            )
            if result:
                self.log.info("Meta data comparison successful")
            else:
                self.log.error("Meta data comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Meta data comparison failed")

            result, diff_output = self.master_machine.compare_checksum(
                src_pod_path, rst_pod_path
            )
            if result:
                self.log.info("Checksum comparison successful")
            else:
                self.log.error("Checksum comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Checksum comparison failed")

            VirtualServerUtils.decorative_log(" FULL BACKUP AND RESTORE VALIDATION DONE")
            VirtualServerUtils.decorative_log(" INCREMENTAL Backup")
            self.kubehelper.upload_pod_data(test_pod, path, self.id, job_type='INCR')
            list_before_backup = self.kubehelper.get_pods_services(self.namespace)

            self.log.info("Listing pods services and replication controllers before backup "
                          "in namespace %s", str(list_before_backup))
            if len(list_before_backup) == 0:
                Exception("Failed to get namespace entities or failed to create")

            self.kubehelper.backup('INCREMENTAL')

            self.kubehelper.delete_and_recreate_namespace()
            self.log.info("Running Restore")
            self.kubehelper.restore_out_of_place(
                                                  self.client.client_name,
                                                  self.tcinputs['StorageType'],
                                                  self.restore_namespace)
            list_pods_services_rst = self.kubehelper.get_pods_services(self.restore_namespace)

            self.log.info("list of pods post restore {0} on namespace {1} "
                          .format(list_pods_services_rst, self.restore_namespace))
            self.log.info("Running Validation")

            # bkp_pod_path = "/tmp/automation_restore_{0}_{1}".format(self.id, self.namespace)
            src_pod_path = self.kubehelper.download_pod_data(test_pod, self.namespace, self.id)
            rst_pod_path = self.kubehelper.download_pod_data(test_pod, self.restore_namespace, self.id)

            result, diff_output = self.master_machine.compare_meta_data(
                src_pod_path,
                rst_pod_path
            )
            if result:
                self.log.info("Meta data comparison successful")
            else:
                self.log.error("Meta data comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Meta data comparison failed")

            result, diff_output = self.master_machine.compare_checksum(
                src_pod_path, rst_pod_path
            )
            if result:
                self.log.info("Checksum comparison successful")
            else:
                self.log.error("Checksum comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Checksum comparison failed")

            VirtualServerUtils.decorative_log(" INCR BACKUP AND RESTORE VALIDATION DONE")
            VirtualServerUtils.decorative_log(" RUNNING SYNTHFULL BACKUP AND RESTORE ")

            self.kubehelper.delete_and_recreate_namespace()
            self.log.info("Running Backup")
            self.kubehelper.backup('SYNTHETIC_FULL')
            self.kubehelper.delete_and_recreate_namespace()
            self.log.info("Running Restore")
            self.kubehelper.restore_out_of_place(
                                                  self.client.client_name,
                                                  self.tcinputs['StorageType'],
                                                  self.restore_namespace)
            list_pods_services_rst = self.kubehelper.get_pods_services(self.restore_namespace)

            self.log.info("list of pods post restore {0} on namespace {1} "
                          .format(list_pods_services_rst, self.restore_namespace))
            self.log.info("Running Validation")

            # bkp_pod_path = "/tmp/automation_restore_{0}_{1}".format(self.id, self.namespace)
            src_pod_path = self.kubehelper.download_pod_data(test_pod, self.namespace, self.id)
            rst_pod_path = self.kubehelper.download_pod_data(test_pod, self.restore_namespace , self.id)

            result, diff_output = self.master_machine.compare_meta_data(
                src_pod_path,
                rst_pod_path
            )
            if result:
                self.log.info("Meta data comparison successful")
            else:
                self.log.error("Meta data comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Meta data comparison failed")

            result, diff_output = self.master_machine.compare_checksum(
                src_pod_path, rst_pod_path
            )
            if result:
                self.log.info("Checksum comparison successful")
            else:
                self.log.error("Checksum comparison failed")
                self.log.info("Diff output: \n%s", diff_output)
                raise Exception("Checksum comparison failed")

            VirtualServerUtils.decorative_log(" SYNTHFULL  BACKUP AND RESTORE VALIDATION DONE")
            self.status = constants.PASSED

        except Exception as exp:
            self.log.error('Failed with error: {0}'.format(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            try:
                self.log.info("Cleaning up testcase entitites")
                if self.master_machine.check_directory_exists(src_pod_path):
                    self.log.info("Directory Exists {0} cleaning it ".format(str(src_pod_path)))
                    self.master_machine.remove_directory(src_pod_path)
                if self.master_machine.check_directory_exists(rst_pod_path):
                    self.log.info("Directory Exists {0} cleaning it ".format(str(rst_pod_path)))
                    self.master_machine.remove_directory(rst_pod_path)
                self.commcell.clients.delete(self.tcinputs.get('ClientName'))
                self.log.info("Client_Deleted  %s", str(self.tcinputs.get('ClientName')))
                self.log.info("TEST CASE COMPLETED SUCESSFULLY")

            except Exception:
                self.log.warning("Testcase and/or Restored vm cleanup was not completed")
