# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                      --  initialize TestCase class

    init_tc()                       --  Initial configuration for the test case

    create_ibmi_client()            --  Create new IBMi client machine.

    retire_ibmi_client()			--  Retire the IBMi client

    create_user_defined_backupset()	--  Create a new backupset

    verify_auto_created_subclients()--  Verify Auto-Created subclients

    delete_user_defined_backupset()	--  Delete the specified backpset

    run()                           --  run function of this test case

Input Example:

    "testCases":
            {
                "58029":
                        {
                            "PlanName":"Test-Auto",
                            "ClientName": "Existing-client",
                            "AccessNode": ["proxy1", "proxy2"],
                            "HostName": "IBMi-host-name",
                            "TestPath": "/QSYS.LIB"
                        }
            }

"""

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.FileServerPages.file_servers import FileServers
from Web.AdminConsole.FileServerPages.fsagent import FsSubclient, FsAgent
from Web.Common.page_object import TestStep, handle_testcase_exception
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from time import sleep


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Validate IBMi client creation from command center"
        self.table = None
        self.browser = None
        self.admin_console = None
        self.client_obj = None
        self.display_name = None
        self.navigator = None
        self.file_servers = None
        self.fs_agent_obj = None
        self.fssubclient_obj = None
        self.backupset = None
        self.new_client_name = None
        self.tcinputs = {
            "PlanName": None,
            "TestPath": None,
            "AccessNode": None,
            "HostName": None
        }

    def init_tc(self):
        """ Initial configuration for the test case. """

        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(username=self.inputJSONnode['commcell']['commcellUsername'],
                                     password=self.inputJSONnode['commcell']['commcellPassword'])
            self.navigator = self.admin_console.navigator
            self.file_servers = FileServers(self.admin_console)
            self.fs_agent_obj = FsAgent(self.admin_console)
            self.fssubclient_obj = FsSubclient(self.admin_console)
            self.backupset = "CCAUTOMATION"
            self.table = Table(self.admin_console)
            self.new_client_name = "IBMi-CCAUTOMATION"

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def create_ibmi_client(self):
        """ Create new IBMi client """
        self.navigator.navigate_to_file_servers()
        notification = self.file_servers.add_ibmi_client(server_name=self.new_client_name,
                                                         file_server_host_name=self.new_client_name,
                                                         username=self.tcinputs['UserName'],
                                                         password=self.tcinputs['Password'],
                                                         access_node=self.tcinputs['AccessNode'],
                                                         data_path="/var/commvault",
                                                         port=0,
                                                         plan=self.tcinputs['PlanName'],
                                                         subsystem_description="QSYS/QCTL",
                                                         job_queue="QSYS/QCTL",
                                                         create_jobq=True,
                                                         job_priority=6,
                                                         run_priority=59
                                                         )
        if 'IBM i client created successfully' not in notification:
            raise CVTestStepFailure(notification)

    @test_step
    def retire_ibmi_client(self):
        """ Retire the IBMi client"""
        self.navigator.navigate_to_file_servers()
        self.file_servers.retire_server(server_name=self.new_client_name)

    @test_step
    def create_user_defined_backupset(self, backupset):
        """ Create a new backupset
        Args:
            backupset (str): name of the BackupSet
        """
        self.fs_agent_obj.add_ibmi_backupset(backup_set=backupset,
                                             plan=self.tcinputs['PlanName'])

    @test_step
    def verify_auto_created_subclients(self, backupset):
        """ Verify Auto-Created subclients
        Args:
            backupset (str): name of the BackupSet
        """
        self.fssubclient_obj.is_ibmi_subclient_exists(backup_set=backupset, subclient_name='All')

    @test_step
    def delete_user_defined_backupset(self):
        """ Delete the specified backpset """
        self.navigator.navigate_to_file_servers()
        self.file_servers.access_server(self.display_name)
        self.fssubclient_obj.delete_backup_set(self.backupset)

    def run(self):
        try:
            self.init_tc()
            self.navigator.navigate_to_file_servers()
            if self.file_servers.is_client_exists(server_name=self.new_client_name):
                self.retire_ibmi_client()
            self.create_ibmi_client()
            sleep(10)
            self.commcell.clients.refresh()
            self.client_obj = self.commcell.clients.get(self.new_client_name)
            self.display_name = self.client_obj.display_name
            self.admin_console.refresh_page()
            self.file_servers.access_server(self.display_name)
            self.create_user_defined_backupset(backupset=self.backupset)
            self.admin_console.refresh_page()
            self.verify_auto_created_subclients(backupset="defaultBackupSet")
            self.verify_auto_created_subclients(backupset=self.backupset)
            self.delete_user_defined_backupset()
            self.retire_ibmi_client()

        except Exception as exception:
            handle_testcase_exception(self, exception)

        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
