# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class definied in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    _run_backup()   --  initiates the backup job for the specified subclient

    run()           --  run function of this test case

    Input Example:

    "testCases": {

				"57937": {
					"ClientName": "client1",
					"subClientContent": "C:\\Small_Copy\\001_0001_test_folder",
					"subClientName": "57937_SC",
					"AgentName": "File System",
					"libraryName1": "57937_lib_1",
					"libraryName2": "57937_lib_2",
					"LibraryPath": "c:\\auto_lib",
					"CloudMA1": "MA1",
					"CloudMA2": "MA2",
					"MA1":"MA3",
					"MA2":"MA4",
					"PsudoClientName": "Azure",
					"VSAProxyForPsudoClient":"proxy1",
					"AzureSubsID":"****",
					"AzureTenantID":"****",
					"AzureAppID":"****",
					"AzureAppPassword":"****",
					"storagePolicyName":"57937_SP",
					"BackupSetName":"57937_BS"
				}
			}

"""

from AutomationUtils.machine import Machine
from AutomationUtils import logger, constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import DedupeHelper
import time, threading
import json
from cvpysdk.exception import SDKException


class TestCase(CVTestCase):
    """Class for executing Basic acceptance test of File System backup and Restore test case"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Power Management Aux Copy Case. (Source :Cloud, Destination : Cloud, DDB1 : non-Cloud, DDB2 : non-Cloud. Single Copy (non-DASH, DASH). All Copy (non-DASH, DASH)"

    def setup(self):
        """Setup function of this test case"""

    def create_pseudo_client(self):
        self.log.info("Creating Azure pseudo client/cloud controller [{0}]".format(self.tcinputs['PsudoClientName']))

        self.commcell._clients.add_azure_client(self.tcinputs['PsudoClientName'], self.tcinputs['VSAProxyForPsudoClient'], azure_options = {
                 "subscription_id": self.tcinputs['AzureSubsID'],
                 "tenant_id": self.tcinputs['AzureTenantID'],
                 "application_id": self.tcinputs['AzureAppID'],
                 "password":self.tcinputs['AzureAppPassword'],
                })
        self.log.info("Created successfully")

    def wait_for_job_complete(self):
        timeout = 60*60
        start_time = time.time()
        s = self.jobObj._get_job_details()
        while s["jobDetail"]["progressInfo"]["state"] != "Completed":
            time.sleep(30)
            s = self.jobObj._get_job_details()
            self.log.info(s["jobDetail"]["progressInfo"]["state"])
            if s["jobDetail"]["progressInfo"]["state"] == "Failed":
                self.log.info("Job failed")
                raise SDKException('Storage', '102', 'Job failed')

            if time.time() > (start_time+timeout):
                self.log.info("Job is not completed within expected time.")
                raise SDKException('Storage', '102', 'Job is not completed within expected time')

        self.log.info("Job completed")
        # Some times it takes a few second to release the MA reservation. If we submit power-off by tha time, it might fail. So, will wait for 30 sec after job completed.
        time.sleep(30)

    def configure_cloud_mediaAgent(self, psudo_client_name, ma_obj):
        self.log.info("Verifying power management configurations of MediaAgent : "+ma_obj._media_agent_name)
        if ma_obj._is_power_mgmt_supported:
            self.log.info("Power Management supported")
            if ma_obj._is_power_management_enabled:
                self.log.info("Power management is enabled")
                if ma_obj._power_management_controller_name == psudo_client_name:
                    self.log.info("MA is using correct cloud controller %s ", psudo_client_name)
                else:
                    self.log.info("MA is not using correct cloud controller. Correcting that")
                    ma_obj.enable_power_management(psudo_client_name)
            else:
                self.log.info("Power management is not enabled. Enabling that")
                ma_obj.enable_power_management(psudo_client_name)
        else:
            raise SDKException('Storage', '102', 'Power management is not supported on MediaAgent ' + ma_obj._media_agent_name)

    def create_disk_library(self, LibraryName, MountPath, ma_obj):
        # checking is the library exists
        self.log.info("Checking if the library exists or not")
        self.log.info("Library : "+LibraryName)
        self.log.info("MA : "+ma_obj._media_agent_name)
        if self.commcell.disk_libraries.has_library(LibraryName):
            self.log.info("Library already exists, NOT creating that")

        else:
            self.log.info("Library is not exists. Will try to power-on if power management is enabled and MA is NOT online")
            if ma_obj.current_power_status != "Online" and  ma_obj._is_power_management_enabled:
                self.log.info("Powering-on the MA to create the library")
                ma_obj.power_on()

            self.log.info("Creating library")
            self.commcell.disk_libraries.add(LibraryName, ma_obj._media_agent_name, MountPath)

    def create_multi_partition_dedupe_copy(self, aux_copy_name, lib, destination_ma_name, ddb_ma1_name, ddb_path, ddb_ma2_name):
        self.log.info("Creating secondary copy")

        self.log.info("Powering on the DDB MAs if power management is enabled")
        if self.ddb_ma1_obj._is_power_management_enabled:
            self.log.info("Powering on MA : " + self.ddb_ma1_obj._media_agent_name)
            self.ddb_ma1_obj.power_on()
        if self.ddb_ma2_obj._is_power_management_enabled:
            self.log.info("Powering on MA : " + self.ddb_ma2_obj._media_agent_name)
            self.ddb_ma2_obj.power_on()
        self.dedupehelper = DedupeHelper(self)
        self.log.info("Creating aux copy. Copy Name : {0}".format(aux_copy_name))
        self.dedupehelper.configure_dedupe_secondary_copy(self.sp_obj, aux_copy_name, lib, destination_ma_name, ddb_path+str(time.time()), ddb_ma1_name)
        sidb_store_ids = self.dedupehelper.get_sidb_ids(self.sp_obj._get_storage_policy_id(), aux_copy_name)
        sp_copy_obj = self.sp_obj.get_copy(str(aux_copy_name))
        self.log.info("Created the copy with 1 partition. Adding 1 more partition to the SIDB store on MA" + ddb_ma2_name)
        self.log.info("SIDB Store ID " + str(sidb_store_ids))
        self.sp_obj.add_ddb_partition(str(sp_copy_obj.get_copy_id()), str(sidb_store_ids[0]), str(ddb_path + str(time.time())), str(ddb_ma2_name))

    def verify_non_cloud_media_agents(self):
        if self.ddb_ma1_obj._is_power_management_enabled or self.ddb_ma2_obj._is_power_management_enabled:
            raise SDKException('Storage', '102', 'Expected non-power managed MediaAgent but found at least one powered managed MediaAgent')

    def power_off_all_cloud_mediaagents(self):
        source_ma_power_thread = threading.Thread(target=self.source_ma_obj.power_off)
        destination_ma_thread = threading.Thread(target=self.destination_ma_obj.power_off)

        self.log.info("Starting the power-off thread. MA : " + self.source_ma_obj._media_agent_name)
        source_ma_power_thread.start()

        self.log.info("Starting the power-off thread. MA : " + self.destination_ma_obj._media_agent_name)
        destination_ma_thread.start()

        self.log.info("Waiting for the MAs to power-off")

        source_ma_power_thread.join()
        destination_ma_thread.join()

        self.log.info("All threads completed successfully")

    def run(self):
        try:
            self.sp=self.tcinputs['storagePolicyName']

            self.sc=self.tcinputs['subClientName']

            # MA for primary copy
            self.source_ma=self.tcinputs['CloudMA1']

            # MAs for aux copy
            self.destination_ma=self.tcinputs['CloudMA2']
            self.ddb_ma1=self.tcinputs['MA1']
            self.ddb_ma2 = self.tcinputs['MA2']

            # All MAs and client have to be unique
            list_of_ma = [self.source_ma, self.destination_ma, self.ddb_ma1, self.ddb_ma2, self.tcinputs['ClientName']]
            if len(set(list_of_ma)) != 5:
                raise SDKException('Storage', '102',
                                   'Each MediaAgent and client have to be unique machine. Make sure you have provided 4 unique MediaAgents and client.')

            # Checking if the PsudoClientName exists, else creating that
            if self._commcell._clients.has_client(self.tcinputs['PsudoClientName']):
                self.log.info("Psudo Exists. Not creating a new one")

            else:
                self.log.info("Psudo client is not present with name [%s]. Creating the client controller (Psudo client)",
                         self.tcinputs['PsudoClientName'])
                self.create_pseudo_client()

            self.log.info("Creating all MA objects")
            self.source_ma_obj = self.commcell.media_agents.get(self.source_ma)
            self.destination_ma_obj=self.commcell.media_agents.get(self.destination_ma)
            self.ddb_ma1_obj=self.commcell.media_agents.get(self.ddb_ma1)
            self.ddb_ma2_obj=self.commcell.media_agents.get(self.ddb_ma2)
            self.log.info("Done. Created all MA objects")

            # Verify and Configure cloud MA
            self.configure_cloud_mediaAgent(self.tcinputs['PsudoClientName'], self.source_ma_obj)
            self.configure_cloud_mediaAgent(self.tcinputs['PsudoClientName'], self.destination_ma_obj)

            # Verifying non-cloud MA
            self.verify_non_cloud_media_agents()

            # Creating the libraries if not exists
            self.create_disk_library(self.tcinputs['libraryName1'], self.tcinputs['LibraryPath'], self.source_ma_obj)
            self.create_disk_library(self.tcinputs['libraryName2'], self.tcinputs['LibraryPath'], self.destination_ma_obj)

            #Creating SP
            if not self.commcell.storage_policies.has_policy(self.sp):
                self.log.info("Creating Storage Policy [%s] as its not exists", self.sp)
                self.commcell.storage_policies.add(self.sp, self.tcinputs['libraryName1'],
                                                   self.source_ma_obj._media_agent_name)
            else:
                self.log.info("Storage policy [%s] exists, skipping storage policy creation", self.sp)

            self.sp_obj=self.commcell.storage_policies.get(self.sp)

            # Creating secondary copy
            self.copy_name1="auxCopy1"
            self.copy_name2 = "auxCopy2"

            self.log.info("Creating 1st aux copy if not exists")
            if not self.sp_obj.has_copy(self.copy_name1):
                self.log.info("Creating copy as its not exists")
                self.create_multi_partition_dedupe_copy(self.copy_name1, self.tcinputs['libraryName2'], self.destination_ma, self.ddb_ma1, "c:\ddb\ddb", self.ddb_ma2)

            self.log.info("Creating 2nd aux copy if not exists")
            if not self.sp_obj.has_copy(self.copy_name2):
                self.log.info("Creating copy as its not exists")
                self.create_multi_partition_dedupe_copy(self.copy_name2, self.tcinputs['libraryName2'], self.destination_ma,  self.ddb_ma1, "c:\ddb\ddb", self.ddb_ma2)

            # creating backup set if not exists
            if not self._agent.backupsets.has_backupset(self.tcinputs['BackupSetName']):
                self.log.info("Creating backup set as its NOT exists")
                self.backup_set_obj = self._agent.backupsets.add(self.tcinputs['BackupSetName'])
            else:
                self.log.info("Backup set exists, NOT creating")
                self.backup_set_obj=self._agent.backupsets.get(self.tcinputs['BackupSetName'])

            # Creating sub client
            self.log.info("Creating subclient \"[%s]\" if not exists", self.sc)
            if not self.backup_set_obj.subclients.has_subclient(self.sc):
                self.log.info("SubClient not exists, creating that")
                self.subclient = self.backup_set_obj.subclients.add(self.sc, self.sp)
                # Adding content
                self.log.info("Adding content to subclient")
                self.subclient.content = [self.tcinputs["subClientContent"]]

            else:
                self.log.info("SubClient exists")
                self.subclient = self.backup_set_obj.subclients.get(self.sc)


            # Starting backup
            self.log.info("Starting backup")
            self.jobObj = self.subclient.backup("FULL")
            self.log.info("Waiting for the BACKUP job to complete. Job ID {0}".format(self.jobObj._job_id))
            self.wait_for_job_complete()

            # Powering off the Cloud MA for aux copy (DASH job) job
            self.log.info("Powering off the cloud MAs for next aux job")
            self.power_off_all_cloud_mediaagents()
            self.log.info("Powered off the cloud MAs for aux copy (DASH) job. Starting the aux job.")
            self.jobObj=self.sp_obj.run_aux_copy(self.copy_name1, use_scale=True, all_copies=False)
            self.log.info("Waiting for the aux copy job to complete. Job ID {0}".format(self.jobObj._job_id))
            self.wait_for_job_complete()

            self.log.info("Starting a BACKUP for aux copy (ALL COPY:DASH) job")
            self.jobObj = self.subclient.backup("FULL")
            self.log.info("Waiting for the BACKUP job to complete. Job ID {0}".format(self.jobObj._job_id))
            self.wait_for_job_complete()

            self.log.info("Powering off the MA for aux copy (ALL COPY:DASH) job")
            self.power_off_all_cloud_mediaagents()
            self.log.info("Powered off the cloud MAs for aux copy (ALL COPY:DASH) job. Starting the aux job.")
            self.jobObj = self.sp_obj.run_aux_copy(use_scale=True, all_copies=True)
            self.log.info("Waiting for the aux copy job to complete. Job ID {0}".format(self.jobObj._job_id))
            self.wait_for_job_complete()

            self.log.info("Test case completed successfully.")

        except Exception as excp:

            self.log.error('Failed with error: ' + str(excp))
            self.result_string = str(excp)
            self.status = constants.FAILED
