# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                                            --  initialize TestCase class
    init_tc()                                             --  Initialize pre-requisites
    run_send_log()                                        -- run send log
    _verify_dr_dump()                                     -- check dmp file exist or not
    verify_dr_dump()                                      -- verify comm serve database in side send log bundle
    run()                                                 --  run function of this test case

Input Example:

    "testCases":
            {
                "57820":
                        {

                        }
            }

"""
from AutomationUtils.machine import Machine
from Web.Common.page_object import handle_testcase_exception
from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.page_object import TestStep
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.Commcell import Commcell
from Web.AdminConsole.AdminConsolePages.send_logs import SendLogs
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Reports.SendLog.utils import SendLogUtils
import time


class TestCase(CVTestCase):
    """Class for executing Test Case: Send logs: Verify DR Dumps"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case


        """
        super(TestCase, self).__init__()
        self.browser = None
        self.admin_console = None
        self.job_id = None
        self.machine = None
        self.path = None
        self.directory = None
        self.send_log_utils = None
        self.navigator = None
        self.commserv_client = None
        self.name = "Sendlogs: Verify DR Dumps"

    def init_tc(self):
        """
            Initial configuration for the test case
            """
        try:
            self.log.info("Initializing pre-requisites")
            self.commserv_client = self.commcell.commserv_client
            self.machine = Machine(self.commcell.commserv_client)
            self.send_log_utils = SendLogUtils(self, self.machine)
            self.directory = self.send_log_utils.create_directory_for_given_path("TC57820")
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(self.inputJSONnode['commcell']["commcellUsername"],
                                     self.inputJSONnode['commcell']["commcellPassword"])
            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_commcell()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def run_send_log(self):
        """Running SendLog job"""
        comm_cell = Commcell(self.admin_console)
        comm_cell.access_sendlogs()
        send_log = SendLogs(self.admin_console)
        send_log.select_local_output(local_path=self.directory)
        advanced_list = send_log.Advancedlist
        send_log.deselect_advanced(advanced_list=[advanced_list.SCRUB_LOGFILES])
        info_list = send_log.Informationlist
        send_log.select_information(information_list=[info_list.CSDB, info_list.OTHER_DB ,
                                                      info_list.OS_LOGS, info_list.LOGS])

        self.job_id = send_log.submit()
        job_obj = self.commcell.job_controller.get(self.job_id)
        job_status = job_obj.wait_for_completion()
        if not job_status:
            raise CVTestStepFailure(
                f"Send log job id [{self.job_id}] failed"
            )

    def _verify_dr_dump(self, file_prefix, file_extension):
        """
                Method to check dump file exist or not
       """
        file = self.machine.get_files_in_path(folder_path=self.path)
        for file_var in file:
            if file_prefix.lower() in file_var.lower() and file_extension in file_var:
                self.log.info(f"[{file_var}]  present at location [{self.path}]")
                return

        raise CVTestStepFailure(
            f"[{file_prefix}] ... [{file_extension}] not present in SendLog files "
        )

    @test_step
    def verify_dr_dump(self):
        """ To unzip send log bundle  and verify  CommServe database in side send log bundle with extension .dmp """
        self.path = self.send_log_utils.unzip_send_log_file(self.commserv_client, self.path)
        file_extension = '.dmp'
        file_list = ["CommServ_", "CVCloud_", "HistoryDB_"]
        for file_prefix in file_list:
            self._verify_dr_dump(file_prefix + self.commcell.commserv_name, file_extension)

    def run(self):
        try:
            self.init_tc()
            self.run_send_log()

            self.log.info('Waiting for 20 sec to check file present at location ' + self.directory +
                          ' for send log job id ' + self.job_id)
            time.sleep(20)
            self.path = self.send_log_utils.is_send_log_bundle_exist(self.directory,
                                                                     self.job_id)
            self.verify_dr_dump()
            self.machine.remove_directory(self.directory)
        except Exception as exp:
            handle_testcase_exception(self, exp)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
