# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup method for test case

    tear_down()     --  tear down method for testcase

    run             --  run function of this test case

Input Example:
    "testCases":
        {
            "57819":
                    {
                        "ClientName":"pgtestunix",
                        "DatabaseUser":"postgres",
                        "DatabasePassword":"postgres",
                        "DatabasePort":"5532",
                        "BinaryDirectory":"/usr/pgsql-10/bin",
                        "LibraryDirectory":"/usr/pgsql-10/lib",
                        "ArchiveLogDirectory":"/var/lib/pgsql/10/wal",
                        "Plan": "db2plan"
                    }
        }

"""


from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import database_helper
from Web.Common.page_object import TestStep, handle_testcase_exception
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Databases.db_instances import DBInstances
from Web.AdminConsole.Databases.db_instance_details import DBInstanceDetails
from Web.AdminConsole.Databases.backupset import PostgreSQLBackupset


class TestCase(CVTestCase):
    """ Class for executing Instance and subclient creation/Deletion testcase for PostgreSQL """
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Instance and subclient creation/Deletion testcase for PostgreSQL"
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.tcinputs = {
            'ClientName': None,
            'DatabaseUser': None,
            'DatabasePassword': None,
            'DatabasePort': None,
            'BinaryDirectory': None,
            'LibraryDirectory': None,
            'ArchiveLogDirectory': None,
            'Plan': None}
        self.postgres_database_object = None
        self.database_instances = None
        self.instance_name = None

    def setup(self):
        """ Method to setup test variables """
        self.log.info("Started executing %s testcase", self.id)
        self.log.info("*" * 10 + " Initialize browser objects " + "*" * 10)
        factory = BrowserFactory()
        self.browser = factory.create_browser_object()
        self.browser.open()

        self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.navigator = self.admin_console.navigator
        self.client = self.commcell.clients.get(self.tcinputs['ClientName'])
        self.postgres_database_object = database_helper.PostgreSQL(
            self.client.client_hostname,
            self.tcinputs['DatabasePort'],
            self.tcinputs['DatabaseUser'],
            self.tcinputs['DatabasePassword'],
            "postgres")

    @test_step
    def add_instance(self):
        """add new postgreSQl instance"""
        self.navigator.navigate_to_db_instances()
        self.database_instances = DBInstances(self.admin_console)
        self.instance_name = self.tcinputs['ClientName']+"_"+self.tcinputs['DatabasePort']
        self.database_instances.add_postgresql_instance(
            self.tcinputs['ClientName'],
            self.instance_name,
            self.tcinputs['Plan'],
            self.tcinputs['DatabaseUser'],
            self.tcinputs['DatabasePassword'],
            self.tcinputs['DatabasePort'],
            self.tcinputs['BinaryDirectory'],
            self.tcinputs['LibraryDirectory'],
            self.tcinputs['ArchiveLogDirectory'],
            maintenance_db="postgres")

    @test_step
    def create_test_data(self):
        """creates test databases for automation"""
        self.postgres_database_object.create_db("dummy_database_1")
        self.postgres_database_object.create_db("dummy_database_2")

    @test_step
    def create_database_group(self):
        """creates database group/subclient in dumpbased backupset"""
        db_instance_details = DBInstanceDetails(self.admin_console)
        db_instance_details.click_on_entity('DumpBasedBackupSet')
        postgres_backupset = PostgreSQLBackupset(self.admin_console)
        postgres_backupset.add_subclient(
            "dummy_subclient", 4, True,
            self.tcinputs['Plan'], ['dummy_database_1', 'dummy_database_2'])
        return db_instance_details, postgres_backupset

    @test_step
    def delete_instance_subclient(self, db_instance_details, postgres_backupset):
        """deletes test created subclient/database group and instance"""
        postgres_backupset.delete_subclient("dummy_subclient")
        self.navigator.navigate_to_db_instances()
        self.database_instances.select_instance(
            DBInstances.Types.POSTGRES, self.instance_name, self.tcinputs['ClientName'])
        db_instance_details.delete_instance()

    def tear_down(self):
        """ tear down method for testcase """
        self.log.info("Deleting automation generated databases")
        self.postgres_database_object.drop_db("dummy_database_1")
        self.postgres_database_object.drop_db("dummy_database_2")


    def run(self):
        """ Main function for test case execution """
        try:
            self.add_instance()
            self.create_test_data()
            db_instance_details, postgres_backupset = self.create_database_group()
            self.delete_instance_subclient(db_instance_details, postgres_backupset)

        except Exception as exp:
            handle_testcase_exception(self, exp)

        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
