# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""
import time
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from FileSystem.FSUtils.fshelper import FSHelper
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.FileServerPages.fsagent import FsSubclient
from Web.AdminConsole.FileServerPages.fssubclientdetails import FsSubclientDetails
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import handle_testcase_exception, TestStep


class TestCase(CVTestCase):
    """ Command center: Add/delete exclusions/exceptions """
    test_step = TestStep()

    def __init__(self):
        """ Initializing the reference variables """
        super(TestCase, self).__init__()
        self.name = "Testcase for file system Add/Remove exclusions/exceptions from command center"
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.table = None
        self.machine = None
        self.subclient = None
        self.fs_sub_client = None
        self.sub_client_details = None
        self.fs_helper = None
        self.delimiter = None
        self.os_name = None
        self.restore_file_path = ''
        self.sub_client_name = 'Test_57801'
        self.content = []
        self.exclusions = []
        self.exceptions = []
        self.tcinputs = {
            "ClientName": None,
            "AgentName": None,
            "BackupsetName": None,
            "PlanName": None,
            "RestorePath": None
        }

    @test_step
    def navigate_to_client_page(self):
        """ Navigates to the input client page """
        self.navigator = self.admin_console.navigator
        self.navigator.navigate_to_file_servers()
        self.table.access_link(self.tcinputs['ClientName'])  # navigates to selected client page

    def wait_for_job_completion(self, job_id):
        """ Function to wait till job completes
                Args:
                    job_id (str): Entity which checks the job completion status
        """
        self.log.info("%s Waits for job completion %s", "*" * 8, "*" * 8)
        job_obj = self.commcell.job_controller.get(job_id)
        return job_obj.wait_for_completion()

    def define_filters(self, content):
        """ Sets the exclusion/exception values
                Args:
                    content     (list(paths)): Entity which checks the job completion status
        """
        self.log.info("%s defines exception/exclusions of subclient %s", "*" * 8, "*" * 8)
        file_path = []
        for path in content:
            files = self.machine.get_files_in_path(path)
            file_path.append(files)
        files_list = [file for lst in file_path for file in lst]
        self.exclusions = [file for file in files_list if file.endswith(".html")]
        for i in range(0, len(self.exclusions)):
            if i % 3 == 0:
                self.exceptions.append(self.exclusions[i])

    @test_step
    def refresh(self):
        """ Refreshes the current page """
        time.sleep(60)
        self.admin_console.refresh_page()

    @test_step
    def define_content(self):
        """ Define the subclient content, exclusions and exceptions """
        self.os_name = self.client._properties['client']['osInfo']['Type']
        self.restore_file_path = self.tcinputs['RestorePath']
        if self.os_name == "Windows":
            self.delimiter = "\\"
        else:
            self.delimiter = "/"
        directory_path = self.client.job_results_directory + self.delimiter + 'Backup_content'
        self.content.append(directory_path)
        self.fs_helper.populate_tc_inputs(self, mandatory=False)
        self.fs_helper.generate_testdata(['.html', '.css'], directory_path, 4)
        self.define_filters(self.content)

    @test_step
    def add_subclient(self):
        """ Creates new subclient
                Raises:
                    Exception:
                        -- if fails to add entity
        """
        self.delete_sub_client()
        self.fs_sub_client.add_fs_subclient(self.tcinputs['BackupsetName'],
                                            self.sub_client_name, self.tcinputs['PlanName'],
                                            True, False, self.content,
                                            None, self.exclusions, self.exceptions,
                                            self.os_name, True)
        self.backupset.subclients.refresh()
        self.subclient = self.backupset.subclients.get(self.sub_client_name)

    @test_step
    def backup_job(self, backup_type):
        """ Function to run a backup job
            Args:
                backup_type (BackupType) : Type of backup (FULL, INCR, DIFFERENTIAL, SYN_FULL)
            Raises:
                Exception :
                 -- if fails to run the backup
        """
        job = self.fs_sub_client.backup_subclient(self.tcinputs['BackupsetName'],
                                                  self.sub_client_name, backup_type, False)
        self.wait_for_job_completion(job)

    @test_step
    def restore(self):
        """ Restores the subclient
                Raises:
                    Exception :
                     -- if fails to run the restore operation
         """
        if self.machine.check_directory_exists(self.restore_file_path):
            self.machine.remove_directory(self.restore_file_path)
        self.machine.create_directory(self.restore_file_path, False)
        if self.os_name == "Windows":
            des_path = self.restore_file_path.replace("\\", "/")
        else:
            des_path = self.restore_file_path[1:]
        restore_job = self.fs_sub_client.restore_subclient(
            backupset_name=self.tcinputs['BackupsetName'],
            subclient_name=self.sub_client_name,
            dest_client=self.client.display_name,
            restore_path=des_path)
        self.wait_for_job_completion(restore_job)
        self.browser.driver.back()
        self.admin_console.wait_for_completion()

    def edit_filters(self, add_exclusion=None, rem_exclusion=None,
                     add_exception=None, rem_exception=None):
        """ To add/remove exclusions or exceptions
                Args:
                     add_exclusion (lst)      : Exclusions to be added
                     rem_exclusion(lst)      : Exclusions to be removed
                     add_exception (lst)   : Exceptions to be added
                     rem_exception (lst)   : Exceptions to be removed
                Raises:
                    Exception:
                        There is no option to edit the content of the collection
        """
        self.log.info("%s Add/Remove filters from a new subclient %s", "*" * 8, "*" * 8)
        self.table.access_link(self.sub_client_name)
        self.sub_client_details.edit_content(browse_and_select_data=False, backup_data=[],
                                             exceptions=add_exception, del_exceptions=rem_exception,
                                             exclusions=add_exclusion, del_exclusions=rem_exclusion,
                                             file_system=self.os_name)
        self.browser.driver.back()
        self.admin_console.wait_for_completion()

    def get_filter_path(self, fltr):
        """  Get the Content path from subclient
                Args:
                    fltr  : Entity to get it's path
                            'includePath'  for Exceptions
                            'excludePath'  for Exclusions
                            'path'         for Content
                Returns :
                     filter_paths : path of the entity
        """
        self.log.info("%s Gets the path of a selected filter of a subclient %s", "*" * 8, "*" * 8)
        self.subclient.refresh()
        sub_cont_obj = self.subclient._content
        keys_list = []
        for dic in sub_cont_obj:
            dic_keys = list(dic.keys())
            keys_list.append(dic_keys)
        keys_list = [key for lst in keys_list for key in lst]
        filter_paths = []
        for idx, key in enumerate(keys_list):
            if keys_list[idx] == fltr:
                filter_paths.append(sub_cont_obj[idx][key])
        return filter_paths

    @test_step
    def edit_paths_and_verify(self):
        """  Edit subclient backup paths and verifies the new content is backed up or not
                Raises:
                    Exception:
                        -- -- There is no option to edit the filters of the collection
                        -- if fails to create file
                        -- if fails to run the backup
                        -- if fails to run the restore operation
        """
        self.log.info("%s Adds new files to content %s", "*" * 8, "*" * 8)
        path1 = self.content[0] + self.delimiter + 'newfile1.html'
        self.machine.create_file(path1, 'New file is added after full backup')
        path2 = self.content[0] + self.delimiter + 'newfile2.html'
        self.machine.create_file(path2, 'New file is added after incremental backup')
        exclusions = self.get_filter_path('excludePath')
        self.edit_filters([path1], exclusions, [path2], self.exceptions)
        self.backup_job(Backup.BackupType.INCR)
        self.restore()
        exceptions = self.get_filter_path('includePath')
        self.fs_helper.validate_backup(content_paths=self.subclient.content,
                                       restore_path=self.restore_file_path,
                                       add_exclusions=[path1],
                                       rem_exclusions=exclusions,
                                       exceptions_list=exceptions)

    @test_step
    def delete_sub_client(self):
        """ Verifies whether subclient exists or not and then deletes the subclient """
        subclient_names = self.table.get_column_data('Name')
        subclient_name = self.sub_client_name
        if subclient_name in subclient_names:
            self.log.info("%s Deletes subclient %s", "*" * 8, "*" * 8)
            self.fs_sub_client.delete_subclient(self.tcinputs['BackupsetName'], self.sub_client_name)
            self.admin_console.wait_for_completion()

    def setup(self):
        """ Pre-requisites for this testcase """
        self.log.info("Initializing pre-requisites")
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser,
                                          self.commcell.webconsole_hostname)
        self.admin_console.login(username=self._inputJSONnode['commcell']['commcellUsername'],
                                 password=self._inputJSONnode['commcell']['commcellPassword'])
        self.table = Table(self.admin_console)
        self.fs_sub_client = FsSubclient(self.admin_console)
        self.sub_client_details = FsSubclientDetails(self.admin_console)
        self.fs_helper = FSHelper(self)
        self.machine = Machine(self.client)

    def run(self):
        """Main function for test case execution"""
        try:
            self.define_content()
            self.navigate_to_client_page()
            self.add_subclient()
            self.backup_job(Backup.BackupType.FULL)
            self.refresh()
            self.restore()
            self.fs_helper.validate_backup(content_paths=self.content,
                                           restore_path=self.tcinputs['RestorePath'],
                                           add_exclusions=self.exclusions,
                                           exceptions_list=self.exceptions)
            self.edit_paths_and_verify()
            self.delete_sub_client()

        except Exception as excp:
            handle_testcase_exception(self, excp)

        finally:
            self.log.info("Performing cleanup")
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
